import { MetricDimension, StatusMetric } from "@generated/graphql";
import { Type  } from "@solid/types";
import uPlot, { Cursor } from "uplot";

type StepOptions = {
  drawStyle: number,
  lineInterpolation: number,
};

export type StepBandSeries = uPlot.Series & {
  drawStyle?: number,
  lineInterpolation?: number,
};

const { linear, stepped, bars, spline } = uPlot.paths;

export const lineInterpolations = {
  linear: 0,
  stepAfter: 1,
  stepBefore: 2,
  spline: 3,
};

export const drawStyles = {
  line: 0,
  bars: 1,
  points: 2,
  barsLeft: 3,
  barsRight: 4,
};

export function paths(u: uPlot, seriesIdx: number, idx0: number, idx1: number): uPlot.Series.Paths | null {
  const s = u.series[seriesIdx] as StepBandSeries;
  const style = s.drawStyle;
  const interp = s.lineInterpolation;

  const renderer = (
    style === drawStyles.line ? (
      interp === lineInterpolations.linear     ? linear && linear() :
        interp === lineInterpolations.stepAfter  ? stepped && stepped({align: 1}) :
          interp === lineInterpolations.stepBefore ? stepped && stepped({align: -1}) :
            interp === lineInterpolations.spline     ? spline && spline() :
              null
    ) :
      style === drawStyles.bars ? (
        bars && bars({size: [0.6, 100]})
      ) :
        style === drawStyles.barsLeft ? (
          bars && bars({size: [1], align: 1})//, ascDesc: true
        ) :
          style === drawStyles.barsRight ? (
            bars && bars({size: [1], align: -1})//, ascDesc: true
          ) :
            style === drawStyles.points ? (
              () => null
            ) : () => null
  );

  return renderer ? renderer(u, seriesIdx, idx0, idx1) : null;
}

export const stepOpt: StepOptions = {
  drawStyle: drawStyles.line,
  lineInterpolation: lineInterpolations.stepAfter,
};

type WheelZoomPluginArgs = {
  factor: number
};

export function wheelZoomPlugin(opts: WheelZoomPluginArgs) {
  const factor = opts.factor || 0.75;

  let xMin: number;
  let xMax: number;
  let yMin: number;
  let yMax: number;
  let xRange: number;
  let yRange: number;

  function clamp(nRange: number | undefined, nMin: number | undefined, nMax: number | undefined, fRange: number | undefined, fMin: number | undefined, fMax: number | undefined) {
    if (!nRange || !nMin || !nMax || !fRange || !fMin || !fMax) {
      return [];
    }
    if (nRange > fRange) {
      nMin = fMin;
      nMax = fMax;
    }
    else if (nMin < fMin) {
      nMin = fMin;
      nMax = fMin + nRange;
    }
    else if (nMax > fMax) {
      nMax = fMax;
      nMin = fMax - nRange;
    }

    return [nMin, nMax];
  }

  return {
    hooks: {
      ready: (u: uPlot) => {
        xMin = u.scales.x.min as number;
        xMax = u.scales.x.max as number;
        yMin = u.scales.y.min as number;
        yMax = u.scales.y.max as number;

        xRange = xMax - xMin;
        yRange = yMax - yMin;

        const over = u.over;

        const scXMin0 = u.scales.x.min as number;
        const scXMax0 = u.scales.x.max as number;
        let left0: number;

        const xUnitsPerPx = u.posToVal(1, "x") - u.posToVal(0, "x");

        function onup(e: MouseEvent) {
          document.removeEventListener("mousemove", onmove);
          document.removeEventListener("mouseup", onup);
        }

        function onmove(e: MouseEvent) {
          e.preventDefault();

          const left1 = e.clientX;

          const dx = xUnitsPerPx * (left1 - left0);

          u.setScale("x", {
            min: scXMin0 - dx,
            max: scXMax0 - dx,
          });
        }

        // wheel drag pan
        over.addEventListener("mousedown", e => {
          if (e.button === 1) {
            e.preventDefault();

            left0 = e.clientX;

            document.addEventListener("mousemove", onmove);
            document.addEventListener("mouseup", onup);
          }
        });

        // wheel scroll zoom
        over.addEventListener("wheel", e => {
          e.preventDefault();

          const {left, top} = u.cursor;

          const rect = over.getBoundingClientRect();

          const leftPct = left as number / rect.width;
          const btmPct = 1 - (top as number) / rect.height;
          const xVal = u.posToVal(left as number, "x");
          const yVal = u.posToVal(top as number, "y");

          const oxRange = (u.scales.x.max as number) - (u.scales.x.min as number);
          const oyRange = (u.scales.y.max as number) - (u.scales.y.min as number);

          const nxRange = e.deltaY < 0 ? oxRange * factor : oxRange / factor;
          let nxMin = xVal - leftPct * nxRange;
          let nxMax = nxMin + nxRange;

          [nxMin, nxMax] = clamp(nxRange, nxMin, nxMax, xRange, xMin, xMax);

          const nyRange = e.deltaY < 0 ? oyRange * factor : oyRange / factor;
          let nyMin = yVal - btmPct * nyRange;
          let nyMax = nyMin + nyRange;

          [nyMin, nyMax] = clamp(nyRange, nyMin, nyMax, yRange, yMin, yMax);

          u.batch(() => {
            u.setScale("x", {
              min: nxMin,
              max: nxMax,
            });

            u.setScale("y", {
              min: nyMin,
              max: nyMax,
            });
          });
        });
      }
    }
  };
}

export type PlotData = {
  x: number[],
  y: (number | null | undefined)[] | (number | null | undefined)[][],
  dimension: string
};

const skipInterval = (type: Type.avatar | Type.camera) => type === Type.avatar ? 5 : 12; //min

export const dateFormatValues = [
  // tick incr        default           year                                  month    day                            hour     min                sec          mode
  [3600 * 24 * 365,   "{YYYY}",         null,                                 null,    null,                          null,    null,              null,        1],
  [3600 * 24 * 28,    "{MMM}",          "\n{YYYY}",                           null,    null,                          null,    null,              null,        1],
  [3600 * 24,         "{DD}.{MM}",      "\n{YYYY}",                           null,    null,                          null,    null,              null,        1],
  [3600,              "{HH}:{mm}",      "\n{DD}.{MM}.{YYYY}",                 null,    "\n{DD}.{MM}",                 null,    null,              null,        1],
  [60,                "{HH}:{mm}",      "\n{DD}.{MM}.{YYYY}",                 null,    "\n{DD}.{MM}",                 null,    null,              null,        1],
  [1,                 "{HH}:{mm}:{ss}", "\n{DD}.{MM}.{YYYY}",                 null,    "\n{DD}.{MM}, {HH}:{mm}",      null,    null,              null,        1],
  [0.001,             ":{ss}.{fff}",    "\n{DD}.{MM}.{YYYY}, {HH}:{mm}",      null,    "\n{DD}.{MM}, {HH}:{mm}",      null,    "\n{HH}:{mm}",     null,        1],
];

const mooSync = uPlot.sync("moo");

const matchSyncKeys = (own: string | null, ext: string | null) => own === ext;

function upDownFilter(type: string) {
  return type !== "mouseup" && type !== "mousedown";
}

export const cursorOpts: Cursor = {
  lock: true,
  focus: {
    prox: 16,
  },
  sync: {
    key: mooSync.key,
    setSeries: true,
    match: [matchSyncKeys, matchSyncKeys],
    filters: {
      pub: upDownFilter,
    }
  },
};

function getNullableValues(xValues: number[], yValues: Array<number | null | undefined | string>, plotType: Type.camera | Type.avatar): Array<number | null | undefined> {
  const valuesWithNull = yValues;
  const statusValues = Object.values(StatusMetric);

  for (let i = 1; i <= valuesWithNull.length; i++) {
    if (typeof valuesWithNull[i - 1] === "undefined") {
      continue;
    }
    if (typeof valuesWithNull[i - 1] === "string") {
      const currentIndex = statusValues.findIndex(value => value === valuesWithNull[i - 1]);
      if (currentIndex !== -1) {
        valuesWithNull[i - 1] = currentIndex;
      } else {
        valuesWithNull[i - 1] = null;
      }
    }
    if (xValues[i] * 1000 - xValues[i - 1] * 1000 > 1000 * 60 * skipInterval(plotType)) {
      valuesWithNull[i - 1] = null;
    }
  }

  return valuesWithNull as Array<number | null | undefined>;
}

export function correctPlotData(xValues: number[], yValues: Array<number | null | undefined | string>, dimension: MetricDimension | undefined, plotType: Type.camera | Type.avatar): PlotData {
  const plotData: PlotData = {
    x: xValues,
    y: getNullableValues(xValues, yValues, plotType),
    dimension: dimension as string
  };

  const numData: number[] = yValues.map(val => typeof val === "number" ? val : 0);
  const maxVal = Math.max(...numData);
  const newValues: number[] = [];

  switch (dimension) {
    case MetricDimension.Bps:
      plotData.dimension = "b/s";
      if (maxVal / 1000000 > 10) {
        plotData.dimension = "Mb/s";
        numData.forEach(value => {
          newValues.push(Number((value / 1000000).toFixed(2)));
        });
        break;
      }
      if (maxVal / 1000 > 10) {
        plotData.dimension = "kb/s";
        numData.forEach(value => {
          newValues.push(Number((value / 1000).toFixed(2)));
        });
      }
      break;
    case MetricDimension.Bytes:
      if (maxVal / (1024 * 1024 * 1024) > 1) {
        plotData.dimension = "GB";
        numData.forEach(value => {
          newValues.push(Number((value / (1024 * 1024 * 1024)).toFixed(2)));
        });
        break;
      }
      if (maxVal / (1024 * 1024) > 10) {
        plotData.dimension = "MB";
        numData.forEach(value => {
          newValues.push(Number((value / (1024 * 1024)).toFixed(2)));
        });
        break;
      }
      if (maxVal / 1024 > 100) {
        plotData.dimension = "kB";
        numData.forEach(value => {
          newValues.push(Number((value / 1024).toFixed(2)));
        });
      }
      break;
    case MetricDimension.Ms:
      if (maxVal / 60000 > 60) {
        plotData.dimension = "minutes";
        numData.forEach(value => {
          newValues.push(Number((value / 60000).toFixed(2)));
        });
        break;
      }
      if (maxVal / 1000 > 60) {
        plotData.dimension = "seconds";
        numData.forEach(value => {
          newValues.push(Number((value / 1000).toFixed(2)));
        });
      }
      break;
    case MetricDimension.Pct:
      plotData.dimension = "%";
      break;
    case MetricDimension.Status:
      plotData.dimension = "";
      const statusData: (number | null | undefined)[][] = [[], [], [], [], []];
      for (let y = 0; y < plotData.y.length; y++) {
        for (let i = 0; i < statusData.length; i++) {
          if (plotData.y[y] === i) {
            statusData[i].push(plotData.y[y] as number);
          } else {
            statusData[i].push(null);
          }
        }
      }
      plotData.y = statusData;
      return plotData;
  }

  if (newValues.length > 0) {
    plotData.y = getNullableValues(xValues, newValues, plotType);
  }

  return plotData;
}
