import React, { useEffect, useState } from "react";
import uPlot from "uplot";
import UplotReact from "uplot-react";
import { UUID, Type } from "@solid/types";
import { MetricDimension, StatusMetric } from "@generated/graphql";
import { correctPlotData, cursorOpts, dateFormatValues, wheelZoomPlugin, paths, StepBandSeries, stepOpt } from "./utils";
import { __ } from "@solid/libs/i18n";

import "uplot/dist/uPlot.min.css";
import "./style.css";

type UplotOptionsWithStepBand = Omit<uPlot.Options, "series"> & {
  series:  StepBandSeries[],
};

type PlotProps = {
  id: UUID,
  title: string,
  data: [xValues: number[], ...yValues:(number | null | undefined | string)[][]],
  plotType: Type.avatar | Type.camera,
  dimension?: MetricDimension,
  target?: HTMLElement
  widthRef?: number
};

const Plot = ({ id, title, data, plotType,  dimension, target, widthRef }: PlotProps) => {
  const [plotData, setPlotData] = useState<uPlot.AlignedData>();
  const [plotTitle, setPlotTitle] = useState<string>(title);
  const [plotDimension, setPlotDimension] = useState<string>(dimension as string);

  const statusValues = Object.values(StatusMetric);

  const options: UplotOptionsWithStepBand = {
    title: plotTitle,
    width: widthRef ? widthRef - 75 : 800,
    height: dimension === MetricDimension.Status ? 250 : 200,
    padding: [20, 20, 20, 60],
    cursor: cursorOpts,
    axes: [
      {
        values: dateFormatValues,
      },
      {
        values: dimension === MetricDimension.Status ?
          (u, val, space) => val.map(v => statusValues[v])
          :
          (u, val, space) => val.map(v => v),
      }
    ],
    series: dimension === MetricDimension.Status ?
      [
        {
          label: __("Date"),
          value: "{DD}.{MM}.{YYYY}, {HH}:{mm}:{ss}"
        },
        {
          label: "",
          stroke: "#21BA45",
          width: 10,
          value: (u, v) => v === null ? "--" : statusValues[v],
        },
        {
          label: "",
          stroke: "#B5CC18",
          width: 10,
          value: (u, v) => v === null ? "--" : statusValues[v],
        },
        {
          label: "",
          stroke: "#FBBD08",
          width: 10,
          value: (u, v) => v === null ? "--" : statusValues[v],
        },
        {
          label: "",
          stroke: "#F2711C",
          width: 10,
          value: (u, v) => v === null ? "--" : statusValues[v],
        },
        {
          label: "",
          stroke: "#DB2828",
          width: 10,
          value: (u, v) => v === null ? "--" : statusValues[v],
        },
      ]
      :
      [
        {
          label: __("Date"),
          value: "{DD}.{MM}.{YYYY}, {HH}:{mm}:{ss}"
        },
        {
          label: "",
          points: { show: false },
          width: 1,
          stroke: "#2185D0",
          fill: "#2185D01F",
          value: (u: uPlot, v: number | null) => v === null ? "--" : `${v} ${plotDimension || ""}`,
          paths,
          drawStyle: stepOpt.drawStyle,
          lineInterpolation: stepOpt.lineInterpolation,
        }
      ],
    plugins: [wheelZoomPlugin({factor: 0.75})],
    scales: {
      x: { time: true },
    },
  };

  useEffect(() => {
    if (!data) {
      return;
    }
    const newData = correctPlotData(data[0], data[1], dimension, plotType);
    setPlotDimension(newData.dimension);

    if (dimension === MetricDimension.Status) {
      const statusData: any = [newData.x];
      newData.y.forEach(values => {
        statusData.push(values);
      });

      setPlotData(statusData);
      return;
    }
    const newTitle = `${title}, ${newData.dimension}`;
    setPlotTitle(newTitle);

    setPlotData([newData.x, newData.y as (number | null)[]]);
  }, [data]);

  return (
    data[0].length === 0 ?
      <div className="Plot-NoData">
        <div className="u-title">{title}</div>
        <div className="Plot-Stub">{__("No Data")}</div>
      </div>
      :
      <UplotReact
        key={`plot_${id}`}
        options={options}
        data={plotData || [[]]}
        target={target}/>
  );
};

export default Plot;
