import React, { useCallback, useEffect, useState } from "react";
import { Form } from "semantic-ui-react";

import { __ } from "@solid/libs";

import InputDate from "components/InputDate";

import "./style.css";

export type PeriodValue = {
  [x in CalendarType]?: Date;
};

export enum CalendarType {
  StartFrom = "startFrom",
  EndTo = "endTo"
}

type PeriodProps = {
  startFrom?: Date;
  endTo?: Date;
  minTime?: PeriodValue;
  maxTime?: PeriodValue;
  onChange: (period: PeriodValue) => void;
};

const Period = ({ startFrom, endTo, minTime, maxTime, onChange }: PeriodProps) => {
  const [period, setPeriod] = useState<PeriodValue>({
    [CalendarType.StartFrom]: startFrom,
    [CalendarType.EndTo]: endTo
  });

  useEffect(() => {
    const period = {
      [CalendarType.StartFrom]: startFrom,
      [CalendarType.EndTo]: endTo
    };
    setPeriod(period);

  }, [startFrom, endTo]);

  const onHandleDateChange = useCallback((calendarType: CalendarType, value: Date) => {
    period[calendarType] = value;
    onChange(period);
    setPeriod(period);
  }, [onChange, period]);

  return (
    <Form.Group inline className="Period">
      <InputDate
        label={__("Start from")}
        placeholder={__("Start from")}
        value={period.startFrom}
        minTime={minTime?.startFrom}
        maxTime={maxTime?.startFrom}
        onChange={(timeStamp) => {
          onHandleDateChange(CalendarType.StartFrom, new Date(timeStamp));
        }}
      />
      <InputDate
        withUpdate
        label={__("End to")}
        placeholder={__("End to")}
        value={period.endTo}
        minTime={minTime?.endTo}
        maxTime={maxTime?.endTo}
        onChange={(timeStamp) => {
          onHandleDateChange(CalendarType.EndTo, new Date(timeStamp));
        }}
      />
    </Form.Group>
  );
};

export default Period;
