import React, {useState} from "react";
import {Button, Form, Header, Message, Segment} from "semantic-ui-react";
import {useNavigate} from "react-router-dom";
import {useLoginDone} from "@core/store/actions";
import Loading from "components/Loading";
import {API} from "@solid/libs/api";
import {__} from "@solid/libs/i18n";
import {ApiError} from "@solid/types";

import "./style.css";

const PasswordExpired = () => {
  const [password, setPassword] = useState("");
  const [newPassword, setNewPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [errorMessage, setErrorMessage] = useState("");
  const [isLoading, setLoading] = useState(false);
  const navigate = useNavigate();

  const { loginDone } = useLoginDone();

  const changePassword = async () => {
    if (newPassword !== confirmPassword) {
      setErrorMessage(__("New password and password confirmation do not match."));
      return;
    }

    setLoading(true);
    try {
      setErrorMessage("");

      const api = new API();
      await api.changePassword({ currentPassword: password, newPassword });
      await loginDone();

      navigate("/");
    }
    catch (e) {
      const {code, message} = e as ApiError;
      setErrorMessage(code ? `[${code}] ${message}` : message);
      setLoading(false);
    }
  };

  return (
    <Segment className="PasswordExpired">
      <Form onSubmit={e => e.preventDefault()} disabled={isLoading}>
        <Form.Field>
          <Header as="p" className="login-header">{__("Password expired. Please change password.")}</Header>
          <Message visible error content={errorMessage} hidden={!errorMessage} />
        </Form.Field>
        <Form.Field control={Form.Input} value={password} type="password" className="PasswordExpired__password"
          label={__("Current Password")} placeholder={__("Current Password")} autoFocus
          onInput={(e: React.FormEvent<HTMLInputElement>) => {
            setErrorMessage("");
            setPassword(e.currentTarget.value);
          }} />
        <Form.Field control={Form.Input} value={newPassword} type="password" className="PasswordExpired__new-password"
          label={__("New Password")} placeholder={__("New Password")}
          onInput={(e: React.FormEvent<HTMLInputElement>) => {
            setErrorMessage("");
            setNewPassword(e.currentTarget.value);
          }} />
        <Form.Field control={Form.Input} value={confirmPassword} type="password" className="PasswordExpired__confirm-password"
          label={__("Confirm New Password")} placeholder={__("Confirm New Password")}
          onInput={(e: React.FormEvent<HTMLInputElement>) => {
            setErrorMessage("");
            setConfirmPassword(e.currentTarget.value);
          }} />
        <Button className="PasswordExpired__change-password" primary fluid onClick={() => changePassword()}>{__("Change Password")} </Button>
      </Form>
      <Loading active={isLoading} text={__("Changing password...")}/>
    </Segment>
  );
};

export default PasswordExpired;
