import React, { useState, useEffect, useRef } from "react";
import { Dropdown, DropdownItemProps, DropdownProps, Divider, Ref, Icon, Popup, Input } from "semantic-ui-react";
import classNames from "classnames";
import { Hierarchy, Label, LabelsQuery, HierarchiesQuery } from "@generated/graphql";
import LabelEditorDialog from "components/LabelsAndHierarchies/LabelEditorDialog";
import HierarchyEditorDialog from "components/LabelsAndHierarchies/HierarchyEditorDialog";
import { isArray, getLocalStorage, setLocalStorage } from "utils";
import { getLabelDropdownItem } from "components/LabelsAndHierarchies/utils";
import { Utils } from "@solid/libs/utils";
import {__} from "@solid/libs/i18n";

import "./style.css";

type ListFilterProps = {
  labelData?: LabelsQuery;
  hierarchyData?: HierarchiesQuery;
  hierarchyId?: string;
  hierarchy?: Hierarchy;
  searchText?: string;
  labels?: Label[];
  hierarchyMode?: boolean;
  hierarchyFilter?: boolean;
  nameFilter?: boolean;
  labelsFilter?: boolean;
  allFilters?: boolean;
  allowLabelHierarchyEdit?: boolean;
  localStoragePrefix?: string;
  rootRef?: React.RefObject<HTMLElement | null>;
  filterTextPlaceholder?: string;
  onHierarchyChange?: (hierarchy?: Hierarchy) => void;
  onSearchTextChange?: (text: string) => void;
  onLabelsChange?: (labels: Label[]) => void;
};

const initialLabelLoadLimit = 5;

enum CustomSearchOptions {
  ShowAll = "ShowAll",
  Divider1 = "Divider1"
}

const customSearchOptions: string[] = [
  CustomSearchOptions.ShowAll,
  CustomSearchOptions.Divider1
];

const ListFilter = ({
  labelData,
  hierarchyData,
  hierarchyId = "",
  hierarchy,
  searchText = "",
  labels,
  hierarchyMode = false,
  hierarchyFilter = false,
  nameFilter = false,
  labelsFilter = false,
  allFilters = false,
  allowLabelHierarchyEdit = false,
  localStoragePrefix = "",
  rootRef,
  filterTextPlaceholder,
  onHierarchyChange,
  onSearchTextChange,
  onLabelsChange
}: ListFilterProps) => {
  const [searchOptions, setSearchOptions] = useState<DropdownItemProps[]>([]);
  const [searchValues, setSearchValues] = useState<string[]>(labels?.map(label => label.id) ?? []);
  const [searchQuery, setSearchQuery] = useState("");
  const [labelEditorOpen, setLabelEditorOpen] = useState(false);
  const [hierarchyEditorOpen, setHierarchyEditorOpen] = useState(false);
  const [dropdownOpen, setDropdownOpen] = useState(false);
  const [labelLoadLimit, setLabelLoadLimit] = useState(initialLabelLoadLimit);
  const [hierarchyOptions, setHierarchyOptions] = useState<DropdownItemProps[]>([]);
  const [hierarchyLabelVisible, setHierarchyLabelVisible] = useState(false);
  const dropdownRef = useRef<HTMLElement>(null);
  const nameInputRef = useRef<Input>(null);
  const resizeObserverRef = useRef<ResizeObserver>();

  useEffect(() => {
    const root = rootRef?.current;
    if (root) {
      resizeObserverRef.current = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
        for (const entry of entries) {
          if (entry.target === root) {
            setHierarchyLabelVisible(entry.contentRect.width >= 275);
            break;
          }
        }
      }));
      resizeObserverRef.current.observe(root);
    }

    nameInputRef.current && nameInputRef.current.focus();

    return function cleanup() {
      root && resizeObserverRef.current?.unobserve(root);
    };
  }, []);

  useEffect(() => {
    if (!hierarchyData) {
      return;
    }

    const hierarchies = Array.from(hierarchyData.hierarchies).sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));

    let hierarchy: Hierarchy | undefined;
    if (hierarchyId) {
      hierarchy = hierarchyData.hierarchies.find(hier => hier.id === hierarchyId);
      if (hierarchyFilter) {
        const storedId = getStoredHierarchyId();
        const storedHierarchy = storedId ? hierarchies.find(hier => hier.id === storedId) : undefined;
        if (storedHierarchy) {
          hierarchy = storedHierarchy;
        }
        if (!hierarchy && hierarchies.length > 0) {
          hierarchy = hierarchies[0];
        }
      }
    }

    onHierarchyChange && onHierarchyChange(hierarchy);

    const options: DropdownItemProps[] = hierarchies.map(({ id, name }) => ({ value: id, text: name }));
    if (allFilters) {
      options.unshift({ key: "", value: undefined, text: "" });
    }
    setHierarchyOptions(options);
  }, [hierarchyData]);

  useEffect(() => {
    setLabelLoadLimit(initialLabelLoadLimit);
  }, [searchQuery]);

  useEffect(() => {
    if (!labelData) {
      return;
    }
    if (hierarchyMode) {
      return;
    }

    let options: DropdownItemProps[];
    let showAllVisible = false;

    let unselectedLabels = labelData.labels.filter(label => !searchValues.includes(label.id));
    if (searchQuery) {
      unselectedLabels = unselectedLabels.filter(({ name }) => name.toLocaleUpperCase().includes(searchQuery.toLocaleUpperCase()));
    }

    if (labelLoadLimit > 0) {
      const selectedLabels = labelData.labels.filter(label => searchValues.includes(label.id));

      showAllVisible = unselectedLabels.length > labelLoadLimit;
      unselectedLabels = unselectedLabels.sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" })).slice(0, labelLoadLimit);

      options = selectedLabels.concat(unselectedLabels).map(label => getLabelDropdownItem(label));
    }
    else {
      options = labelData.labels.map(label => getLabelDropdownItem(label));
    }

    options = options.sort((a, b) => a.text?.toString().localeCompare(b.text?.toString() ?? "", undefined, { sensitivity: "base" }) ?? 0);

    if (showAllVisible) {
      options.push({ key: CustomSearchOptions.Divider1, value: CustomSearchOptions.Divider1, text: "", content: <Divider/>, disabled: true });
      options.push({ key: CustomSearchOptions.ShowAll, value: CustomSearchOptions.ShowAll, text: __("Show all labels") });
    }

    setSearchOptions(options);
  }, [labelData, searchQuery, searchValues, labelLoadLimit, hierarchyMode]);

  useEffect(() => {
    if (labelData) {
      const oldLabels = labels ?? [];
      const newLabels = labelData.labels.filter(label => searchValues.includes(label.id));
      if (onLabelsChange &&
          JSON.stringify(oldLabels.map(label => label.id).sort()) !== JSON.stringify(newLabels.map(label => label.id).sort())) {
        onLabelsChange(newLabels);
      }
    }
  }, [searchValues]);

  function filterSearchOptions(options: DropdownItemProps[], searchValue: string): DropdownItemProps[] {
    return options.filter(({ value, text }) =>
      value &&
      (customSearchOptions.includes(value.toString()) ||
       text?.toString().toLocaleUpperCase().includes(searchValue.toLocaleUpperCase())));
  }

  function onSearchValueChange(e: React.SyntheticEvent, { value }: DropdownProps): void {
    if (!isArray<string>(value, "string")) {
      return;
    }

    setSearchValues(value.filter(option => !customSearchOptions.includes(option)));

    if (value.includes(CustomSearchOptions.ShowAll)) {
      setLabelLoadLimit(0);
    }
  }

  function adjustDropdownMenuHeight(): void {
    if (!rootRef?.current || !dropdownRef.current) {
      return;
    }

    const menu = dropdownRef.current.querySelector(".menu") as HTMLElement;
    if (!menu) {
      return;
    }

    const rootRect = rootRef.current.getBoundingClientRect();
    const dropdownRect = dropdownRef.current.getBoundingClientRect();
    const availHeight = rootRect.bottom - dropdownRect.bottom;

    menu.style.maxHeight = "";
    const style = window.getComputedStyle(menu, null);
    const maxHeight = parseInt(style.maxHeight);
    if (maxHeight > availHeight) {
      menu.style.maxHeight = `${availHeight}px`;
    }
  }

  function getStoredHierarchyId(): string {
    return localStoragePrefix ? getLocalStorage(`${localStoragePrefix}.hierarchyId`) ?? "" : "";
  }

  function hierarchyChange(value: string | number | boolean | (string | number | boolean)[]  | undefined): void {
    const hierarchy = hierarchyData?.hierarchies.find(hier => hier.id === value);
    if (hierarchy && localStoragePrefix) {
      setLocalStorage(`${localStoragePrefix}.hierarchyId`, hierarchy.id);
    }
    onHierarchyChange && onHierarchyChange(hierarchy);
  }

  return (
    <div className="ListFilter">
      {(hierarchyFilter || allFilters) &&
        <div className="ListFilter-HierarchyDropdown">
          {hierarchyLabelVisible && <span>{__("Hierarchy")}</span>}
          <Popup trigger={
            <Dropdown
              fluid
              selection
              options={hierarchyOptions}
              value={hierarchy?.id}
              onChange={(e, { value }) => hierarchyChange(value)}
            />}
            content={__("Hierarchy")}
            disabled={hierarchyLabelVisible}
          />
          {allowLabelHierarchyEdit &&
          <Popup trigger={
            <Icon
              className="ListFilter-EditButton"
              name="edit"
              onClick={() => setHierarchyEditorOpen(true)}
            />}
            content={__("Edit Hierarchy")}
            position="bottom right"
          />}
        </div>}

      {(nameFilter || allFilters) &&
      <div className="ListFilter-SearchByName">
        <Input
          ref={nameInputRef}
          placeholder={filterTextPlaceholder || __("Filter by name")}
          icon="search"
          value={searchText}
          onChange={e => { onSearchTextChange && onSearchTextChange(e.currentTarget.value); }}
        />
        <Popup trigger={
          <Icon
            className="ListFilter-EditButton"
            name="cancel"
            onClick={() => { onSearchTextChange && onSearchTextChange(""); }}
          />}
          content={__("Clear")}
          position="bottom right"
        />
      </div>}

      {((labelsFilter || allFilters) && !hierarchyMode) &&
        <div className="ListFilter-LabelsDropdown">
          <Ref innerRef={dropdownRef}>
            <Dropdown
              className={classNames("ListFilter-Dropdown", { "empty": searchValues.length === 0 })}
              icon="search"
              placeholder={__("Filter by labels")}
              fluid
              selection
              multiple
              search={filterSearchOptions}
              options={searchOptions}
              value={searchValues}
              searchQuery={searchQuery}
              open={dropdownOpen}
              onChange={onSearchValueChange}
              onSearchChange={(e, { searchQuery }) => setSearchQuery(searchQuery)}
              onOpen={() => {
                setDropdownOpen(true);
                adjustDropdownMenuHeight();
              }}
              onClose={() => {
                setDropdownOpen(false);
              }}
            />
          </Ref>
          {allowLabelHierarchyEdit &&
          <Popup trigger={
            <Icon
              className="ListFilter-EditButton"
              name="edit"
              onClick={() => setLabelEditorOpen(true)}
            />}
            content={__("Edit Labels")}
            position="bottom right"
          />}
        </div>}

      <LabelEditorDialog open={labelEditorOpen} onClose={() => setLabelEditorOpen(false)}/>
      <HierarchyEditorDialog open={hierarchyEditorOpen} onClose={() => setHierarchyEditorOpen(false)} hierarchyId={hierarchy?.id}/>
    </div>
  );
};

export default ListFilter;
