import { TabPanel, Widget, TabBar } from "@lumino/widgets";
import { Message } from "@lumino/messaging";
import { RBaseWidget, RBaseWidgetProps, WrapperHelper, IWrappedWidget, ILayoutWidget } from "./RBaseWidget";
import { EventPubSub, EventPubSubFunc, isRecord } from "utils";
import { CurrentChangedArgs, CanChangeToDisabledTabArgs, LayoutType } from "components/Layout";

type CurrentChangedFunc = EventPubSubFunc<CurrentChangedArgs>;

class CurrentChangedEvent extends EventPubSub<CurrentChangedArgs> {}

class CanChangeToDisabledTabEvent extends EventPubSub<CanChangeToDisabledTabArgs> {}

type RTabPanelProps = RBaseWidgetProps & {
  onCurrentChanged?: CurrentChangedFunc;
};

type RTabPanelLayout = {
  currentIndex: number;
  tabOrder: Record<string, number>;
};

class WrapperTabPanel extends TabPanel implements IWrappedWidget, ILayoutWidget {
  readonly wrapper: WrapperHelper;
  private restoringCurrentIndex: boolean = false;

  constructor(id: string, name?: string, options?: TabPanel.IOptions) {
    super(options);
    this.wrapper = new WrapperHelper(this, id, name);
    this.currentChanged.connect(this.onCurrentChanged, this);
    this.tabBar.tabMoved.connect(this.onTabMoved, this);
    this.tabBar.tabCloseRequested.connect(this.onTabClose, this);
  }

  override onCloseRequest(msg: Message) {
    super.onCloseRequest(msg);
    this.wrapper.publishClose();
    this.currentChanged.disconnect(this.onCurrentChanged, this);
    this.tabBar.tabMoved.disconnect(this.onTabMoved, this);
    this.tabBar.tabCloseRequested.disconnect(this.onTabClose, this);
  }

  private onCurrentChanged(sender: TabPanel, args: TabPanel.ICurrentChangedArgs) {
    if (!this.restoringCurrentIndex) {
      this.wrapper.publishLayoutChanged(this.id);
    }
    this.restoringCurrentIndex = false;
  }

  private onTabMoved(sender: TabBar<Widget>, args: TabBar.ITabMovedArgs<Widget>) {
    this.wrapper.publishLayoutChanged(this.id);
  }

  private onTabClose(sender: TabBar<Widget>, { index }: TabBar.ITabCloseRequestedArgs<Widget>) {
    if (index === this.currentIndex && index > 0) {
      this.currentIndex = index - 1;
    }
    this.wrapper.publishLayoutChanged(this.id);
  }

  getLayout(): LayoutType | undefined {
    const layout: RTabPanelLayout = { currentIndex: this.currentIndex, tabOrder: {} };
    for (let i = 0; i < this.widgets.length; i++) {
      layout.tabOrder[this.widgets[i].id] = i;
    }
    return layout;
  }

  restoreTabOrder(obj: LayoutType): void {
    const { tabOrder } = obj as RTabPanelLayout;
    const { tabOrder: prevTabOrder } = this.getLayout() || { tabOrder: {} };
    if (isRecord<number>(tabOrder, "number") && JSON.stringify(tabOrder) !== JSON.stringify(prevTabOrder)) {
      const order: Record<string, number> = {};
      Object.keys(tabOrder).filter(id => this.widgets.some(w => w.id === id))
        .sort((a, b) => tabOrder[a] - tabOrder[b]).forEach((id, index) => { order[id] = index; });
      for (const id in order) {
        const curIndex = this.widgets.findIndex(w => w.id === id);
        const newTabsBefore = this.widgets.filter((w, i) => i < curIndex && order[w.id] === undefined).length;
        const index = order[id] + newTabsBefore;
        if (index >= 0 && index < this.widgets.length) {
          const widget = this.widgets.find(w => w.id === id);
          if (widget) {
            this.insertWidget(index, widget);
          }
        }
      }
    }
  }

  restoreLayout(obj: LayoutType): void {
    this.restoreTabOrder(obj);
    const { currentIndex } = obj as RTabPanelLayout;
    if (typeof currentIndex === "number" && currentIndex >= 0 &&
        currentIndex < this.widgets.length && this.currentIndex !== currentIndex) {
      this.restoringCurrentIndex = true;
      this.currentIndex = currentIndex;
    }
    this.wrapper.layoutRestored.publish({ id: this.id });
  }
}

export class RTabPanel extends RBaseWidget<RTabPanelProps, WrapperTabPanel> {
  readonly currentChanged = new CurrentChangedEvent();
  readonly canChangeToDisabledTab = new CanChangeToDisabledTabEvent();
  private tabsDisabled: boolean = false;
  private forceTabChange: boolean = false;

  constructor(props: RTabPanelProps) {
    super(props);
    if (this.layoutContainer) {
      this.layoutContainer.currentChanged.connect(this.onCurrentChanged, this);
    }
  }

  protected createLayoutContainer(): WrapperTabPanel {
    return new WrapperTabPanel(this.props.id, this.props.name, { tabsMovable: true });
  }

  override componentWillUnmount() {
    this.currentChanged.unsubscribeAll();
    if (this.layoutContainer) {
      this.layoutContainer.currentChanged.disconnect(this.onCurrentChanged, this);
    }
  }

  get currentIndex(): number {
    return this.layoutContainer ? this.layoutContainer.currentIndex : 0;
  }

  set currentIndex(index: number) {
    if (this.layoutContainer) {
      this.layoutContainer.currentIndex = index;
    }
  }

  get currentTabId(): string {
    return this.layoutContainer && this.layoutContainer.currentIndex >= 0 &&
      this.layoutContainer.currentIndex < this.layoutContainer.widgets.length ?
      this.layoutContainer.widgets[this.layoutContainer.currentIndex].id : "";
  }

  get tabCount(): number {
    return this.layoutContainer?.widgets.length ?? 0;
  }

  private onCurrentChanged(sender: TabPanel, { currentIndex: index, currentWidget, previousIndex: prevIndex, previousWidget }: TabPanel.ICurrentChangedArgs): void {
    if (index < 0 && this.layoutContainer) {
      let newIndex = prevIndex >= 0 ? prevIndex : 0;
      newIndex = newIndex < this.layoutContainer.widgets.length ? newIndex : this.layoutContainer.widgets.length - 1;
      if (newIndex >= 0) {
        this.forceTabChange = true;
        this.layoutContainer.currentIndex = prevIndex;
        return;
      }
    }
    if (this.forceTabChange) {
      this.forceTabChange = false;
      return;
    }
    const id = currentWidget?.id ?? "";
    const prevId = previousWidget?.id ?? "";
    if (this.tabsDisabled && this.layoutContainer) {
      const args: CanChangeToDisabledTabArgs = { id, index, prevId, prevIndex, canChange: false };
      this.canChangeToDisabledTab.publish(args);
      if (!args.canChange) {
        this.forceTabChange = true;
        this.layoutContainer.currentIndex = prevIndex;
        return;
      }
    }
    this.publishCurrentChanged(id, index, prevId, prevIndex);
  }

  private publishCurrentChanged(id: string, index: number, prevId: string, prevIndex: number): void {
    this.currentChanged.publish({ index, id, prevIndex, prevId });
    if (this.props.onCurrentChanged) {
      this.props.onCurrentChanged({ index, id, prevIndex, prevId });
    }
  }

  restoreTabOrder(layout: LayoutType): void {
    const tabLayout = this.findLayout(layout);
    if (tabLayout && this.layoutContainer) {
      this.layoutContainer.restoreTabOrder(tabLayout);
    }
  }

  getTabIndex(id: string): number {
    if (this.layoutContainer) {
      return this.layoutContainer.widgets.findIndex(w => w.id === id);
    }
    return -1;
  }

  setTabsEnabled(enabled: boolean): void {
    if (!this.layoutContainer) {
      return;
    }
    this.tabsDisabled = !enabled;
    this.layoutContainer.tabsMovable = enabled;
    for (const widget of this.layoutContainer.widgets) {
      widget.title.closable = enabled;
    }
    if (enabled) {
      this.layoutContainer.tabBar.removeClass("lm-mod-disabled");
    }
    else {
      this.layoutContainer.tabBar.addClass("lm-mod-disabled");
    }
    if (enabled) {
      this.publishCurrentChanged(this.currentTabId, this.currentIndex, this.currentTabId, this.currentIndex);
    }
  }

  override render() {
    return this.renderComponent();
  }
}
