import React from "react";
import ReactDOM from "react-dom";
import { Widget } from "@lumino/widgets";
import { ModuleInfo, ViewLayoutItem, ViewLayoutItemType, ViewLayoutItemOrientation } from "@core/types";
import { RBaseWidgetProps, isWrapped, addWidget as addWidgetHelper } from "./RBaseWidget";
import { WrapperSplitPanel } from "./RSplitPanel";
import { WrapperWidget } from "./RWidget";
import WidgetErrorBoundary from "./WidgetErrorBoundary";
import { ViewWidgetProps, Widgets } from "components/Widgets";
import WidgetRender from "components/Widgets/WidgetRender";
import { ViewLayout } from "components/ViewLayouts";
import { copyObject } from "utils";
import { LayoutType, RWidgetLoader } from "./index";
import {__} from "@solid/libs/i18n";

type WidgetInfo = ViewLayoutItem & {
  widget?: Widget;
  node?: HTMLElement;
  component?: JSX.Element;
  children?: WidgetInfo[];
};

type RLayoutRenderProps = RBaseWidgetProps & {
  layout: ViewLayoutItem[];
  widgets?: ModuleInfo[];
  widgetProps: object;
  onGetLayout?: () => LayoutType | undefined;
  loader?: RWidgetLoader;
};

type RLayoutRenderState = {
  widgetInfo: WidgetInfo[];
};

export class RLayoutRender extends React.Component<RLayoutRenderProps, RLayoutRenderState> {
  private widgetsToClose?: Widget[];

  constructor(props: RLayoutRenderProps) {
    super(props);
    this.onWidgetsUpdating();
    this.state = {
      widgetInfo: this.createWidgets()
    };
    window.requestAnimationFrame(() => this.onWidgetsUpdated());
  }

  override shouldComponentUpdate(nextProps: RLayoutRenderProps, nextState: RLayoutRenderState): boolean {
    const propsCopy = copyObject(this.props, undefined, ["creatingView"]);
    const nextPropsCopy = copyObject(nextProps, undefined, ["creatingView"]);

    const layout: ViewLayoutItem[] = copyObject(
      this.state.widgetInfo, undefined, ["widget", "node", "component", "children"]) as ViewLayoutItem[];
    const nextLayout: ViewLayoutItem[] = copyObject(
      nextState.widgetInfo, undefined, ["widget", "node", "component", "children"]) as ViewLayoutItem[];

    return JSON.stringify(propsCopy) !== JSON.stringify(nextPropsCopy) ||
      JSON.stringify(layout) !== JSON.stringify(nextLayout) ||
      JSON.stringify(nextLayout) !== JSON.stringify(nextProps.layout);
  }

  /* eslint-disable react/no-did-update-set-state */
  override componentDidUpdate(prevProps: RLayoutRenderProps): void {
    const propsCopy = copyObject(this.props);
    const prevPropsCopy = copyObject(prevProps);

    const layout: ViewLayoutItem[] = copyObject(
      this.state.widgetInfo, undefined, ["widget", "node", "component", "children"]) as ViewLayoutItem[];

    if (JSON.stringify(propsCopy) !== JSON.stringify(prevPropsCopy) ||
      JSON.stringify(this.props.layout) !== JSON.stringify(layout)) {
      this.onWidgetsUpdating();
      this.widgetsToClose = this.state.widgetInfo.filter(info => !!info.widget).map(info => info.widget!);
      for (const widget of this.widgetsToClose) {
        widget.hide();
      }
      this.setState({ widgetInfo: this.createWidgets() });
      window.requestAnimationFrame(() => this.onWidgetsUpdated());
    }
  }
  /* eslint-enable react/no-did-update-set-state */

  private createWidgets(): WidgetInfo[] {
    let widgetIndex = 0;
    const getWidgetIndex = (next: boolean = true): number => (next ? widgetIndex++ : widgetIndex);
    return this.createLayoutWidgets(this.props, getWidgetIndex, 0);
  }

  private createLayoutWidgets(
    { id, name, index, addWidget, layout, widgets, widgetProps }: RLayoutRenderProps,
    getWidgetIndex: (next?: boolean) => number,
    level: number,
    widgetInfo: WidgetInfo[] = []): WidgetInfo[] {
    if (!addWidget) {
      return [];
    }

    const startWidgetIndex = getWidgetIndex(false);

    for (let i = 0; i < layout.length; i++) {
      const item = layout[i];
      const info: WidgetInfo = copyObject(item) as WidgetInfo;

      if (item.type === ViewLayoutItemType.Split && item.items) {
        const widgetId = `${id}_${i}`;

        const splitPanel = new WrapperSplitPanel(widgetId, name, { orientation: item.orientation });
        info.widget = splitPanel;

        this.setWidgetMinMaxSize(info.widget, layout, i, widgets, startWidgetIndex);
        addWidget(info.widget, index);

        const widget = info.widget;
        const addChildWidget = (dock: Widget, index?: number): Widget => addWidgetHelper(widget, dock, index);

        const props: RLayoutRenderProps = { id: widgetId, addWidget: addChildWidget, layout: item.items, widgets, widgetProps };
        info.children = this.createLayoutWidgets(props, getWidgetIndex, level + 1);

        const sizes = item.items.map(it => it.size);
        if (!this.props.onGetLayout || !this.props.onGetLayout()) {
          splitPanel.forceRelativeSizes(sizes);
        }
        else {
          splitPanel.setRelativeSizes(sizes);
        }
      }
      else if (item.type === ViewLayoutItemType.Cell) {
        const wi = getWidgetIndex();
        const module = this.getModule(wi, widgets);

        info.node = document.createElement("div");
        info.widget = new WrapperWidget(info.node, module.id, module.name, { index: module.index });
        if (typeof widgetProps["widgetClass"] === "string") {
          info.widget.addClass(widgetProps["widgetClass"]);
        }
        info.component = ReactDOM.createPortal(
          <WidgetErrorBoundary><WidgetRender module={module} index={wi} {...widgetProps}/></WidgetErrorBoundary>, info.node);

        this.setWidgetMinMaxSize(info.widget, layout, i, widgets, wi);
        addWidget(info.widget, index);
      }

      widgetInfo.push(info);
    }
    return widgetInfo;
  }

  private setWidgetMinMaxSize(widget: Widget, layout: ViewLayoutItem[], itemIndex: number, widgets: ModuleInfo[] | undefined, widgetIndex: number): void {
    if (!isWrapped(widget)) {
      return;
    }

    const loader = (this.props.widgetProps as ViewWidgetProps).loader;
    if (loader && loader.props.ignoreFixedSizes) {
      return;
    }

    const item = layout[itemIndex];
    let wi = widgetIndex;

    if (item.type === ViewLayoutItemType.Split) {
      const getWidgetIndex = (): number => wi++;

      const viewLayout = new ViewLayout(layout);
      viewLayout.foreach((it, parent, layout, parentLayout, i, wi, level) => {
        if (parent === item && it.type === ViewLayoutItemType.Cell) {
          const module = this.getModule(wi, widgets);
          if (module.props) {
            if (item.orientation === ViewLayoutItemOrientation.Horizontal) {
              const { fixedHeight } = module.props as Record<string, number>;
              if (fixedHeight !== undefined) {
                widget.wrapper.setMinMaxSize(undefined, undefined, fixedHeight, fixedHeight);
                return true;
              }
            }
            else if (item.orientation === ViewLayoutItemOrientation.Vertical) {
              const { fixedWidth } = module.props as Record<string, number>;
              if (fixedWidth !== undefined) {
                widget.wrapper.setMinMaxSize(fixedWidth, fixedWidth, undefined, undefined);
                return true;
              }
            }
          }
        }
        return false;
      },
      layout, 0, getWidgetIndex);
    }
    else if (item.type === ViewLayoutItemType.Cell) {
      const module = this.getModule(wi, widgets);
      let minWidth: number | undefined;
      let maxWidth: number | undefined;
      let minHeight: number | undefined;
      let maxHeight: number | undefined;
      if (module.props) {
        const { fixedWidth, fixedHeight } = module.props as Record<string, number>;
        if (fixedWidth !== undefined) {
          minWidth = fixedWidth;
          maxWidth = fixedWidth;
        }
        if (fixedHeight !== undefined) {
          minHeight = fixedHeight;
          maxHeight = fixedHeight;
        }
      }
      const widg = Widgets.find(w => w.id === module.widgetId);
      if (widg?.options.minWidth !== undefined) {
        if (minWidth !== undefined && minWidth < widg?.options.minWidth) {
          minWidth = widg?.options.minWidth;
        }
        if (maxWidth !== undefined && maxWidth < widg?.options.minWidth) {
          maxWidth = widg?.options.minWidth;
        }
        if (minWidth === undefined) {
          minWidth = widg?.options.minWidth;
        }
      }
      if (widg?.options.minHeight !== undefined) {
        if (minHeight !== undefined && minHeight < widg?.options.minHeight) {
          minHeight = widg?.options.minHeight;
        }
        if (maxHeight !== undefined && maxHeight < widg?.options.minHeight) {
          maxHeight = widg?.options.minHeight;
        }
        if (minHeight === undefined) {
          minHeight = widg?.options.minHeight;
        }
      }
      if (minWidth !== undefined || maxWidth !== undefined || minHeight !== undefined || maxHeight !== undefined) {
        widget.wrapper.setMinMaxSize(minWidth, maxWidth, minHeight, maxHeight);
      }
    }
  }

  private getModule(i: number, widgets?: ModuleInfo[]): ModuleInfo {
    const w = widgets && i < widgets.length ? widgets[i] : undefined;
    return {
      ...w,
      id: w?.id || `${this.props.id}_widget${i}`,
      name: w?.name || `${__("Widget")} ${i + 1}`,
      module: w?.module ?? ""
    };
  }

  private onWidgetsUpdating(): void {
    this.props.loader?.setLayoutChangeEnabled(false);
  }

  private onWidgetsUpdated(): void {
    if (this.widgetsToClose) {
      for (const widget of this.widgetsToClose) {
        widget.close();
      }
      this.widgetsToClose = undefined;
    }
    this.restoreLayout();
    this.props.loader?.setLayoutChangeEnabled(true);
  }

  private restoreLayout(): void {
    const layout = this.props.onGetLayout && this.props.onGetLayout();
    if (!layout) {
      return;
    }
    let widget = this.state.widgetInfo.find(info => !!info.widget)?.widget;
    while (widget && widget.id !== this.props.id) {
      widget = widget.parent ?? undefined;
    }
    if (widget && isWrapped(widget)) {
      widget.wrapper.restoreLayout({ [widget.id]: layout });
    }
  }

  private getComponents(widgetInfo: WidgetInfo[], components: JSX.Element[] = []): JSX.Element[] {
    for (const info of widgetInfo) {
      if (info.component) {
        components.push(info.component);
      }
      if (info.children) {
        this.getComponents(info.children, components);
      }
    }
    return components;
  }

  override render() {
    return this.getComponents(this.state.widgetInfo);
  }
}
