import React from "react";
import { Widget } from "@lumino/widgets";
import { EventPubSub, EventPubSubFunc } from "utils";
import { AddWidgetFunc, WidgetCloseArgs, LayoutChangedArgs, LayoutType } from "components/Layout";

export interface RBaseWidgetProps {
  id: string,
  name?: string;
  index?: number;
  addWidget?: AddWidgetFunc;
  onClose?: WidgetCloseFunc;
  rootElementId?: string;
  layoutElementId?: string;
  minWidth?: number;
  maxWidth?: number;
  minHeight?: number;
  maxHeight?: number;
  widgetClass?: string;
}

export class WrapperHelper {
  minWidth?: number;
  maxWidth?: number;
  minHeight?: number;
  maxHeight?: number;
  readonly widgetClose = new WidgetCloseEvent();
  readonly layoutChanged = new WidgetLayoutChangedEvent();
  readonly layoutRestored = new WidgetLayoutRestoredEvent();
  readonly widgetResize = new WidgetResizeEvent();
  private layoutChangedDisabled = false;
  private layoutChangedDisabledForced = false;
  layoutChangedDisabledAlways = false;

  constructor(readonly widget: Widget, id: string, public name?: string) {
    this.widget.id = id;
    if (this.name !== undefined) {
      this.widget.title.label = this.name;
      this.widget.title.closable = true;
    }
  }

  setMinMaxSize(minWidth?: number, maxWidth?: number, minHeight?: number, maxHeight?: number): void {
    this.minWidth = minWidth;
    this.maxWidth = maxWidth;
    this.minHeight = minHeight;
    this.maxHeight = maxHeight;

    if (minWidth && minWidth === maxWidth) {
      this.widget.addClass("fixed-width");
    }
    if (minHeight && minHeight === maxHeight) {
      this.widget.addClass("fixed-height");
    }
  }

  publishClose() {
    this.widgetClose.publish({ id: this.widget.id, name: this.name });
    this.widgetClose.unsubscribeAll();
    this.layoutChanged.unsubscribeAll();
  }

  publishLayoutChanged(id: string) {
    if (this.layoutChangedDisabled || this.layoutChangedDisabledForced || this.layoutChangedDisabledAlways) {
      return;
    }
    this.layoutChanged.publish({ id });
    let parent = this.widget.parent;
    while (parent) {
      if (isWrapped(parent)) {
        parent.wrapper.publishLayoutChanged(id);
        break;
      }
      parent = parent.parent;
    }
  }

  getWrappedWidgetsRecursive(): IWrappedWidget[] {
    let widgets: IWrappedWidget[] = [];
    if (isWrapped(this.widget)) {
      widgets.push(this.widget);
    }
    const children = getWrappedChildren(this.widget);
    for (const child of children) {
      widgets = widgets.concat(child.wrapper.getWrappedWidgetsRecursive());
    }
    return widgets;
  }

  getLayout(): LayoutType | undefined {
    let layout: LayoutType = {};
    if (isLayoutWidget(this.widget)) {
      const selfLayout = this.widget.getLayout();
      if (selfLayout) {
        layout = selfLayout;
      }
    }
    const children = getWrappedChildren(this.widget);
    const widgets: LayoutType = {};
    for (const child of children) {
      const childLayout = child.wrapper.getLayout();
      if (childLayout && childLayout[child.id]) {
        widgets[child.id] = childLayout[child.id];
      }
    }
    if (Object.keys(widgets).length > 0) {
      layout.widgets = widgets;
    }
    return Object.keys(layout).length > 0 ? { [this.widget.id]: layout } : undefined;
  }

  restoreLayout(obj: LayoutType, isChild?: boolean, skipDisableEvent?: boolean, widgetsMap?: Map<string, ILayoutWidget>): void {
    const layout = isChild ? obj : obj[this.widget.id];
    if (!layout) {
      return;
    }

    const layoutWidgets = widgetsMap || new Map<string, ILayoutWidget>();
    const subscriptionId = `${Date.now()}.${Math.random()}`;
    if (!skipDisableEvent) {
      this.setLayoutChangedEnabled(false);
      for (const widget of this.getWrappedWidgetsRecursive()) {
        if (isLayoutWidget(widget)) {
          layoutWidgets.set(widget.id, widget);
          widget.wrapper.layoutRestored.subscribe(subscriptionId, ({ id }: WidgetLayoutRestoredArgs) => {
            const widget = layoutWidgets.get(id);
            if (widget && isWrapped(widget)) {
              widget.wrapper.layoutRestored.unsubscribe(subscriptionId);
            }
            layoutWidgets.delete(id);
            if (layoutWidgets.size === 0) {
              this.setLayoutChangedEnabled(true);
            }
          });
        }
      }
    }

    if (isLayoutWidget(this.widget)) {
      this.widget.restoreLayout(layout);
    }
    else {
      layoutWidgets.delete(this.widget.id);
    }

    const children = getWrappedChildren(this.widget);
    const restoreIds = new Set<string>();
    if (layout.widgets) {
      for (const id in layout.widgets) {
        if (layout.widgets[id]) {
          const widget = children.find(w => w.id === id);
          if (widget) {
            restoreIds.add(widget.id);
            widget.wrapper.restoreLayout(layout.widgets[id], true, true, layoutWidgets);
          }
        }
      }
    }

    for (const { id } of children.filter(({ id }) => !restoreIds.has(id))) {
      const widget = layoutWidgets.get(id);
      if (widget) {
        layoutWidgets.delete(id);
        if (isWrapped(widget)) {
          for (const child of widget.wrapper.getWrappedWidgetsRecursive()) {
            layoutWidgets.delete(child.id);
          }
        }
      }
    }

    if (!skipDisableEvent && layoutWidgets.size === 0) {
      this.setLayoutChangedEnabled(true);
    }
  }

  setLayoutChangedEnabled(enabled: boolean): void {
    this.layoutChangedDisabled = !enabled;
    if (enabled) {
      this.layoutChanged.enable();
    }
    else {
      this.layoutChanged.disable();
    }
    const children = getWrappedChildren(this.widget);
    for (const child of children) {
      child.wrapper.setLayoutChangedEnabled(enabled);
    }
  }

  forceLayoutChangedEnabled(enabled: boolean): void {
    this.layoutChangedDisabledForced = !enabled;
  }

  disableLayout(): void {
    this.layoutChangedDisabledAlways = true;
  }
}

export interface IWrappedWidget extends Widget {
  wrapper: WrapperHelper;
}

export function isWrapped(widget: Widget): widget is IWrappedWidget {
  return !!(widget as IWrappedWidget).wrapper;
}

export type WidgetCloseFunc = EventPubSubFunc<WidgetCloseArgs>;

export class WidgetCloseEvent extends EventPubSub<WidgetCloseArgs> {}

class LayoutChangedEvent extends EventPubSub<LayoutChangedArgs> {}

class WidgetResizeEvent extends EventPubSub<Widget.ResizeMessage> {}

type WidgetLayoutChangedArgs = {
  id: string;
};

class WidgetLayoutChangedEvent extends EventPubSub<WidgetLayoutChangedArgs> {}

type WidgetLayoutRestoredArgs = {
  id: string;
};

class WidgetLayoutRestoredEvent extends EventPubSub<WidgetLayoutRestoredArgs> {}

export interface IAddWidget extends Widget {
  addWidget(widget: Widget): void;
}

export interface IAddInsertWidget extends IAddWidget {
  insertWidget(index: number, widget: Widget): void;
}

export function isAddWidget(widget: Widget): widget is IAddWidget {
  return !!(widget as IAddWidget).addWidget;
}

export function isAddInsertWidget(widget: Widget): widget is IAddInsertWidget {
  return isAddWidget(widget) && !!(widget as IAddInsertWidget).insertWidget;
}

export interface ILayoutWidget extends Widget {
  getLayout(): LayoutType | undefined;
  restoreLayout(layout: LayoutType): void;
}

export function isLayoutWidget(widget: Widget): widget is ILayoutWidget {
  return !!(widget as ILayoutWidget).getLayout && !!(widget as ILayoutWidget).restoreLayout;
}

export abstract class RBaseWidget<TProps extends RBaseWidgetProps, TWidget extends Widget> extends React.PureComponent<TProps> {
  readonly layoutElementId: string;
  readonly layoutContainer?: TWidget;
  readonly widget?: TWidget;
  readonly widgetClose = new WidgetCloseEvent();
  readonly layoutChanged = new LayoutChangedEvent();
  protected subscriptionId = `${Date.now()}.${Math.random()}`;
  protected layoutChangedTimeout: NodeJS.Timeout | undefined;

  constructor(props: TProps) {
    super(props);

    this.layoutElementId = this.props.layoutElementId ?? "main";

    this.layoutContainer = this.createLayoutContainer();

    if (this.layoutContainer) {
      this.layoutContainer.id = this.props.id;
    }

    this.widget = this.getWidgetToAdd();

    if (this.widget && isWrapped(this.widget)) {
      const { minWidth, maxWidth, minHeight, maxHeight, widgetClass = "" } = this.props;
      this.widget.wrapper.setMinMaxSize(minWidth, maxWidth, minHeight, maxHeight);

      if (widgetClass) {
        this.widget.addClass(widgetClass);
      }
    }

    if (props.addWidget && this.widget) {
      props.addWidget(this.widget, this.props.index);
    }

    if (this.widget && isWrapped(this.widget)) {
      this.widget.wrapper.widgetClose.subscribe(this.subscriptionId, (args: WidgetCloseArgs) => this.onClose(args));
      this.widget.wrapper.layoutChanged.subscribe(this.subscriptionId, (args: WidgetLayoutChangedArgs) => this.onLayoutChanged(args));
    }

    this.onWindowResize = this.onWindowResize.bind(this);

    if (this.layoutContainer) {
      if (this.layoutContainer.id === this.layoutElementId) {
        const main = document.getElementById(this.layoutElementId);
        if (main) {
          main.remove();
        }
        window.addEventListener("resize", this.onWindowResize);
        if (!props.rootElementId) {
          throw new Error("Root element ID is not defined for the main widget.");
        }
        const root = document.getElementById(props.rootElementId);
        if (!root) {
          throw new Error(`Root element #${props.rootElementId} for attaching main widget is not found.`);
        }
        Widget.attach(this.layoutContainer, root);
        setMinMaxSize(this.layoutContainer);
      }
    }
  }

  override componentWillUnmount() {
    this.widgetClose.unsubscribeAll();
    this.layoutChanged.unsubscribeAll();
    if (this.widget && isWrapped(this.widget)) {
      this.widget.wrapper.widgetClose.unsubscribe(this.subscriptionId);
      this.widget.wrapper.layoutChanged.unsubscribe(this.subscriptionId);
    }
    if (this.layoutContainer && this.layoutContainer.id === this.layoutElementId) {
      window.removeEventListener("resize", this.onWindowResize);
      const main = document.getElementById(this.layoutElementId);
      if (main) {
        main.remove();
      }
    }
  }

  protected abstract createLayoutContainer(): TWidget | undefined;

  protected getWidgetToAdd(): TWidget | undefined {
    return this.layoutContainer;
  }

  private onWindowResize(): void {
    if (this.layoutContainer) {
      this.layoutContainer.update();
    }
  }

  addWidget(dock: Widget, index?: number): Widget {
    if (this.layoutContainer) {
      return addWidget(this.layoutContainer, dock, index);
    }
    return dock;
  }

  getTitle(widgetId: string): string {
    const widget = this.layoutContainer ? findChild(this.layoutContainer, widgetId) : undefined;
    return widget ? widget.title.label : "";
  }

  setTitle(widgetId: string, title: string): void {
    const widget = this.layoutContainer ? findChild(this.layoutContainer, widgetId) : undefined;
    if (widget) {
      widget.title.label = title;
    }
  }

  setClosable(widgetId: string, closable: boolean): void {
    const widget = this.layoutContainer ? findChild(this.layoutContainer, widgetId) : undefined;
    if (widget) {
      widget.title.closable = closable;
    }
  }

  setEditIcon(widgetId: string, edit: boolean): void {
    const widget = this.layoutContainer ? findChild(this.layoutContainer, widgetId) : undefined;
    if (widget) {
      widget.title.iconClass = edit ? "lm-TabBar-tabEditIcon" : "";
    }
  }

  findWidget(widgetId: string): Widget | undefined {
    if (this.widget && this.widget.id === widgetId) {
      return this.widget;
    }
    return this.layoutContainer ? findChild(this.layoutContainer, widgetId) : undefined;
  }

  addClass(widgetId: string, className: string): boolean {
    const widget = this.findWidget(widgetId);
    if (widget) {
      widget.addClass(className);
    }
    return !!widget;
  }

  removeClass(widgetId: string, className: string): boolean {
    const widget = this.findWidget(widgetId);
    if (widget) {
      widget.removeClass(className);
    }
    return !!widget;
  }

  setSelected(widgetId: string, selected: boolean): boolean {
    return selected ? this.addClass(widgetId, "Widget-Content_selected") : this.removeClass(widgetId, "Widget-Content_selected");
  }

  getLayout(): LayoutType | undefined {
    return this.widget && isWrapped(this.widget) ? this.widget.wrapper.getLayout() : undefined;
  }

  restoreLayout(layout: LayoutType): void {
    if (this.widget && isWrapped(this.widget)) {
      this.widget.wrapper.restoreLayout(layout);
    }
  }

  getWidgetLayout(id: string): LayoutType | undefined {
    if (!this.widget) {
      return undefined;
    }
    const widget = findChild(this.widget, id);
    return widget && isWrapped(widget) ? widget.wrapper.getLayout() : undefined;
  }

  protected onClose(args: WidgetCloseArgs): void {
    if (this.widget && isWrapped(this.widget)) {
      this.widget.wrapper.widgetClose.unsubscribe(this.subscriptionId);
    }
    this.widgetClose.publish(args);
    this.widgetClose.unsubscribeAll();
    if (this.props.onClose) {
      this.props.onClose(args);
    }
  }

  protected onLayoutChanged({ id }: WidgetLayoutChangedArgs): void {
    if (this.layoutChanged.subscribersCount > 0) {
      if (this.layoutChangedTimeout) {
        clearTimeout(this.layoutChangedTimeout);
      }
      this.layoutChangedTimeout = setTimeout(() => this.publishLayoutChanged(id), 1000);
    }
  }

  protected publishLayoutChanged(id: string): void {
    const layout = this.getLayout();
    if (layout) {
      this.layoutChanged.publish({ id, layout });
    }
  }

  findLayout(layout: LayoutType): LayoutType | undefined {
    if (layout[this.props.id]) {
      return layout[this.props.id];
    }
    if (layout.widgets) {
      return this.findLayout(layout.widgets);
    }
    for (const id in layout) {
      const childLayout = this.findLayout(layout[id]);
      if (childLayout) {
        return childLayout;
      }
    }
    return undefined;
  }

  renderComponent(): React.ReactNode | undefined {
    const children = React.Children.map(this.props.children, child => {
      if (!React.isValidElement(child)) return undefined;
      return React.cloneElement(child, { ...child.props, addWidget: (dock: Widget, index?: number) => this.addWidget(dock, index) });
    });
    return children;
  }
}

export function findChild(widget: Widget, widgetId: string): Widget | undefined {
  const children = widget.children();
  let child = children.next();
  while (child) {
    if (child.id === widgetId) {
      return child;
    }
    child = findChild(child, widgetId);
    if (child) {
      return child;
    }
    child = children.next();
  }
  return undefined;
}

export function getChildren(widget: Widget): Widget[] {
  const result: Widget[] = [];
  const children = widget.children();
  let child = children.next();
  while (child) {
    result.push(child);
    child = children.next();
  }
  return result;
}

export function getWrappedChildren(widget: Widget): IWrappedWidget[] {
  let wrapped: IWrappedWidget[] = [];
  const children = getChildren(widget);
  for (const child of children) {
    if (isWrapped(child)) {
      wrapped.push(child);
    }
    else {
      wrapped = wrapped.concat(getWrappedChildren(child));
    }
  }
  return wrapped;
}

export function addWidget(container: Widget, dock: Widget, index?: number): Widget {
  if (isAddInsertWidget(container)) {
    if (index === undefined) {
      container.addWidget(dock);
    }
    else {
      container.insertWidget(index, dock);
    }
  }
  else if (isAddWidget(container)) {
    container.addWidget(dock);
  }
  setMinMaxSize(dock);
  return dock;
}

export function setMinMaxSize(widget: Widget, unset?: boolean): void {
  if (isWrapped(widget)) {
    const elem = widget.node;
    const { minWidth, maxWidth, minHeight, maxHeight } = widget.wrapper;
    if (minWidth !== undefined) {
      elem.style.minWidth = minWidth + "px";
    }
    else if (unset) {
      elem.style.minWidth = "";
    }
    if (maxWidth !== undefined) {
      elem.style.maxWidth = maxWidth + "px";
    }
    else if (unset) {
      elem.style.maxWidth = "";
    }
    if (minHeight !== undefined) {
      elem.style.minHeight = minHeight + "px";
    }
    else if (unset) {
      elem.style.minHeight = "";
    }
    if (maxHeight !== undefined) {
      elem.style.maxHeight = maxHeight + "px";
    }
    else if (unset) {
      elem.style.maxHeight = "";
    }
  }
}

export function getWidgetRect(widget: Widget): DOMRect {
  return widget.node.getBoundingClientRect();
}
