import React, { useState, useEffect, useRef } from "react";
import { useApolloClient } from "@apollo/client";
import { Input, Popup, Icon, Button, Modal, Header, Ref } from "semantic-ui-react";
import {
  Label,
  LabelInput,
  DevicesByAspectTypesShortDocument,
  DevicesByAspectTypesShortQuery,
  DevicesByAspectTypesShortQueryVariables,
  DeviceFunctionalAspectType,
  HierarchiesDocument,
  HierarchiesQuery,
  HierarchiesQueryVariables,
  Hierarchy,
  HierarchyLevel,
  LabelType
} from "@generated/graphql";
import { useLabelActions } from "@core/actions";
import { EventPubSub } from "utils";
import { getLabelIcon } from "components/LabelsAndHierarchies/utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";

import "./style.css";

export type EditLabelEventArgs = {
  id: string;
};

export class EditLabelEvent extends EventPubSub<EditLabelEventArgs> {}

type LabelHierarchy = {
  hierarchy: Hierarchy;
  levels: HierarchyLevel[];
};

type LabelListItemProps = {
  label: Label;
  labels: Label[];
  editable: boolean;
  editEvent: EditLabelEvent;
  onLoading: (loading: boolean) => void;
  onUpdating: (updating: boolean) => void;
};

const LabelListItem = ({ label, labels, editable, editEvent, onLoading, onUpdating }: LabelListItemProps) => {
  const { id, name, objects } = label;
  const [isEdit, setIsEdit] = useState(false);
  const [editName, setEditName] = useState(name);
  const [okDisabled, setOkDisabled] = useState(false);
  const inputRef = useRef<Input>(null);
  const okButtonRef = useRef<HTMLButtonElement>(null);
  const { updateLabel, deleteLabel } = useLabelActions();
  const [deleteDenyOpen, setDeleteDenyOpen] = useState(false);
  const [deleteDenyText, setDeleteDenyText] = useState<React.ReactNode>("");
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [deleteConfirmText, setDeleteConfirmText] = useState<React.ReactNode>("");
  const client = useApolloClient();

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    editEvent.subscribe(id, args => {
      if (args.id !== label.id) {
        setIsEdit(false);
      }
    });

    return () => {
      editEvent.unsubscribe(id);
    };
  }, []);

  useEffect(() => {
    setEditName(name);
  }, [name]);

  useEffect(() => {
    setOkDisabled(!editName || labels.some(label => label.id !== id && label.name === editName));
  }, [labels, id, editName]);

  function editLabel(e: React.MouseEvent): void {
    e.stopPropagation();
    setIsEdit(true);
    setEditName(name);
    editEvent.publish({ id });
    window.requestAnimationFrame(() => {
      if (inputRef.current) {
        inputRef.current.focus();
      }
    });
  }

  function updateLabelName(e: React.UIEvent): void {
    e.stopPropagation();
    const label: LabelInput = { id, name: editName };
    setIsEdit(false);
    editEvent.publish({ id: "" });
    onUpdating(true);
    updateLabel(label)
      .catch(e => Log.error(`${__("Label update error")}: ${e.message}`))
      .finally(() => onUpdating(false));
  }

  function cancelEditLabel(e?: React.UIEvent): void {
    e?.stopPropagation();
    setIsEdit(false);
    editEvent.publish({ id: "" });
  }

  async function destroyLabelOrConfirm(e: React.MouseEvent): Promise<void> {
    e.stopPropagation();

    onLoading(true);
    try {
      const hierarchies = await getLabelHierarchies();
      if (hierarchies.length > 0) {
        const names = hierarchies
          .map(hierarchy => getHierarchyText(hierarchy))
          .sort((a, b) => a.localeCompare(b, undefined, { sensitivity: "base" }));

        const text =
          <div>
            {__("Label '{{name}}' cannot be deleted, it is used in the following hierarchy:", {name})}
            <ul>
              {names.slice(0, 3).map((name, index) => <li key={`${name}_${index}`}>{name}</li>)}
            </ul>
            {names.length > 3 && <>{__("...and {{number}} more.", {number: names.length - 3})}<br/><br/></>}
            {__("Please use hierarchy editor to remove this label from every hierarchy.")}
          </div>;

        setDeleteDenyText(text);
        setDeleteDenyOpen(true);
        return;
      }

      if (objects.length === 0) {
        destroyLabel();
        return;
      }

      const { data } = await client.query<DevicesByAspectTypesShortQuery, DevicesByAspectTypesShortQueryVariables>({
        query: DevicesByAspectTypesShortDocument,
        variables: { types: [{ type: DeviceFunctionalAspectType.Media }] }
      });

      const cameras = objects
        .map(({ objectId }) => ({ id: objectId, name: data.devicesByAspectTypes.find(dev => dev.id === objectId)?.name ?? "" }))
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));

      const text =
        <div>
          {`${__("The following camera(s) are marked with label")} '${name}':`}
          <ul>
            {cameras.slice(0, 3).map(({ id, name }, index) => <li key={`${id}_${index}`}>{name}</li>)}
          </ul>
          {cameras.length > 3 && <>{__("...and {{number}} more.", {number: cameras.length - 3})}<br/><br/></>}
          {`${__("Are you sure you want to delete label")} '${name}'?`}
        </div>;

      setDeleteConfirmText(text);
      setDeleteConfirmOpen(true);
    }
    catch (e: any) {
      Log.error(`${__("Devices query error")}: ${e.message}`);
    }
    finally {
      onLoading(false);
    }
  }

  function destroyLabel(): void {
    onUpdating(true);
    deleteLabel(id)
      .catch(e => Log.error(`${__("Label delete error")}: ${e.message}`))
      .finally(() => onUpdating(false));
  }

  async function getLabelHierarchies(): Promise<LabelHierarchy[]> {
    const { data } = await client.query<HierarchiesQuery, HierarchiesQueryVariables>({ query: HierarchiesDocument });
    const hierarchies: LabelHierarchy[] = [];
    for (const hierarchy of data.hierarchies) {
      const path: HierarchyLevel[] = [];
      const levelLevels = hierarchy.level.levels ?? [];
      getLabelLevels(hierarchies, hierarchy, path, levelLevels);
    }
    return hierarchies;
  }

  function getLabelLevels(hierarchies: LabelHierarchy[], hierarchy: Hierarchy, path: HierarchyLevel[], levels: HierarchyLevel[]): void {
    for (const level of levels) {
      path.push(level);
      const levelLevels = level.levels ?? [];
      if (levelLevels.length === 0 && level.labelIds.includes(id)) {
        hierarchies.push({ hierarchy, levels: [...path] });
      }
      else {
        getLabelLevels(hierarchies, hierarchy, path, levelLevels);
      }
      path.pop();
    }
  }

  function getHierarchyText({ hierarchy, levels }: LabelHierarchy): string {
    let text = hierarchy.name;
    for (const level of levels) {
      text += "/" + level.name;
    }
    return text;
  }

  function onInputKeyDown(e: React.KeyboardEvent): void {
    e.stopPropagation();
    if (e.key === "Enter" && !okDisabled) {
      updateLabelName((e as unknown) as React.UIEvent);
    }
    if (e.key === "Escape") {
      cancelEditLabel((e as unknown) as React.UIEvent);
    }
  }

  return (
    <div className="LabelListItem">
      {isEdit ?
        <Input
          ref={inputRef}
          placeholder={__("Label name")}
          fluid
          action
          value={editName}
          onChange={e => setEditName(e.currentTarget.value)}
          onClick={(e: React.MouseEvent) => e.stopPropagation()}
          onKeyDown={onInputKeyDown}
          onKeyUp={(e: React.KeyboardEvent) => e.stopPropagation()}
          onBlur={(e: React.FocusEvent) => {
            if (e.nativeEvent.relatedTarget !== okButtonRef.current) {
              cancelEditLabel();
            }
          }}>
          <input/>
          <Ref innerRef={okButtonRef}>
            <Button positive icon disabled={okDisabled} onClick={e => updateLabelName((e as unknown) as React.MouseEvent)}><Icon name="check"/></Button>
          </Ref>
          <Button negative icon onClick={e => cancelEditLabel((e as unknown) as React.MouseEvent)}><Icon name="cancel"/></Button>
        </Input> :
        <>
          <Icon name={getLabelIcon(label)}/>
          <div className="LabelListItem-Name">{name}</div>
          <div className="LabelListItem-Buttons">
            {editable && label.type === LabelType.Label &&
            <>
              <Popup content={__("Edit Label")} trigger={<Icon name="edit" onClick={editLabel}/>}/>
              <Popup content={__("Delete Label")} trigger={<Icon name="trash alternate" onClick={destroyLabelOrConfirm}/>}/>
            </>}
            <div className="LabelListItem-Count">({objects.length})</div>
          </div>
        </>}

      {deleteDenyOpen &&
        <Modal open={deleteDenyOpen} onClose={() => setDeleteDenyOpen(false)} onClick={(e: React.MouseEvent) => e.stopPropagation()}>
          <Header content={__("Delete Label")}/>
          <Modal.Content>
            {deleteDenyText}
          </Modal.Content>
          <Modal.Actions>
            <Button positive onClick={() => setDeleteDenyOpen(false)}><Icon name="checkmark"/>{__("OK")}</Button>
          </Modal.Actions>
        </Modal>}

      {deleteConfirmOpen &&
        <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)} onClick={(e: React.MouseEvent) => e.stopPropagation()}>
          <Header content={__("Delete Label")}/>
          <Modal.Content>
            {deleteConfirmText}
          </Modal.Content>
          <Modal.Actions>
            <Button
              negative
              onClick={() => {
                destroyLabel();
                setDeleteConfirmOpen(false);
              }}>
              <Icon name="trash"/>{__("Delete")}
            </Button>
            <Button onClick={() => setDeleteConfirmOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>}
    </div>
  );
};

export default LabelListItem;
