import React, { useState, useEffect, useRef, useMemo } from "react";
import { Header, Dropdown, DropdownItemProps, DropdownProps, Divider, Icon, Popup } from "semantic-ui-react";
import { Label, useLabelsQuery } from "@generated/graphql";
import LabelListItem, { EditLabelEvent } from "components/LabelsAndHierarchies/LabelListItem";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import LabelEditorDialog from "components/LabelsAndHierarchies/LabelEditorDialog";
import { isArray } from "utils";
import { Utils } from "@solid/libs/utils";
import {__} from "@solid/libs/i18n";

import "./style.css";

type LabelListProps = {
  selectedLabels?: Label[];
  onSelectionChange?: (labels: Label[]) => void;
  editable?: boolean;
  onBeforeLabelEditorOpen?: () => Promise<void>;
  onLabelEditorOpen?: (open: boolean) => void;
  onUpdating?: (updating: boolean) => void;
};

enum CustomSearchOptions {
  ShowAll = "ShowAll",
  Divider = "Divider"
}

const customSearchOptions: string[] = [CustomSearchOptions.ShowAll, CustomSearchOptions.Divider];

const initialLoadLimit = 10;
const itemHeight = 34;
const dividerHeight = 6;

const LabelList = ({ selectedLabels = [], onSelectionChange, editable = true, onBeforeLabelEditorOpen, onLabelEditorOpen, onUpdating }: LabelListProps) => {
  const { data, loading, error } = useLabelsQuery();
  const [searchOptions, setSearchOptions] = useState<DropdownItemProps[]>([]);
  const [searchValues, setSearchValues] = useState<string[]>([]);
  const [searchQuery, setSearchQuery] = useState("");
  const [loadLimit, setLoadLimit] = useState(initialLoadLimit);
  const [labelLoading, setLabelLoading] = useState(false);
  const [updating, setUpdating] = useState(false);
  const [labelEditorOpen, setLabelEditorOpen] = useState(false);
  const labelMap = useMemo(() => getLabelMap(), [data]);
  const editEventRef = useRef(new EditLabelEvent());
  const isEditRef = useRef(false);
  const containerRef = useRef<HTMLDivElement>(null);
  const resizeObserverRef = useRef<ResizeObserver>();

  useEffect(() => {
    const editEvent = editEventRef.current;
    const id = `${Date.now()}.${Math.random()}`;
    editEvent.subscribe(id, args => {
      isEditRef.current = !!args.id;
    });

    const container = containerRef.current;
    const dropdown: HTMLElement | null | undefined = container?.querySelector(".LabelList-Dropdown");
    const menu: HTMLElement | null | undefined = container?.querySelector(".LabelList-Dropdown .menu");
    if (container && dropdown) {
      resizeObserverRef.current = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
        let resized = false;
        for (const entry of entries) {
          if (entry.target === container || entry.target === dropdown) {
            resized = true;
            break;
          }
        }

        if (resized && menu) {
          const rect = container.getBoundingClientRect();
          const menuRect = menu.getBoundingClientRect();
          const height = rect.bottom - menuRect.top - 4;
          if (height !== menuRect.height) {
            menu.style.height = `${height}px`;
          }
          updateLoadLimit();
        }
      }));

      resizeObserverRef.current.observe(container);
      resizeObserverRef.current.observe(dropdown);
    }

    if (dropdown) {
      dropdown.addEventListener("keydown", ev => {
        if (ev.key === "ArrowDown" || ev.key === "ArrowUp" || ev.key === "Enter") {
          if (!isEditRef.current) {
            ev.stopPropagation();
          }
        }
      });
    }

    window.addEventListener("keydown", onWindowKeyDown, true);

    return function cleanup() {
      window.removeEventListener("keydown", onWindowKeyDown, true);

      if (container && dropdown) {
        resizeObserverRef.current?.unobserve(container);
        resizeObserverRef.current?.unobserve(dropdown);
      }
      editEvent.unsubscribe(id);
    };
  }, []);

  useEffect(() => {
    error && console.error("Label query error:", error);
  }, [error]);

  useEffect(() => {
    setSearchValues(values => {
      const newValues = selectedLabels.map(label => label.id);
      if (JSON.stringify(values) === JSON.stringify(newValues)) {
        return values;
      }
      return newValues;
    });
  }, [selectedLabels]);

  useEffect(() => {
    const limit = updateLoadLimit();
    setLoadLimit(limit || initialLoadLimit);
  }, [searchQuery]);

  useEffect(() => {
    updateLoadLimit();
  }, [data, searchValues]);

  useEffect(() => {
    if (!data) {
      return;
    }

    let options: DropdownItemProps[];
    let showAllVisible = false;

    let unselectedLabels = data.labels.filter(label => !searchValues.includes(label.id));
    if (searchQuery) {
      unselectedLabels = unselectedLabels.filter(({ name }) => name.toLocaleUpperCase().includes(searchQuery.toLocaleUpperCase()));
    }

    if (loadLimit > 0) {
      const selectedLabels = data.labels.filter(label => searchValues.includes(label.id));

      showAllVisible = unselectedLabels.length > loadLimit;
      unselectedLabels = unselectedLabels.sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" })).slice(0, loadLimit);

      options = selectedLabels.concat(unselectedLabels).map(label => getOption(label));
    }
    else {
      options = data.labels.map(label => getOption(label));
    }

    options = options.sort((a, b) => a.text?.toString().localeCompare(b.text?.toString() ?? "", undefined, { sensitivity: "base" }) ?? 0);

    if (showAllVisible) {
      options.push({ key: CustomSearchOptions.Divider, value: CustomSearchOptions.Divider, text: "", content: <Divider/>, disabled: true });
      options.push({ key: CustomSearchOptions.ShowAll, value: CustomSearchOptions.ShowAll, text: __("Show all labels"), className: "LabelList-ShowAll" });
    }

    setSearchOptions(options);
  }, [data, searchValues, searchQuery, loadLimit]);

  useEffect(() => {
    onLabelEditorOpen && onLabelEditorOpen(labelEditorOpen);
  }, [labelEditorOpen]);

  function getLabelMap(): Map<string, Label> {
    const map = new Map<string, Label>();
    if (!data) {
      return map;
    }
    for (const label of data.labels) {
      map.set(label.id.toString(), label);
    }
    return map;
  }

  /* eslint-disable react/jsx-indent */
  function getOption(label: Label): DropdownItemProps {
    const { id, name } = label;
    return {
      key: id,
      value: id,
      text: name,
      content:
        <LabelListItem
          label={label}
          labels={data?.labels ?? []}
          editable={editable}
          editEvent={editEventRef.current}
          onLoading={loading => setLabelLoading(loading)}
          onUpdating={updating => changeUpdating(updating)}
        />
    };
  }
  /* eslint-enable react/jsx-indent */

  function filterSearchOptions(options: DropdownItemProps[], searchValue: string): DropdownItemProps[] {
    return options.filter(({ value, text }) =>
      value &&
      (customSearchOptions.includes(value.toString()) ||
       text?.toString().toLocaleUpperCase().includes(searchValue.toLocaleUpperCase())));
  }

  function onSearchValueChange(e: React.SyntheticEvent, { value }: DropdownProps): void {
    if (!isArray<string>(value, "string") || !data) {
      return;
    }

    const newValues = value.filter(option => !customSearchOptions.includes(option));
    setSearchValues(newValues);

    if (value.includes(CustomSearchOptions.ShowAll)) {
      setLoadLimit(0);
    }

    if (onSelectionChange) {
      const labels = newValues
        .map(value => labelMap.get(value))
        .filter(label => !!label)
        .map(label => label!);
      onSelectionChange(labels);
    }
  }

  function updateLoadLimit(): number {
    const menu: HTMLElement | null | undefined = containerRef.current?.querySelector(".LabelList-Dropdown .menu");
    if (!menu) {
      return 0;
    }

    const height = menu.getBoundingClientRect().height;
    let limit = Math.trunc(height / itemHeight);
    if (data) {
      let unselectedLabels = data.labels.filter(label => !searchValues.includes(label.id.toString()));
      if (searchQuery) {
        unselectedLabels = unselectedLabels.filter(({ name }) => name.toLocaleUpperCase().includes(searchQuery.toLocaleUpperCase()));
      }
      if (unselectedLabels.length > limit) {
        const availHeight = height - itemHeight - dividerHeight;
        limit = Math.trunc(availHeight / itemHeight);
      }
    }

    if (limit > 0) {
      setLoadLimit(value => value > 0 ? limit : value);
    }
    return limit;
  }

  async function openLabelEditor(): Promise<void> {
    if (onBeforeLabelEditorOpen) {
      await onBeforeLabelEditorOpen();
    }
    setLabelEditorOpen(true);
  }

  function changeUpdating(updating: boolean): void {
    setUpdating(updating);
    onUpdating && onUpdating(updating);
  }

  function onWindowKeyDown(ev: KeyboardEvent): void {
    if (ev.key === "Backspace" && isEditRef.current) {
      ev.stopPropagation();
    }
  }

  return (
    <div className="LabelList">
      <WithQueryStatus loading={loading} error={error}>
        <div className="LabelList-Top">
          <Header as="h2" className="LabelList-Header">{__("Labels")}</Header>
          {!editable &&
          <Popup
            content={__("Edit Labels")}
            trigger={<Icon name="edit" size="large" onClick={() => openLabelEditor()}/>}
          />}
        </div>

        <div className="LabelList-Labels" ref={containerRef}>
          <Dropdown
            placeholder={__("Filter by labels...")}
            className="LabelList-Dropdown"
            fluid
            selection
            multiple
            open
            search={filterSearchOptions}
            options={searchOptions}
            value={searchValues}
            searchQuery={searchQuery}
            onChange={onSearchValueChange}
            onSearchChange={(e, { searchQuery }) => setSearchQuery(searchQuery)}
          />
        </div>

        {labelLoading && <Loading/>}
        {!onUpdating && updating && <Loading text={__("Updating...")}/>}

        <LabelEditorDialog open={labelEditorOpen} onClose={() => setLabelEditorOpen(false)} showBackButton={!editable}/>
      </WithQueryStatus>
    </div>
  );
};

export default LabelList;
