import React, { useState, useEffect, useMemo } from "react";
import { Input, Message } from "semantic-ui-react";
import { Label, useLabelsQuery } from "@generated/graphql";
import LabelList from "components/LabelsAndHierarchies/LabelList";
import CameraList from "components/LabelsAndHierarchies/CameraList";
import Loading from "components/Loading";
import {__} from "@solid/libs/i18n";
import { useStore } from "@core/store";

import "./style.css";

const LabelEditor = () => {
  const { data, error } = useLabelsQuery();
  const [searchName, setSearchName] = useState("");
  const [labels, setLabels] = useState<Label[]>([]);
  const [updating, setUpdating] = useState(false);
  const labelMap = useMemo(() => getLabelMap(), [data]);
  const { store: {session: {isAdmin}} } = useStore();

  useEffect(() => {
    error && console.error("Labels query error:", error);

    if (data) {
      setLabels(value => {
        const newValue = value
          .map(label => labelMap.get(label.id))
          .filter(label => !!label)
          .map(label => label!);
        return newValue;
      });
    }
  }, [data, error]);

  function getLabelMap(): Map<string, Label> {
    const map = new Map<string, Label>();
    if (!data) {
      return map;
    }
    for (const label of data.labels) {
      map.set(label.id, label);
    }
    return map;
  }

  return (
    <div className="LabelEditor">
      {isAdmin ?
        <>
          <div className="LabelEditor-Labels">
            <LabelList selectedLabels={labels} onSelectionChange={labels => setLabels(labels)} onUpdating={updating => setUpdating(updating)}/>
          </div>

          <div className="LabelEditor-Cameras">
            <div className="LabelEditor-CameraSearch">
              <Input icon="search" placeholder={__("Filter by name")} fluid value={searchName} onChange={e => setSearchName(e.currentTarget.value)}/>
            </div>
            <div className="LabelEditor-CameraList">
              <CameraList searchName={searchName} searchLabels={labels}/>
            </div>
          </div>

          {updating && <Loading text={__("Updating...")}/>}
        </> :
        <Message info className="Access-Denied">
          <Message.Header>Not enough rights</Message.Header>
        </Message>
    }

    </div>
  );
};

export default LabelEditor;
