import React, { useState, useEffect, useRef } from "react";
import { Dropdown, DropdownItemProps, DropdownProps, Divider, Icon, Popup, Input, Button, Ref, Modal, Header } from "semantic-ui-react";
import { useHierarchiesQuery, Hierarchy, HierarchyInput, HierarchyLevel } from "@generated/graphql";
import HierarchyTree from "components/LabelsAndHierarchies/HierarchyTree";
import WithQueryStatus from "components/WithQueryStatus";
import { useHierarchyActions } from "@core/actions";
import AddHierarchyDialog from "components/LabelsAndHierarchies/AddHierarchyDialog";
import { TreeNode, TreeViewEvent, TreeViewEventType } from "components/TreeView";
import { queryToInput } from "utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";

import "./style.css";

type HierarchyViewProps = {
  hierarchyId?: string;
  onSelectedLevelChange?: (hierarchy: Hierarchy | undefined, level: HierarchyLevel | undefined) => void;
  onUpdating?: (updating: boolean) => void;
  onBeforeNewChild?: (node?: TreeNode) => Promise<boolean>;
  onBeforeAddEdit?: (hierarchy?: Hierarchy) => Promise<void>;
  onBeforeDelete?: (hierarchy: Hierarchy) => void;
  onDialogOpen?: (open: boolean) => void;
};

enum CustomHierarchyOptions {
  AddHierarchy = "AddHierarchy",
  Divider = "Divider"
}

const customHierarchyOptions: string[] = [CustomHierarchyOptions.AddHierarchy, CustomHierarchyOptions.Divider];

type ViewListItem = {
  id: string;
  name: string;
  userId: string;
  userName: string;
};

const HierarchyView = ({ hierarchyId, onSelectedLevelChange, onUpdating, onBeforeNewChild, onBeforeAddEdit, onBeforeDelete, onDialogOpen }: HierarchyViewProps) => {
  const { data, loading, error } = useHierarchiesQuery();
  const [hierarchy, setHierarchy] = useState<Hierarchy | undefined>();
  const [hierarchyOptions, setHierarchyOptions] = useState<DropdownItemProps[]>([]);
  const [hierarchyValue, setHierarchyValue] = useState(hierarchyId ?? "");
  const [addDialogOpen, setAddDialogOpen] = useState(false);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [deleteDenyOpen, setDeleteDenyOpen] = useState(false);
  const [deleteErrorMessage, setDeleteErrorMessage] = useState("");
  const [hierarchyViews, setHierarchyViews] = useState<ViewListItem[]>([]);
  const [isHierarchyEdit, setIsHierarchyEdit] = useState(false);
  const [hierarchyName, setHierarchyName] = useState("");
  const [hierarchyNameOk, setHierarchyNameOk] = useState(false);
  const { insertHierarchy, updateHierarchy, deleteHierarchy } = useHierarchyActions();
  const treeViewEventRef = useRef(new TreeViewEvent());
  const inputRef = useRef<Input>(null);
  const okButtonRef = useRef<HTMLButtonElement>(null);

  useEffect(() => {
    error && console.error(error);
  }, [error]);

  useEffect(() => {
    if (!data) {
      return;
    }

    const hierarchies = [...data.hierarchies];
    const options: DropdownItemProps[] = hierarchies
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(hierarchy => getOption(hierarchy));

    if (options.length > 0) {
      options.push({ key: CustomHierarchyOptions.Divider, value: CustomHierarchyOptions.Divider, text: "", content: <Divider/>, disabled: true });
    }
    options.push({ key: CustomHierarchyOptions.AddHierarchy, value: CustomHierarchyOptions.AddHierarchy, text: __("Add hierarchy..."), className: "HierarchyView-AddHierarchy" });

    setHierarchyOptions(options);

    setHierarchyValue(value => {
      let newValue = "";
      if (data.hierarchies.some(({ id }) => id === value)) {
        newValue = value;
      }
      else if (data.hierarchies.length > 0 && options[0].value) {
        newValue = options[0].value.toString();
      }
      setHierarchy(data?.hierarchies.find(({ id }) => id === newValue));
      return newValue;
    });
  }, [data]);

  useEffect(() => {
    setHierarchy(data?.hierarchies.find(({ id }) => id === hierarchyValue));
  }, [hierarchyValue]);

  useEffect(() => {
    if (!hierarchy && onSelectedLevelChange) {
      onSelectedLevelChange(undefined, undefined);
    }
  }, [hierarchy]);

  useEffect(() => {
    if (!data || !hierarchy) {
      return;
    }
    setHierarchyNameOk(!!hierarchyName && !data.hierarchies.some(hier => hier.id !== hierarchy.id && hier.name.toLocaleUpperCase() === hierarchyName.toLocaleUpperCase()));
  }, [data, hierarchy, hierarchyName]);

  useEffect(() => {
    if (isHierarchyEdit) {
      window.requestAnimationFrame(() => {
        if (inputRef.current) {
          inputRef.current.focus();
        }
      });
    }
  }, [isHierarchyEdit]);

  useEffect(() => {
    if (onDialogOpen) {
      onDialogOpen(deleteConfirmOpen || deleteDenyOpen || addDialogOpen);
    }
  }, [deleteConfirmOpen, deleteDenyOpen, addDialogOpen]);

  function getOption(hierarchy: Hierarchy): DropdownItemProps {
    const { id, name } = hierarchy;
    return { key: id, value: id, text: name };
  }

  async function onHierarchyValueChange(e: React.SyntheticEvent, { value }: DropdownProps): Promise<void> {
    if (typeof value !== "string") {
      return;
    }

    if (!customHierarchyOptions.includes(value)) {
      setHierarchyValue(value);
    }

    if (value === CustomHierarchyOptions.AddHierarchy) {
      const keyEvent = e.nativeEvent instanceof KeyboardEvent ? e.nativeEvent : undefined;
      if (keyEvent && (keyEvent.code === "ArrowDown" || keyEvent.code === "ArrowUp")) {
        return;
      }
      if (onBeforeAddEdit) {
        await onBeforeAddEdit();
      }
      setAddDialogOpen(true);
    }
  }

  async function addHierarchy(name: string): Promise<void> {
    const id = await insertHierarchy({ id: "", name });
    setHierarchyValue(id);
  }

  async function onHierarchyTreeChange(newHierarchy: Hierarchy): Promise<boolean> {
    onUpdating && onUpdating(true);
    try {
      const input = queryToInput<Hierarchy, HierarchyInput>(newHierarchy);
      input.name = undefined;
      await updateHierarchy(input);
      return false; // Always return false to avoid new nodes assignment in TreeView component.
    }
    catch (e: any) {
      console.error("Hierarchy update error:", e);
      Log.error(__("Hierarchy update error: {{message}}", {message: e.message}));
      return false;
    }
    finally {
      onUpdating && onUpdating(false);
    }
  }

  async function onEditHierarchy(): Promise<void> {
    if (hierarchy) {
      if (onBeforeAddEdit) {
        await onBeforeAddEdit(hierarchy);
      }
      setHierarchyName(hierarchy.name);
      setIsHierarchyEdit(true);
    }
  }

  async function onDeleteHierarchy(): Promise<void> {
    if (!hierarchy) {
      return;
    }
    onBeforeDelete && onBeforeDelete(hierarchy);
    onUpdating && onUpdating(true);
    try {
      await deleteHierarchy(hierarchy.id);
    }
    catch (e: any) {
      console.error("Hierarchy delete error:", e);
      setDeleteErrorMessage(e.message);
      let views: ViewListItem[] = [];
      if (e.extensions?.info?.viewList?.views) {
        views = views.concat(e.extensions?.info?.viewList?.views);
      }
      if (e.extensions?.info?.viewList?.sharedViews) {
        views = views.concat(e.extensions?.info?.viewList?.sharedViews);
      }
      views.sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
      setHierarchyViews(views);
      setDeleteDenyOpen(true);
    }
    finally {
      onUpdating && onUpdating(false);
    }
  }

  async function onApplyHierarchyEdit(): Promise<void> {
    setIsHierarchyEdit(false);
    if (!hierarchy || !hierarchyNameOk) {
      return;
    }
    onUpdating && onUpdating(true);
    try {
      await updateHierarchy({ id: hierarchy.id, name: hierarchyName });
    }
    catch (e: any) {
      console.error("Hierarchy update error:", e);
      Log.error(__("Hierarchy update error: {{message}}", {message: e.message}));
    }
    finally {
      onUpdating && onUpdating(false);
    }
  }

  function onCancelHierarchyEdit(): void {
    setIsHierarchyEdit(false);
  }

  function onInputKeyDown(e: React.KeyboardEvent): void {
    if (e.key === "Enter" && hierarchyNameOk) {
      onApplyHierarchyEdit();
    }
    if (e.key === "Escape") {
      onCancelHierarchyEdit();
    }
  }

  return (
    <div className="HierarchyView">
      <WithQueryStatus loading={loading} error={error}>
        {!!data &&
          <>
            <div className="HierarchyView-Select">
              <Dropdown
                className="HierarchyView-Dropdown"
                fluid
                selection
                options={hierarchyOptions}
                value={hierarchyValue}
                onChange={onHierarchyValueChange}
              />
            </div>

            {!!hierarchy &&
            <div className="HierarchyView-TopButtons">
              {isHierarchyEdit ?
                <Input
                  ref={inputRef}
                  placeholder={__("Name")}
                  fluid
                  action
                  value={hierarchyName}
                  onChange={e => setHierarchyName(e.currentTarget.value)}
                  onBlur={(e: React.FocusEvent) => {
                    if (e.nativeEvent.relatedTarget !== okButtonRef.current) {
                      onCancelHierarchyEdit();
                    }
                  }}
                  onKeyDown={onInputKeyDown}>
                  <input/>
                  <Ref innerRef={okButtonRef}>
                    <Button positive icon disabled={!hierarchyNameOk} onClick={() => onApplyHierarchyEdit()}>
                      <Icon name="check"/>
                    </Button>
                  </Ref>
                  <Button negative icon onClick={() => onCancelHierarchyEdit()}>
                    <Icon name="cancel"/>
                  </Button>
                </Input> :
                <>
                  <Popup content={__("Add Branch")}
                    trigger={<Icon name="plus" onClick={() => treeViewEventRef.current.publish({ type: TreeViewEventType.AddRootItemCmd })}/>}
                  />
                  <Popup content={__("Edit Hierarchy")}
                    trigger={<Icon name="edit" onClick={onEditHierarchy}/>}
                  />
                  <Popup content={__("Delete Hierarchy")}
                    trigger={<Icon name="trash alternate" onClick={() => setDeleteConfirmOpen(true)}/>}
                  />
                </>}
            </div>}

            <div className="HierarchyView-Hierarchy">
              {!!hierarchy &&
              <HierarchyTree
                hierarchy={hierarchy}
                treeViewEvent={treeViewEventRef.current}
                onSelectedLevelChange={onSelectedLevelChange}
                onHierarchyChange={onHierarchyTreeChange}
                onBeforeNewChild={onBeforeNewChild}
              />}
            </div>

            <AddHierarchyDialog
              open={addDialogOpen}
              onClose={() => setAddDialogOpen(false)}
              onOk={addHierarchy}
              hierarchies={data.hierarchies}
            />

            {deleteConfirmOpen &&
              <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
                <Header>{__("Delete Hierarchy")}</Header>
                <Modal.Content>{__("Are you sure you want to delete hierarchy '{{name}}'?", {name: hierarchy?.name})}</Modal.Content>
                <Modal.Actions>
                  <Button
                    negative
                    onClick={() => {
                      onDeleteHierarchy();
                      setDeleteConfirmOpen(false);
                    }}>
                    <Icon name="trash"/>{__("Delete")}
                  </Button>
                  <Button onClick={() => setDeleteConfirmOpen(false)}>
                    <Icon name="cancel"/>{__("Cancel")}
                  </Button>
                </Modal.Actions>
              </Modal>}

            {deleteDenyOpen &&
              <Modal open={deleteDenyOpen} onClose={() => setDeleteDenyOpen(false)}>
                <Header>{__("Delete Hierarchy")}</Header>
                <Modal.Content>
                  <div>{__("Hierarchy '{{name}}' could not be deleted.", {name: hierarchy?.name})}</div>
                  {hierarchyViews.length === 0 && <div>{deleteErrorMessage}</div>}
                  {hierarchyViews.length > 0 &&
                  <>
                    <div>{__("Hierarchy '{{name}}' is used in the following views:", {name: hierarchy?.name})}</div>
                    <ul>
                      {hierarchyViews
                        .map(({ name, userName }) => <li>{name} ({userName})</li>)}
                    </ul>
                  </>}
                </Modal.Content>
                <Modal.Actions>
                  <Button positive onClick={() => setDeleteDenyOpen(false)}>
                    <Icon name="check"/>{__("OK")}
                  </Button>
                </Modal.Actions>
              </Modal>}
          </>}
      </WithQueryStatus>
    </div>
  );
};

export default HierarchyView;
