import React, { useState, useEffect, useRef, useMemo } from "react";
import { Header, Button, Icon, Popup, Modal, Step, StepProps, Message } from "semantic-ui-react";
import { Label, useLabelsQuery, Hierarchy, HierarchyLevel, HierarchyInput } from "@generated/graphql";
import { useHierarchyActions } from "@core/actions";
import HierarchyView from "components/LabelsAndHierarchies/HierarchyView";
import LabelList from "components/LabelsAndHierarchies/LabelList";
import CameraList from "components/LabelsAndHierarchies/CameraList";
import Loading from "components/Loading";
import WithQueryStatus from "components/WithQueryStatus";
import { TreeNode } from "components/TreeView";
import { queryToInput, EventPubSub } from "utils";
import { clone } from "@solid/libs/utils";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import { useStore } from "@core/store";

import "./style.css";

export type HierarchyEditorEventArgs = {
  close: () => void;
};

export class HierarchyEditorEvent extends EventPubSub<HierarchyEditorEventArgs> {}

type HierarchyEditorProps = {
  hierarchyId?: string;
  closeEvent?: HierarchyEditorEvent;
};

const HierarchyEditor = ({ hierarchyId, closeEvent }: HierarchyEditorProps) => {
  const { data, error, loading } = useLabelsQuery();
  const { updateHierarchy } = useHierarchyActions();
  const [hierarchy, setHierarchy] = useState<Hierarchy | undefined>();
  const [level, setLevel] = useState<HierarchyLevel | undefined>();
  const [labels, setLabels] = useState<Label[]>([]);
  const [levelLabels, setLevelLabels] = useState<Label[]>([]);
  const [newLabels, setNewLabels] = useState<Label[] | undefined>();
  const labelMap = useMemo(() => getLabelMap(), [data]);
  const [breadcrumbs, setBreadcrumbs] = useState<StepProps[]>([]);
  const [updating, setUpdating] = useState(false);
  const [applyConfirmOpen, setApplyConfirmOpen] = useState(false);
  const [dialogOpen, setDialogOpen] = useState(false);
  const onApplyConfirmRef = useRef<(() => void) | undefined>();
  const onApplyCancelRef = useRef<(() => void) | undefined>();
  const clearNewLabelsFlagRef = useRef(false);
  const { store: {session: {isAdmin}} } = useStore();

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    closeEvent?.subscribe(id, args => {
      if (newLabels && hierarchy && level) {
        onApplyConfirmRef.current = async () => {
          await onApplyLabels(hierarchy, level, newLabels);
          args.close();
        };
        onApplyCancelRef.current = () => args.close();
        setApplyConfirmOpen(true);
      }
      else {
        args.close();
      }
    });

    return () => {
      closeEvent?.unsubscribe(id);
    };
  });

  useEffect(() => {
    error && console.error("Labels query error:", error);
  }, [error]);

  useEffect(() => {
    if (clearNewLabelsFlagRef.current) {
      clearNewLabelsFlagRef.current = false;
      setNewLabels(undefined); // Clear new assigned labels only after refresh to avoid undesired visual effects.
    }

    if (!level || !data) {
      setLabels([]);
      setLevelLabels([]);
      return;
    }

    const levelLevels = level.levels ?? [];
    if (levelLevels.length > 0) {
      setLabels([]);
      const levelLabels: Label[] = [];
      getLevelLabels(levelLabels, levelLevels);
      setLevelLabels(levelLabels);
      return;
    }

    const labels = level.labelIds.map(id => labelMap.get(id)).filter(label => !!label).map(label => label!);

    setLabels(labels.sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" })));
    setLevelLabels(labels);
  }, [level, data]);

  function getLabelMap(): Map<string, Label> {
    const map = new Map<string, Label>();
    if (!data) {
      return map;
    }
    for (const label of data.labels) {
      map.set(label.id, label);
    }
    return map;
  }

  function onSelectedLevelChange(newHierarchy: Hierarchy | undefined, newLevel: HierarchyLevel | undefined): void {
    if (!clearNewLabelsFlagRef.current && newLabels && hierarchy && level && level.id !== newLevel?.id) {
      onApplyConfirmRef.current = () => onApplyLabels(hierarchy, level, newLabels);
      onApplyCancelRef.current = undefined;
      setApplyConfirmOpen(true);
    }

    setHierarchy(newHierarchy);
    setLevel(newLevel);
    setNewLabels(undefined);

    const path: HierarchyLevel[] = [];
    if (newHierarchy && newLevel) {
      const levelLevels = newHierarchy.level.levels ?? [];
      getLevelPath(path, newLevel.id, levelLevels);
    }
    setBreadcrumbs(path.map(({ id, name }) => ({ key: id, content: name })));
  }

  function onSelectedLabelsChange(selectedLabels: Label[]): void {
    let newValue: Label[] | undefined;
    if (JSON.stringify(selectedLabels.map(label => label.id).sort()) !== JSON.stringify(labels.map(label => label.id).sort())) {
      newValue = selectedLabels;
    }
    setNewLabels(value => {
      if (!value && !newValue) {
        return value;
      }
      if (!value || !newValue) {
        return newValue;
      }
      if (JSON.stringify(value.map(label => label.id).sort()) !== JSON.stringify(newValue.map(label => label.id).sort())) {
        return newValue;
      }
      return value;
    });
  }

  async function onApplyLabels(hierarchy?: Hierarchy, level?: HierarchyLevel, newLabels?: Label[]): Promise<boolean> {
    if (!hierarchy || !level || !newLabels) {
      return false;
    }

    setUpdating(true);
    clearNewLabelsFlagRef.current = true; // Set flag to clear new labels after refresh only to avoid undesired visual effects.
    try {
      const newHierarchy: Hierarchy = clone(hierarchy);
      const levelLevels = newHierarchy.level.levels ?? [];
      updateLabels(level, newLabels, levelLevels);
      const input = queryToInput<Hierarchy, HierarchyInput>(newHierarchy);
      input.name = undefined;
      await updateHierarchy(input);
      return true;
    }
    catch (e: any) {
      clearNewLabelsFlagRef.current = false;
      console.error("Hierarchy update error:", e);
      Log.error(__("Hierarchy update error: {{message}}", {message: e.message}));
      return false;
    }
    finally {
      setUpdating(false);
    }
  }

  function onRevertLabels(): void {
    setNewLabels(undefined);
  }

  function updateLabels(level: HierarchyLevel, labels: Label[], levels: HierarchyLevel[]): boolean {
    const index = levels.findIndex(lev => lev.id === level.id);
    if (index >= 0) {
      levels[index] = { ...levels[index], labelIds: labels.map(label => label.id) };
      return true;
    }
    for (const child of levels) {
      const childLevels = child.levels ?? [];
      if (updateLabels(level, labels, childLevels)) {
        return true;
      }
    }
    return false;
  }

  function getLevelLabels(labels: Label[], levels: HierarchyLevel[]): void {
    if (!data) {
      return;
    }
    const labelIdSet = new Set<string>(labels.map(label => label.id));
    for (const level of levels) {
      const levelLevels = level.levels ?? [];
      if (levelLevels.length === 0) {
        for (const id of level.labelIds.filter(id => !labelIdSet.has(id))) {
          const label = labelMap.get(id);
          if (label) {
            labels.push(label);
            labelIdSet.add(label.id);
          }
        }
      }
      getLevelLabels(labels, levelLevels);
    }
  }

  // function getLevelPath(path: HierarchyFromQuery[], levelId: string, levels: HierarchyFromQueryLevels[]): boolean {
  function getLevelPath(path: HierarchyLevel[], levelId: string, levels: HierarchyLevel[]): boolean {
    for (const level of levels) {
      if (level.id === levelId) {
        path.push(level);
        return true;
      }
      const levelLevels = level.levels ?? [];
      if (getLevelPath(path, levelId, levelLevels)) {
        path.unshift(level);
        return true;
      }
    }
    return false;
  }

  function applyOrCancelLabels(): Promise<boolean> {
    return new Promise<boolean>(resolve => {
      if (!newLabels) {
        resolve(false);
        return;
      }
      onApplyConfirmRef.current = async () => {
        const updated = await onApplyLabels(hierarchy, level, newLabels);
        resolve(updated);
      };
      onApplyCancelRef.current = () => {
        setNewLabels(undefined);
        resolve(false);
      };
      setApplyConfirmOpen(true);
    });
  }

  function onBeforeNewChild(node?: TreeNode): Promise<boolean> {
    return applyOrCancelLabels();
  }

  async function onBeforeLabelEditorOpen(): Promise<void> {
    await applyOrCancelLabels();
  }

  async function onBeforeAddEdit(hierarchy?: Hierarchy): Promise<void> {
    await applyOrCancelLabels();
  }

  function onBeforeDelete(hierarchy: Hierarchy): void {
    setNewLabels(undefined);
  }

  return (
    <div className="HierarchyEditor">
      {isAdmin ?
        <WithQueryStatus loading={loading} error={error}>
          {updating && <Loading text={__("Updating...")}/>}

          {data &&
          <>
            <div className="HierarchyEditor-Left">
              <Header as="h2" className="HierarchyEditor-Header">{__("Hierarchy")}</Header>

              <div className="HierarchyEditor-Hierarchy">
                <HierarchyView
                  hierarchyId={hierarchyId}
                  onSelectedLevelChange={onSelectedLevelChange}
                  onUpdating={updating => setUpdating(updating)}
                  onBeforeNewChild={onBeforeNewChild}
                  onBeforeAddEdit={onBeforeAddEdit}
                  onBeforeDelete={onBeforeDelete}
                  onDialogOpen={open => setDialogOpen(open)}
                    />
              </div>

              {!!level && (!level.levels || level.levels.length === 0) &&
              <>
                <div className="HierarchyEditor-Labels">
                  <Popup
                    trigger={<div/>}
                    open={(newLabels ?? labels).length === 0 && !updating && !applyConfirmOpen && !dialogOpen}
                    content={__("Assign labels to the selected hierarchy level")}
                    position="right center"
                      />
                  <LabelList
                    selectedLabels={newLabels ?? labels}
                    onSelectionChange={onSelectedLabelsChange}
                    editable={false}
                    onBeforeLabelEditorOpen={onBeforeLabelEditorOpen}
                    onLabelEditorOpen={open => setDialogOpen(open)}
                      />
                </div>

                {!!newLabels &&
                <div className="HierarchyEditor-LabelsButtons">
                  <Button positive onClick={() => onApplyLabels(hierarchy, level, newLabels)}><Icon name="checkmark"/>{__("Save")}</Button>
                  <Button negative onClick={onRevertLabels}><Icon name="undo"/>{__("Discard")}</Button>
                </div>}
              </>}
            </div>

            <div className="HierarchyEditor-Right">
              <div className="HierarchyEditor-Path">
                <Step.Group items={breadcrumbs}/>
              </div>

              <CameraList searchLabels={newLabels ?? levelLabels} editable={false} mandatoryLabelSearch/>
            </div>

            {applyConfirmOpen &&
            <Modal open={applyConfirmOpen} onClose={() => {
              onApplyCancelRef.current && onApplyCancelRef.current();
              setApplyConfirmOpen(false);
            }}>
              <Header>{__("Unsaved Changes")}</Header>
              <Modal.Content>{__("Assigned labels were changed. Would you like to save the changes?")}</Modal.Content>
              <Modal.Actions>
                <Button
                  positive
                  onClick={() => {
                    onApplyConfirmRef.current && onApplyConfirmRef.current();
                    setApplyConfirmOpen(false);
                  }}>
                  <Icon name="check"/>{__("Save")}
                </Button>
                <Button
                  negative
                  onClick={() => {
                    onApplyCancelRef.current && onApplyCancelRef.current();
                    setApplyConfirmOpen(false);
                  }}>
                  <Icon name="undo"/>{__("Discard")}
                </Button>
              </Modal.Actions>
            </Modal>}
          </>}
        </WithQueryStatus> :
        <Message info className="Access-Denied">
          <Message.Header>Not enough rights</Message.Header>
        </Message>
    }

    </div>
  );
};

export default HierarchyEditor;
