import React, { useState } from "react";
import { Button, Icon, Modal, Header, Form, Message } from "semantic-ui-react";
import { Hierarchy } from "@generated/graphql";
import Loading from "components/Loading";
import {__} from "@solid/libs/i18n";

import "./style.css";

type AddHierarchyDialogProps = {
  open: boolean;
  onOk: (name: string) => Promise<void>;
  onClose: () => void;
  hierarchies: Hierarchy[];
};

const AddHierarchyDialog = ({ open, onOk, onClose, hierarchies }: AddHierarchyDialogProps) => {
  const [name, setName] = useState("");
  const [nameError, setNameError] = useState("");
  const [updating, setUpdating] = useState(false);
  const [updateError, setUpdateError] = useState<Error | undefined>();

  function onNameChange(name: string): void {
    setName(name);
    if (!name) {
      setNameError(__("Name should not be empty"));
      return;
    }
    if (hierarchies.some(hier => hier.name.toLocaleUpperCase() === name.toLocaleUpperCase())) {
      setNameError(__("Hierarchy name already exists"));
      return;
    }
    setNameError("");
  }

  async function onOkClick(): Promise<void> {
    if (!name || nameError) {
      return;
    }
    setUpdating(true);
    setUpdateError(undefined);
    try {
      await onOk(name);
      onClose();
    }
    catch (e: any) {
      setUpdateError(e);
    }
    finally {
      setUpdating(false);
    }
  }

  function onInputKeyDown(e: React.KeyboardEvent): void {
    if (e.key === "Enter" && name && !nameError && !updating) {
      onOkClick();
    }
  }

  return (
    <Modal className="AddHierarchyDialog" open={open} onClose={onClose} onMount={() => setName("")}>
      <Header content={__("Add Hierarchy")}/>

      <Modal.Content>
        {!!updateError && <Message visible error header={__("Hierarchy update error")} content={updateError.message}/>}

        <Form onSubmit={(e) => { e.preventDefault(); }}>
          <Form.Field
            control={Form.Input}
            value={name}
            label={__("Name")}
            placeholder={__("Name")}
            autoFocus
            onInput={(e: React.FormEvent<HTMLInputElement>) => onNameChange(e.currentTarget.value)}
            disabled={updating}
            error={nameError ? { content: nameError, pointing: "below" } : undefined}
            onKeyDown={onInputKeyDown}
          />
        </Form>

        {updating && <Loading text={__("Updating...")}/>}
      </Modal.Content>

      <Modal.Actions>
        <Button positive onClick={onOkClick} disabled={!name || !!nameError || updating}><Icon name="plus"/>{__("Add")}</Button>
        <Button negative onClick={onClose}><Icon name="cancel"/>{__("Cancel")}</Button>
      </Modal.Actions>
    </Modal>
  );
};

export default AddHierarchyDialog;
