import React, { useState, useEffect } from "react";
import { Segment } from "semantic-ui-react";
import { ImageMapList, ImageMapEditor } from "@libs/imagemap";
import * as imt from "@libs/imagemap";
import { Camera, Type, Event, UUID } from "@solid/types";
import {
  useImageMapConfigQuery,
  DeviceFunctionalAspectType,
  AuditEntry,
  useDeviceListByAspectTypesQuery
} from "@generated/graphql";
import { useImageMapActions, imageMapVersion as version, lazyLoadImages, isError, useAuditEntries } from "@core/actions";
import { WidgetProps, CommonWidgetEvent, EventClickEventArgs } from "components/Widgets";
import WithQueryStatus from "components/WithQueryStatus";
import PlayerCell from "components/PlayerCellExternal";
import { withStore, WithStoreProps } from "@core/store";
import { clone } from "@solid/libs/utils";

import "./style.css";

type ImageMapProps = WidgetProps & WithStoreProps & {
  editMode?: boolean;
  maxEventCount?: string;
  showEventsInInvisibleLayers?: boolean;
  mapSetId?: string;
  showMapSetList?: boolean;
};

const defaultMaxDeviceEvents = 10;
const eventCategories = ["47"];

type EventWithEntry = Event & {
  entry: AuditEntry;
};

function isEventWithEntry(event: Event): event is EventWithEntry {
  return !!(event as EventWithEntry).entry;
}

type DeviceEventsWithEntry = {
  [id in UUID]: {
    isChecked: boolean;
    events: EventWithEntry[];
  }
};

const ImageMap = ({
  editMode,
  maxEventCount = defaultMaxDeviceEvents.toString(),
  showEventsInInvisibleLayers,
  mapSetId = "",
  showMapSetList = true,
  widgetEvent,
  setStore,
  index: widgetIndex = 0,
  widgetId,
  viewId
}: ImageMapProps) =>
{
  const { data: mapData, error: mapError, loading: mapLoading } = useImageMapConfigQuery({ variables: { version, lazyLoadImages } });
  const { data: devData, error: devError, loading: devLoading } = useDeviceListByAspectTypesQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
  const { saveImageMapConfig, updateImageMapCache, getImageMapImage } = useImageMapActions();
  const [config, setConfig] = useState<imt.ImageMapConfig>({ set: {} });
  const [cameras, setCameras] = useState<Camera[]>([]);
  const [camerasLoaded, setCamerasLoaded] = useState(false);
  const [editMapId, setEditMapId] = useState("");
  const { subscribe, unsubscribe, subscription: { data: subData, error: subError } } = useAuditEntries();
  const [deviceEvents, setDeviceEvents] = useState<DeviceEventsWithEntry>({});
  const maxDeviceEvents = parseInt(maxEventCount);
  const subscriberId = `${viewId}_widget${widgetIndex}`;

  useEffect(() => {
    subscribe(subscriberId);

    return () => {
      unsubscribe(subscriberId);
    };
  }, []);

  useEffect(() => {
    setEditMapId(mapSetId);
  }, [mapSetId]);

  useEffect(() => {
    if (mapData) {
      const config = { set: {} };
      for (const set of mapData.imageMapConfig.sets) {
        const { id: setId, name, layers } = set;
        config.set[setId] = { name, layer: {} };
        for (const layer of layers) {
          const { id: layerId, name, maps } = layer;
          config.set[setId].layer[layerId] = { name, map: {} };
          for (const map of maps) {
            const { id: mapId, name, order, base64, objects } = map;
            config.set[setId].layer[layerId].map[mapId] = { name, order, base64, object: {} };
            for (const object of objects) {
              const { id, objectId, position } = object;
              config.set[setId].layer[layerId].map[mapId].object[id] = { obj: objectId, position };
            }
          }
        }
      }
      setConfig(config);
    }
  }, [mapData]);

  useEffect(() => {
    if (devData) {
      setCameras(devData.devicesByAspectTypes.map(({ id, name }) => ({
        obj: id,
        name,
        type: Type.camera,
        attributes: {}
      })));

      setCamerasLoaded(true);
    }
  }, [devData]);

  useEffect(() => {
    subError && console.error("Image map audit subscription error:", subError);

    if (!subData) {
      return;
    }

    const entry = subData.auditEntries;
    if (isError(entry)) {
      console.error("Image map audit subscription error:", entry.error);
      return;
    }

    const { context, triggeredAt, modifiedAt, message, witnesses, category } = entry;

    const event: EventWithEntry = {
      eventId: context,
      from: new Date(triggeredAt).getTime(),
      to: new Date(triggeredAt).getTime(),
      when: new Date(triggeredAt).getTime(),
      lastModified: new Date(modifiedAt).getTime(),
      message,
      objects: witnesses.map(({ id, name }) => ({ objId: id, name })),
      category: parseInt(category.id),
      entry: { ...entry, metadata: [] }
    };

    setDeviceEvents(value => {
      const deviceIds = witnesses.filter(w => cameras.some(cam => cam.obj === w.id)).map(w => w.id);
      if (deviceIds.length === 0) {
        return value;
      }

      const newValue: DeviceEventsWithEntry = clone(value);

      for (const id of deviceIds) {
        let devEvents = newValue[id];
        if (!devEvents) {
          devEvents = { isChecked: false, events: [] };
          newValue[id] = devEvents;
        }
        else {
          devEvents.isChecked = false;
        }

        const index = devEvents.events.findIndex(ev => ev.eventId === event.eventId);
        if (index < 0) {
          devEvents.events.push(event);
        }
        else {
          devEvents.events[index] = event;
        }

        devEvents.events.sort((a, b) => b.when - a.when);

        if (devEvents.events.length > maxDeviceEvents) {
          devEvents.events = devEvents.events.slice(0, maxDeviceEvents);
        }
      }
      return newValue;
    });
  }, [subData, subError]);

  const activateEvent = (events: Event[], obj: UUID): void => {
    if (events.length === 0 || !obj) {
      return;
    }

    const args: EventClickEventArgs = {
      widgetIndex,
      deviceId: obj,
      deviceName: cameras.find(cam => cam.obj === obj)?.name ?? "",
      categories: eventCategories
    };
    widgetEvent?.publish({ event: CommonWidgetEvent.EventClick, args });

    const event = events[0];
    const entry = isEventWithEntry(event) ? event.entry : undefined;
    if (entry) {
      setStore({ workspace: { event: { entry, hasSnapshots: true } } });
    }

    setDeviceEvents(value => {
      const newValue: DeviceEventsWithEntry = clone(value);
      const devEvents = newValue[obj];
      if (!devEvents) {
        return value;
      }

      const eventIdSet = new Set<string>(events.map(ev => ev.eventId));
      devEvents.events = devEvents.events.filter(ev => !eventIdSet.has(ev.eventId));
      devEvents.isChecked = devEvents.events.length === 0;
      return newValue;
    });
  };

  const onConfigChange = (config: imt.ImageMapConfig): void => {
    // setConfig(config);
    updateImageMapCache(config);
  };

  return (
    <Segment className="ImageMap">
      <WithQueryStatus loading={mapLoading || devLoading || (!camerasLoaded && !devError)} error={mapError || devError}>
        { !editMapId && editMode && showMapSetList ?
          <ImageMapList imageMapConfig={config} cameras={cameras} saveImageMapConfig={saveImageMapConfig}
            onEditMap={id => setEditMapId(id)} onConfigChange={onConfigChange}/> :
          <ImageMapEditor mapSetId={editMapId} imageMapConfig={config} cameras={cameras} saveImageMapConfig={saveImageMapConfig}
            deviceEvents={deviceEvents} playerCell={PlayerCell} isView={!editMode} activateEvent={activateEvent}
            onBackToList={() => setEditMapId("")} onConfigChange={onConfigChange} getImageMapImage={getImageMapImage}
            hideableControls={!editMode} showEventsInInvisibleLayers={showEventsInInvisibleLayers} hideMapSetList={!showMapSetList}
            widgetId={widgetId}/>
        }
      </WithQueryStatus>
    </Segment>
  );
};

export default withStore(ImageMap);
