import React, { useEffect, useState } from "react";
import { Button, Header, List, Message, Modal, Segment } from "semantic-ui-react";
import { ApolloError } from "@apollo/client";
import { __, isElectron } from "@solid/libs";
import DownloadInstaller from "../DownloadInstaller";
import WithQueryStatus from "components/WithQueryStatus";
import { useStore } from "@core/store";
import useLogout from "@core/useLogout";

import "./style.css";

type GuiListProps = {
  guiList: {id: string, name: string, url: string, description: string}[] | undefined;
  loading: boolean;
  error: ApolloError | undefined;
  setGui: (url: string, id?: string, isHash?: boolean) => void
};

const GuiList = ({ guiList, loading, error, setGui }: GuiListProps) => {
  const { store: { session: { info } } } = useStore();
  const [open, setOpen] = useState(false);
  const { logout } = useLogout();

  useEffect(() => {
    if (!isElectron()) {
      return;
    }
    if (!loading && !error && guiList) {
      const isSolid = guiList.some((gui) => gui.id === "SOLID");
      if (!isSolid) {
        logout();
      }
    }
  }, [guiList, loading, error]);

  return (
    <div className="gui-wrapper">
      { !isElectron() ?
        <WithQueryStatus loading={loading} error={error} noData={!guiList}>
          { !!info &&
          <Segment className="guis-list">
            <Header className="GiuList-Header" size="huge" textAlign="center">{__("Select UI to Enter")}</Header>
            { !!guiList && guiList.length < 1 &&
            <Message negative>
              <Message.Header>{__("Access is denied")}</Message.Header>
              <p>{__("Gui list not found")}</p>
            </Message>
            }
            { !!guiList && guiList.length >= 1 &&
            <div className="guis">
              {guiList.map((gui) => {
                return (
                  <div className="current-gui" key={gui.id} onClick={() => setGui(gui.url, gui.id, false)}>
                    <List.Item className="left-item">
                      <List.Icon name="th" size="huge" verticalAlign="middle" className="left-icon"/>
                      <List.Content>
                        <Header data-guiid={gui.id} size="medium">{gui.name}</Header>
                        <Header.Subheader>{gui.description}</Header.Subheader>
                      </List.Content>
                    </List.Item>
                    <List.Item className="right-item" onClick={(e) => e.stopPropagation()} >
                      {gui.id === "SOLID" &&
                      <Modal className="DownloadInstaller-Modal"
                        closeIcon
                        open={open}
                        trigger={<Button primary className="install-app">{__("Install App")}</Button>}
                        onClose={() => setOpen(false)}
                        onOpen={() => setOpen(true)}
                      >
                        <DownloadInstaller appId={gui.id} />
                      </Modal>
                      }
                    </List.Item>
                  </div>
                );
              })}
            </div>}
          </Segment>
          }
        </WithQueryStatus>
        :
        <Segment className="guis-list">
          <Header className="GiuList-Header" size="huge" textAlign="center">{__("Select UI to Enter")}</Header>
          <div className="guis">
            <div className="current-gui" onClick={() => setGui("/main", "SOLID", true)}>
              <List.Item className="left-item">
                <List.Icon name="th" size="huge" verticalAlign="middle" className="left-icon"/>
                <List.Content>
                  <Header size="medium">{__("Solid")}</Header>
                  <Header.Subheader>{__("Solid")}</Header.Subheader>
                </List.Content>
              </List.Item>
            </div>
          </div>
        </Segment>}
    </div>
  );
};

export default GuiList;
