import React, {RefObject, useEffect, useRef} from "react";
import {useMap, useMapEvent} from "react-leaflet";
import {Utils} from "@solid/libs/utils";
import {LatLngTuple} from "leaflet";
import {GEOMapProperties} from "./GEOMapCellStub/GEOMapCellStub";

type ResizeComponentProps = {
  mapRef: RefObject<HTMLDivElement>,
  latLngList: LatLngTuple[],
  defaultPosition: boolean,
  defaultPositionProperties?: GEOMapProperties,
  setDefaultPositionToStore?: (position: GEOMapProperties) => void
};

const ResizeComponent = ({mapRef, latLngList, defaultPosition, setDefaultPositionToStore, defaultPositionProperties}: ResizeComponentProps) => {
  const resizeObserverRef = useRef<ResizeObserver>();
  const map = useMap();
  useMapEvent("dragend", (e) => {
    if (defaultPosition && setDefaultPositionToStore) {
      const position =  e.target.getCenter();
      const zoom = e.target.getZoom();

      setDefaultPositionToStore({position, zoom});
    }
  });

  useMapEvent("zoom", (e) => {
    if (defaultPosition && setDefaultPositionToStore) {
      const position =  e.target.getCenter();
      const zoom = e.target.getZoom();

      setDefaultPositionToStore({position, zoom});
    }
  });

  useEffect(() => {
    const root = mapRef.current;
    if (root) {
      resizeObserverRef.current = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
        for (const entry of entries) {
          if (entry.target === root) {
            map.invalidateSize(true);

            if (latLngList.length === 0) {
              // do nothing
            } else
            if (latLngList.length === 1) {
              map.setView(latLngList[0], defaultPosition ? defaultPositionProperties?.zoom || 15 : 15);
            } else {
              map.fitBounds(latLngList, {animate: true, maxZoom: 15});
            }

            break;
          }
        }
      }));
      resizeObserverRef.current.observe(root);
    }

    return function cleanup() {
      root && resizeObserverRef.current?.unobserve(root);
    };
  }, [mapRef, latLngList]);

  useEffect(() => {
    if (latLngList.length === 0) {
      return;
    }

    if (latLngList.length === 1) {
      map.setView(latLngList[0], defaultPosition ? defaultPositionProperties?.zoom || 15 : 15);
    } else {
      map.fitBounds(latLngList, {animate: true, maxZoom: 15});
    }
  }, [latLngList]);

  return <></>;
};

export default ResizeComponent;
