import React, {useRef, useMemo, useState, useEffect} from "react";
import {LatLng} from "generated/graphql";
import {Marker, Tooltip, Popup, Polyline} from "react-leaflet";
import L, { LatLngTuple } from "leaflet";
import { Device } from "@core/actions";
import PlayerCellLegacy from "@solid/player-cell/components/PlayerCellLegacy";
import { __ } from "electron/i18n";
import { Event } from "@solid/types";
import classNames from "classnames";

import "leaflet/dist/leaflet.css";
import "./style.css";

type DraggableMarkerProps = {
  defaultPosition?: LatLng,
  device?: Device,
  mode: "edit" | "view",
  event?: Event,
  tracking?: LatLngTuple[],
  updatePosition?: (pos: LatLng) => void
  onCalloutClick?: (event?: Event) => void,
};

const eventBlockClassName = "leaflet-default-icon-event-label";

const DraggableMarker = ({ device, defaultPosition, mode, updatePosition, event, onCalloutClick, tracking }: DraggableMarkerProps) => {
  const [position, setPosition] = useState<LatLng | undefined | null>(device?.position || defaultPosition);
  const markerRef = useRef<L.Marker>(null);
  const popupRef = useRef<L.Popup>(null);
  const [currentEvent, setCurrentEvent] = useState<Event | undefined>(event);
  const [isShowPlayer, showPlayer] = useState(false);
  const [playerDeviceId, setPlayerDeviceId] = useState<string>("");

  const eventHandlers = useMemo<L.LeafletEventHandlerFnMap>(() => ({
    click(e: L.LeafletMouseEvent) {
      // @ts-ignore
      if (e.originalEvent.srcElement.className !== eventBlockClassName) {
        showPlayer(true);
      }
    },
    dragend() {
      const marker: L.Marker | null = markerRef.current;
      if (marker !== null) {
        const latLng = marker.getLatLng();

        updatePosition && updatePosition({lat: latLng.lat, lng: latLng.lng, alt: latLng.alt});

        setPosition(latLng);
      }
    }
  }), []);

  useEffect(() => {
    if (tracking && tracking.length > 0) {
      const lat = tracking[tracking.length - 1][0];
      const lng = tracking[tracking.length - 1][1];
      const newPosition = { lat, lng };
      setPosition(newPosition);
      return;
    }
    if (device?.position) {
      setPosition(device.position);
    }
    if (defaultPosition) {
      setPosition(defaultPosition);
    }
  }, [device?.position?.lat, device?.position?.lng, defaultPosition, tracking?.length]);

  useEffect(() => {
    setCurrentEvent(event);
  }, [event]);

  useEffect(() => {
    const span = document.createElement("span");
    span.addEventListener("click", () => {
      if (onCalloutClick) {
        onCalloutClick(event);
      }

      setCurrentEvent(undefined);
    });
    span.className = eventBlockClassName;
    span.innerText = currentEvent ? currentEvent.message : "";

    const marker = new L.DivIcon({
      className: classNames(
        mode === "edit" || currentEvent
          ? "leaflet-default-icon-path custom-rad-icon"
          : device?.enabled
            ? "leaflet-default-icon-path"
            : "leaflet-default-icon-path custom-gray-icon"
      ),
      html: span,
      iconSize: [25, 41],
      iconAnchor: [12.5, 41],
      popupAnchor: [3, -25],
    });

    markerRef.current?.setIcon(marker);
  }, [currentEvent, markerRef?.current]);

  const trackingPositions = Array.from(tracking || []);

  useEffect(() => {
    if (!popupRef.current?.isOpen()) {
      showPlayer(false);
    }
  }, [popupRef.current?.isOpen()]);

  useEffect(() => {
    if (device?.deviceType === "SENSOR") {
      //@ts-ignore
      device?.aspects[0]?.associatedDevices[0] ? setPlayerDeviceId(device?.aspects[0]?.associatedDevices[0]) : showPlayer(false);
    } else {
      setPlayerDeviceId(device?.id || "");
    }
  }, []);

  return (
    <>
      {position &&
      <Marker
        draggable={mode === "edit"}
        eventHandlers={eventHandlers}
        position={[position.lat, position.lng]}
        ref={markerRef}
      >
        <Tooltip>
          {device?.name || __("New device")}
        </Tooltip>
        {device &&
        <Popup ref={popupRef} closeOnClick={false}>
          {isShowPlayer && <PlayerCellLegacy
            obj={playerDeviceId}
            header={device.name}
            isLiveOnly
            fullScreenControl
            playOnStart
            showPauseButton
            onClose={() => {}}
          />}
        </Popup>}
      </Marker>}
      { tracking && tracking.length > 0 && <Polyline pathOptions={{ color: "lime" }} positions={trackingPositions} /> }
    </>
  );
};

export default DraggableMarker;
