import React, { useEffect, useMemo, useState } from "react";
import { Button, Icon, Segment } from "semantic-ui-react";

import { AuditCategoriesQuery, DeviceFunctionalAspectType, useAuditCategoriesLazyQuery, useDeviceWitnessesByAspectTypeLazyQuery, useUserWitnessesLazyQuery } from "@generated/graphql";
import { __ } from "@solid/libs";

import { MultipleOption } from "components/MultipleDropdown";
import EventFilter, { EventFilterSet, EventFilterState } from "components/EventFilter";
import { WidgetProps } from "components/Widgets";
import EventList from "components/EventList";
import { PeriodValue } from "components/Period";
import Loading from "components/Loading";

import "./style.css";

export enum EventsType {
  Users = "Users",
  Devices = "Devices",
  Archive = "Archive"
}

type AuditCategories = AuditCategoriesQuery["auditCategories"];
type EventsProps = {
  titleName?: string;
  liveOnly?: boolean;
  eventsType?: EventsType;
  withFilter?: boolean;
  filterSet?: EventFilterSet;
  paginate?: boolean;
} & WidgetProps;

export const defaultPeriodMaxValue: PeriodValue = { startFrom: new Date(Date.now()), endTo: new Date(Date.now()) };
export const defaultPeriodMinValue: PeriodValue = { startFrom: new Date(Date.now() - 60 * 24 * 60 * 60 * 1000), endTo: new Date(Date.now() - 60 * 24 * 60 * 60 * 1000) };
export const defaultPeriodValue: PeriodValue = { startFrom: new Date(Date.now() - 24 * 60 * 60 * 1000), endTo: new Date(Date.now()) };
export const userCategories = [101, 102, 103];
export const userAccessCategories = [...userCategories, 105, 106, 107];
export const userAccessCategoriesOld = [2, 3, 4, 5, 7, 8, 14, 17, 20, 24, 34, 45, 56, 57];
export const deviceCategoriesOld = [26, 27, 29, 30, 31, 32, 63, 64, 65, 66, 67, 68, 69];
export const deviceCategories = [109, 110, 111, 113, 114, 115, 117, 118, 119];
const initialCategories = {
  [EventsType.Users]: [...userAccessCategories, ...userAccessCategoriesOld],
  [EventsType.Devices]: [...deviceCategoriesOld, ...deviceCategories],
  [EventsType.Archive]: [47]
};
const defaultFilterSetValue: EventFilterSet = {
  cameras: true,
  avatars: true,
  categories: true,
  sensors: true,
  users: true
};

const Events = ({
  eventsType,
  titleName,
  liveOnly = true,
  withFilter = false,
  widgets,
  cellProps,
  paginate = false,
  filterSet = defaultFilterSetValue,
  setCellProps
}: EventsProps) => {
  const [getCategories, { data: categoriesData, loading: categoriesLoading, error: categoriesError }] = useAuditCategoriesLazyQuery();
  const [getCameras, { data: camerasData, loading: cameraLoading, /*error: cameraError*/ }] = useDeviceWitnessesByAspectTypeLazyQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
  const [getSensors, { data: sensorsData, loading: sensorLoading, /*error: sensorError*/ }] = useDeviceWitnessesByAspectTypeLazyQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Sensor }] } });
  const [getAvatars, { data: avatarsData, loading: avatarLoading, /*error: avatarError*/ }] = useDeviceWitnessesByAspectTypeLazyQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] }});
  const [getUsers, { data: usersData, loading: userLoading, /*error: userError*/ }] = useUserWitnessesLazyQuery();

  const [filterChanged, setFilterChanged] = useState<boolean>(true);
  const [getEventsCommand, setGetEventsCommand] = useState<number>();
  const [eventFilter, setEventFilter] = useState<EventFilterState>({
    categories: eventsType ? initialCategories[eventsType] : undefined,
    period: defaultPeriodValue
  });

  const updateEventFilter = (filter: keyof EventFilterState, value: EventFilterState[keyof EventFilterState]) => {
    setEventFilter(prevFilter => ({ ...prevFilter, ...{ [filter]: value } }));
    filter !== "text" && setFilterChanged(true);
  };

  const getEvents = () => {
    setGetEventsCommand(Math.random());
    setFilterChanged(false);
  };

  const entities = useMemo(() => {
    const entities = {
      cameras: getMultipleOptionsFromQuery(camerasData?.devicesByAspectTypes),
      sensors: getMultipleOptionsFromQuery(sensorsData?.devicesByAspectTypes),
      avatars: getMultipleOptionsFromQuery(avatarsData?.devicesByAspectTypes),
      users: getMultipleOptionsFromQuery(usersData?.users),
      categories: getMultipleOptionsFromAuditCategoriesQuery(categoriesData?.auditCategories, eventsType),
    };

    return entities;
  }, [camerasData, sensorsData, avatarsData, usersData, categoriesData, eventsType]);

  useEffect(() => {
    if (camerasData) {
      const cameras = camerasData.devicesByAspectTypes.map(camera => camera.id);
      setEventFilter(prev => ({ ...prev, cameras }));
    }
  }, [camerasData]);

  useEffect(() => {
    if (sensorsData) {
      const sensors = sensorsData.devicesByAspectTypes.map(sensor => sensor.id);
      setEventFilter(prev => ({ ...prev, sensors }));
    }
  }, [sensorsData]);

  useEffect(() => {
    if (avatarsData) {
      const avatars = avatarsData.devicesByAspectTypes.map(avatar => avatar.id);
      setEventFilter(prev => ({ ...prev, avatars }));
    }
  }, [avatarsData]);

  useEffect(() => {
    if (usersData) {
      const users = usersData.users.map(user => user.id);
      setEventFilter(prev => ({ ...prev, users }));
    }
  }, [usersData]);

  useEffect(() => {
    if (withFilter) {
      const { cameras, sensors, avatars, users, categories } = filterSet;

      cameras && getCameras();
      sensors && getSensors();
      avatars && getAvatars();
      users && getUsers();
      categories && getCategories();
    }

  }, [withFilter, filterSet]);

  const selectedEntities = [...(eventFilter?.avatars || []), ...(eventFilter?.cameras || []), ...(eventFilter?.sensors || []), ...(eventFilter?.users || [])];
  const selectedCategories = [...(eventFilter?.categories || [])];
  const loading = cameraLoading || sensorLoading || avatarLoading || userLoading || categoriesLoading;

  return (
    <Segment loading={eventsType !== EventsType.Archive && loading} error={categoriesError} className="Events-Wrapper">
      { eventsType === EventsType.Archive && loading ?
        <Loading/> :
        <>
          { withFilter &&
          <div className="Events-TopPanel">
            <EventFilter
              avatars={entities?.avatars}
              cameras={entities?.cameras}
              users={entities?.users}
              sensors={entities?.sensors}
              categories={entities?.categories}
              onChange={updateEventFilter}
              filterSet={filterSet}
            />
            <Button
              onClick={() => getEvents()}
              className="Events-GetButton"
              disabled={selectedCategories.length === 0 || selectedEntities.length === 0}
            >
              <Icon name="list ol" />
              {__("Get Events")}
            </Button>
          </div> }
          <EventList
            updateEventsCommand={getEventsCommand}
            filterCategories={selectedCategories as number[]}
            witnessesList={selectedEntities}
            titleName={titleName}
            widgets={widgets}
            cellProps={cellProps}
            setCellProps={setCellProps}
            liveOnly={liveOnly}
            withFilter={withFilter}
            startFrom={eventFilter.period?.startFrom}
            endTo={eventFilter.period?.endTo}
            filterText={eventFilter.text}
            paginate={paginate}
            eventFilterChanged={filterChanged}
        />
        </> }
    </Segment>
  );
};

export default Events;

function getMultipleOptionsFromQuery(queryData?: { __typename?: string; id: string; name: string; }[]): MultipleOption[] {
  if (!queryData) return [];
  return queryData
    .map(entity =>  {
      const multipleOption: MultipleOption = {
        options: [{
          label: entity.name,
          value: entity.id,
          checked: true,
        }]
      };
      return multipleOption;
    })
    .sort((a, b) => a.options[0].label.localeCompare(b.options[0].label));
}

function getMultipleOptionsFromAuditCategoriesQuery(categoriesData?: AuditCategories, eventsType?: EventsType): MultipleOption[] {
  if (!categoriesData) return [];

  const auditCategories = !eventsType ?
    categoriesData?.sort((a, b) => Number(b.id) - Number(a.id)) :
    categoriesData
      .filter(category => initialCategories[eventsType].includes(Number(category.id)))
      .sort((a, b) => Number(b.id) - Number(a.id));

  const categoryParents = categoriesData?.filter(category => !category.parent);
  const categories = auditCategories
    .filter(category => category.parent)
    .map(category => {
      const parent = categoryParents.find(item => category.parent === item.id);
      const multipleOption: MultipleOption = {
        group: parent ? {
          label: parent.shortName,
          value: parent.id,
          checked: true
        } : undefined,
        options: [{
          label: category.shortName,
          value: category.id,
          checked: true,
        }]
      };
      return multipleOption;
    });

  return categories;
}
