import React, { useState, useEffect } from "react";
import classNames from "classnames";
import {EventInfo, useAuditEntriesWithSnapshotsLazyQuery, DeviceFunctionalAspectType, useDeviceListByAspectTypesQuery} from "@generated/graphql";
import { AuditEntryWithSnapshots } from "@core/actions";
import WithQueryStatus from "components/WithQueryStatus";
import EventVisual from "components/EventList/EventVisual";
import EventAcknowledge from "components/EventList/EventAcknowledge";
import { EventPreviewPlacement } from "components/EventList";
import { formatDateTime } from "@core/utils";
import { WidgetEvent } from "components/Widgets";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";

import "./style.css";

type EventPreviewProps = {
  event: EventInfo;
  placement: EventPreviewPlacement;
  widgetEvent?: WidgetEvent;
  widgetIndex?: number;
  isTimeline?: boolean;
};

const EventPreview = ({ event, placement, widgetEvent, widgetIndex, isTimeline = false }: EventPreviewProps) => {
  const { loading: devLoading, error: devError, data: devData } = useDeviceListByAspectTypesQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
  const [getAuditEntries, { data, loading, error }] = useAuditEntriesWithSnapshotsLazyQuery({ fetchPolicy: "no-cache" });
  const [entry, setEntry] = useState<AuditEntryWithSnapshots | undefined>();
  const [cameraId, setCameraId] = useState<string | undefined>();

  useEffect(() => {
    if (event.hasSnapshots) {
      setEntry(event.entry);
    }
    else {
      getAuditEntries({
        variables: {
          filter: {
            from: event.entry.triggeredAt,
            to: event.entry.triggeredAt,
            witnesses: event.entry.witnesses.map(w => w.id),
            categories: ["47", "61"]
          }
        }});
    }
  }, [event]);

  useEffect(() => {
    if (data) {
      const entry = data.auditEntries.find(ev => ev.context === event.entry.context);
      setEntry(entry);
    }
  }, [data]);

  useEffect(() => {
    if (devError) {
      Log.error(`${__("Devices query error")}: ${devError.message}`);
    }
  }, [devError]);

  useEffect(() => {
    setCameraId(entry?.witnesses.find(w => devData?.devicesByAspectTypes.some(dev => dev.id === w.id))?.id);
  }, [entry, devData]);

  return (
    <div className={classNames("EventPreview", placement)}>
      <WithQueryStatus loading={loading || devLoading} error={error}>
        {!!entry &&
        <>
          <div className="EventPreview-Left">
            <div className="EventPreview-Detail">
              <div className="EventPreview-Name">{entry.witnesses.reduce((acc, { name }) => acc + (acc ? ", " : "") + name, "")}</div>
              <small className="EventPreview-Date">{formatDateTime(new Date(entry.triggeredAt))}</small>
              <div className="EventPreview-Message">{entry.message}</div>
            </div>

            {(entry.category.id === "47" || entry.category.id === "61") &&
            <div className="EventPreview-Acknowledge">
              <EventAcknowledge event={entry}/>
            </div>}
          </div>

          {(entry.snapshots.some(snap => snap.snapshot) || !!cameraId) &&
          <div className="EventPreview-Visual">
            <EventVisual event={entry} cameraId={cameraId} widgetEvent={widgetEvent} widgetIndex={widgetIndex} isTimeline={isTimeline}/>
          </div>}
        </>}
      </WithQueryStatus>
    </div>
  );
};

export default EventPreview;
