import React, { useState, useEffect, useRef, useMemo, useCallback } from "react";
import { useReactiveVar } from "@apollo/client";
import { useDrag, useDrop, DragSourceHookSpec } from "react-dnd";
import classNames from "classnames";
import { Segment, Ref, Popup, Icon, Button, Form } from "semantic-ui-react";
import { Table, Column, AutoSizer, ScrollEventData } from "react-virtualized";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { EventInfo, AuditSnapshot, WidgetId, AuditEntryFilter, useAuditCategoriesQuery, useDeviceListByAspectTypesLazyQuery, DeviceFunctionalAspectType, useEventsAcknowledgementMutation, AcknowledgeAuditEntryInput, useAuditEntriesWithSnapshotsLazyQuery, useRequireAcknowledgeWithSnapshotsEntriesLazyQuery } from "@generated/graphql";
import { useEventLog, AuditEntryWithSnapshots, isWithSnapshots, useEventsQuery, AuditEntriesAutoUpdateWidgets, isError, isAlert, AuditEntry, getAuditEntryInput, isAcknowledged, isAckRequired, isConfirmed, isExternal } from "@core/actions";
import { useStore, withStore, WithStoreProps } from "@core/store";
import { DragObjectType, DragObject, isObject } from "@core/types";
import { UUID } from "@solid/types";
import WithQueryStatus from "components/WithQueryStatus";
import {
  WidgetProps,
  WidgetEvent,
  EventClickEventArgs,
  CommonWidgetEvent,
  WidgetEventArgs,
  EventListAssociatedWidgets,
  WidgetWarningEventArgs
} from "components/Widgets";
import ListFilter from "../ListFilter";
import { EventPubSub } from "utils/EventPubSub";
import { formatDateTime } from "@core/utils";
import { Utils } from "@solid/libs/utils";
import StatusTitle from "components/StatusTitle";
import EventPreview from "components/EventList/EventPreview";
import EventListStub from "components/Events/EventListStub";
import PointMenu from "components/PointMenu";
import EventListFilter, { EventTypeFilter } from "./EventListFilter";
import { SolidConfigRoot, currentVersion } from "@core/store/actions";
import { parseJSON } from "utils";
import {__, Log} from "@solid/libs";

import "./style.css";

export class ShowHideFilterEvent extends EventPubSub<{}> {}

const rowHeight = 74;
const eventListLength = 1000;

export enum EventView {
  Content = "Content",
  Tiles = "Tiles"
}

export enum ResetToLiveInterval {
  Never = "Never",
  FiveMin = "5min",
  TenMin = "10min",
  FifteenMin = "15min",
  ThirtyMin = "30min"
}

export enum EventPreviewPlacement {
  Top = "Top",
  Bottom = "Bottom",
  Left = "Left",
  Right = "Right"
}

export enum EventListWarning {
  TooSmallForPreview = "TooSmallForPreview"
}

export type Entry = (AuditEntry | AuditEntryWithSnapshots) & {
  showSnapshots: boolean;
  userInteraction: boolean;
  isTimeline: boolean;
  widgetEvent?: WidgetEvent;
  widgetIndex: number;
  selected?: boolean;
  selectedEvent?: boolean;
};

type EventListProps = WidgetProps & WithStoreProps & {
  witnessesLoading?: boolean;
  titleName?: string;
  witnessesList?: UUID[];
  showPreview?: boolean;
  previewPlacement?: EventPreviewPlacement;
  eventView?: EventView;
  liveOnly?: boolean;
  associatedWidgets?: string[];
  resetToLiveAfter?: ResetToLiveInterval;
  showSnapshots?: boolean;
  userInteraction?: boolean;
  showHideFilterEvent?: ShowHideFilterEvent;
  exportCsvFile?: boolean;
  filterCategories?: number[];
  fromDays?: number;
  isCategoriesFilterVisible?: boolean;
  updateEventsCommand?: number;
  withFilter?: boolean;
  startFrom?: Date;
  endTo?: Date;
  filterText?: string;
  paginate?: boolean;
  limit?: number;
  eventFilterChanged?: boolean;
  multipleSelect?: boolean;
  onRowClick?: (event: EventInfo) => void;
  eventAcknowledgment?: boolean;
  eventTypeFilter?: EventTypeFilter[]
};

const paginateEntry: Entry = {
  context: "",
  message: "paginate",
  triggeredAt: new Date(),
  modifiedAt: new Date(),
  isAcknowledged: true,
  isTimeline: false,
  category: {
    id: "0",
    shortName: "paginate",
    digest: "paginate"
  },
  widgetIndex: 0,
  witnesses: [],
  snapshots: [],
  userInteraction: false,
  showSnapshots: false,
  metadata: []
};

const queryTypes = [
  { type: DeviceFunctionalAspectType.Media },
  { type: DeviceFunctionalAspectType.Sensor },
];

const defaultScrollEventData: ScrollEventData = {
  clientHeight: 0,
  scrollHeight: 0,
  scrollTop: 0,
};

const EventList = ({
  eventFilterChanged = false,
  paginate,
  filterText,
  limit = 100,
  startFrom = new Date(Date.now() - 24 * 60 * 60 * 1000),
  endTo = new Date(Date.now()),
  withFilter = false,
  updateEventsCommand,
  witnessesLoading = false,
  fromDays = 1,
  exportCsvFile,
  titleName,
  witnessesList,
  isViewActive,
  cellProps,
  setCellProps,
  widgets,
  previewPlacement = EventPreviewPlacement.Top,
  eventView = EventView.Content,
  liveOnly = true,
  associatedWidgets = [],
  filterCategories = [],
  resetToLiveAfter = ResetToLiveInterval.Never,
  showSnapshots = true,
  userInteraction = true,
  widgetEvent,
  showHideFilterEvent,
  index: widgetIndex = 0,
  store: { workspace: { updatedEvent, solidConfigJSON } },
  isCategoriesFilterVisible = false,
  multipleSelect = false,
  onRowClick,
  eventAcknowledgment = false,
  eventTypeFilter,
  ...props
}: EventListProps) => {
  const { data: auditData, loading: auditLoading, error: auditError } = useAuditCategoriesQuery();
  const isTimeline = !!widgets?.some(w => w.widgetId === WidgetId.Timeline);
  const isLiveOnly = liveOnly && !isTimeline;
  const [witnesses, setWitnesses] = useState<UUID[]>([]);
  const [categories, setCategories] = useState<string[] | undefined>();
  const [deviceName, setDeviceName] = useState("");
  const [filterVisible, setFilterVisible] = useState(false);
  const [searchText, setSearchText] = useState("");
  const [showPreview, setShowPreview] = useState(props.showPreview ?? false);
  const [isLive, setIsLive] = useState(isLiveOnly);
  const widgetId = `${props.viewId}_widget${widgetIndex}`;
  const [getDevices, { data: devData, loading: devLoading, error: devError }] = useDeviceListByAspectTypesLazyQuery({ variables: { types: queryTypes } });
  const {
    entry: liveEvent,
    subscriptionLoading: subLoading,
    subscriptionError: subError,
    entries: liveEvents,
    queryError: liveError,
    queryLoading: liveLoading
  } = useEventLog({
    fromDays,
    witnesses,
    categories,
    skipSubscription: !isViewActive || !isLiveOnly,
    skipQuery: isTimeline || withFilter,
    withSnapshots: showSnapshots,
    widgetId
  });
  const [getEvents] = useAuditEntriesWithSnapshotsLazyQuery({ fetchPolicy: "no-cache" });
  const [getArchEvents, { data: archEvents, loading: archLoading, error: archError }] = useEventsQuery(showSnapshots);
  const [getRequireAckEvents, { data: reqAckEvents, loading: reqAckLoading, error: reqAckError }] = useRequireAcknowledgeWithSnapshotsEntriesLazyQuery({ fetchPolicy: "network-only" });
  const [acknowledgeEvents, { data: ackData, loading: ackLoading, error: ackError }] = useEventsAcknowledgementMutation();
  const [entries, setEntries] = useState<Entry[]>([]);
  const [eventList, setEventList] = useState<Entry[]>([]);
  const [stopTime, setStopTime] = useState<Date | undefined>();
  const [overscanRows, setOverscanRows] = useState(10);
  const [event, setEvent] = useState<EventInfo | undefined>();
  const [exportCsv, setExportCsv] = useState<boolean>(false);
  const [updateArchiveEventsFlag, setUpdateArchiveEventsFlag] = useState<number>();
  const [paginationEventCount, setPaginationEventCount] = useState<number>(0);
  const [scrollData, setScrollData] = useState<ScrollEventData | undefined>(isLive ? defaultScrollEventData : undefined);
  const [autoScroll, setAutoScroll] = useState<boolean>(false);
  const [selectedRowId, setSelectedRowId] = useState<string>();
  const [multipleSelectColumnVisible, setMultipleSelectColumnVisible] = useState<boolean>(multipleSelect);
  const [selectedEventIds, setSelectedEventIds] = useState<Set<string>>(new Set());
  const [entryTypeFilter, setEntryTypeFilter] = useState<EventTypeFilter[] | undefined>(eventTypeFilter);
  const currentLanguage = localStorage.getItem("language") ?? "";
  const rootRef = useRef<HTMLDivElement>(null);
  const resizeObserverRef = useRef<ResizeObserver>();
  const resetToLiveTimeoutRef = useRef<NodeJS.Timeout | undefined>();
  const tableRef = useRef<Table>(null);
  const newEventCount = useMemo(() => getNewEventCount(), [liveEvent, isLive, isTimeline, stopTime, witnesses]);
  const withLoadMoreButton = (archEvents?.auditEntries.length || 0) === limit;
  const autoUpdateWidgets = useReactiveVar(AuditEntriesAutoUpdateWidgets);

  const isViewHasLiveWidgets: boolean = useMemo(() => {
    const currentConfig = parseJSON<SolidConfigRoot>(solidConfigJSON)?.[currentVersion];
    if (!currentConfig || !props.viewId) return false;

    const eventVideoWidget = currentConfig.allViews
      ?.find(view => view.id === props.viewId)
      ?.widgets?.find(w => w.widgetId === WidgetId.EventVideo);

    const eventInfoWidget = currentConfig.allViews
      ?.find(view => view.id === props.viewId)
      ?.widgets?.find(w => w.widgetId === WidgetId.EventDetails);

    if (!eventVideoWidget && !eventInfoWidget) return false;

    return (!eventVideoWidget?.props || eventVideoWidget?.props?.["autoUpdate"] === true) || (!eventInfoWidget?.props || eventInfoWidget?.props?.["autoUpdate"] === true);
  }, [props.viewId, solidConfigJSON]);

  const isAllSelected = useMemo(() => {
    return eventList.filter(ev => !isAcknowledged(ev)).length === selectedEventIds.size && selectedEventIds.size !== 0;
  }, [eventList, selectedEventIds]);

  const [{ dragOver }, dropRef] = useDrop<DragObject, {}, { dragOver: boolean }>({
    accept: [DragObjectType.Camera],

    drop: (item, monitor) => {
      if (!isObject(item.obj)) {
        return undefined;
      }
      switchToArchive({ deviceId: item.obj["data-id"], deviceName: item.obj.name, widgetIndex });
      return undefined;
    },

    collect: monitor => ({
      dragOver: monitor.isOver()
    }),

    canDrop: (item, monitor) => {
      if (liveOnly || isTimeline) {
        return false;
      }
      if (!item.sourceWidgetId || !associatedWidgets.includes(item.sourceWidgetId)) {
        return false;
      }
      if (!isObject(item.obj)) {
        return false;
      }
      return true;
    }
  });

  useEffect(() => {
    const root = rootRef.current;
    if (root) {
      resizeObserverRef.current = new ResizeObserver(Utils.throttleToDraw((entries: ResizeObserverEntry[]) => {
        for (const entry of entries) {
          if (entry.target === root) {
            setOverscanRows(Math.max(Math.trunc(entry.contentRect.height / 74), 1));
            break;
          }
        }
      }));
      resizeObserverRef.current.observe(root);
    }

    return function cleanup() {
      isViewHasLiveWidgets && AuditEntriesAutoUpdateWidgets(true);
      clearResetToLiveTimer();
      root && resizeObserverRef.current?.unobserve(root);
    };
  }, []);

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    widgetEvent?.subscribe(id, onWidgetEvent);

    return function cleanup() {
      widgetEvent?.unsubscribe(id);
    };
  });

  useEffect(() => {
    setEntryTypeFilter(eventTypeFilter);
  }, [eventTypeFilter]);

  useEffect(() => {
    const newEntries: Entry[] = entries.filter(entry => isEntryBelongToFilter(entry));
    setEventList(newEntries);
  }, [entries, searchText, entryTypeFilter]);

  useEffect(() => {
    const events = eventAcknowledgment ? (reqAckEvents?.requireAcknowledgeEntries ?? []) : (archEvents?.auditEntries ?? []);
    if (events.length > 0) {
      setEntries(entries => {
        const contextSet = new Set<string>();
        let maxTime = 0;
        const archEventsData = events.map(entry => {
          contextSet.add(entry.context);
          maxTime = Math.max(maxTime, new Date(entry.triggeredAt).getTime());
          const newEntries = getEntry(entry);
          return newEntries;
        });
        let records = entries.filter(entry => new Date(entry.triggeredAt).getTime() >= maxTime && !contextSet.has(entry.context));
        records = records.concat(archEventsData);
        records.sort((a, b) => new Date(b.triggeredAt).getTime() - new Date(a.triggeredAt).getTime());
        if (records.length > eventListLength) {
          const newRecords = getSlicedEntries(records, eventListLength);
          return newRecords;
        }
        return records;
      });
    }
  }, [archEvents, reqAckEvents]);

  useEffect(() => {
    if (liveEvents && liveEvents.length > 0 && !isLiveOnly) {
      setEntries(entries => {
        const contextSet = new Set<string>();
        let maxTime = 0;
        const liveEventsData = liveEvents.map(entry => {
          contextSet.add(entry.context);
          maxTime = Math.max(maxTime, new Date(entry.triggeredAt).getTime());
          const newEntry = getEntry(entry);
          return newEntry;
        });
        let records = entries.filter(entry => new Date(entry.triggeredAt).getTime() >= maxTime && !contextSet.has(entry.context));
        records = records.concat(liveEventsData);
        records.sort((a, b) => new Date(b.triggeredAt).getTime() - new Date(a.triggeredAt).getTime());
        if (records.length > eventListLength) {
          const newRecords = getSlicedEntries(records, eventListLength);
          return newRecords;
        }
        return records;
      });

      if (autoScroll) {
        const entryIds = eventList.map(entry => entry.context);
        const newEventsCount = liveEvents.filter(event => !entryIds.includes(event.context)).length;
        if (newEventsCount > 0) {
          setScrollData(prevScrollData => {
            if (!prevScrollData) return prevScrollData;
            const { scrollTop: prevScroll } = prevScrollData;
            const newScrollTop = prevScroll + (newEventsCount * rowHeight);

            if (autoScroll) {
              return { ...prevScrollData,  scrollTop: newScrollTop };
            }

            return prevScroll > 0 ? { ...prevScrollData,  scrollTop: newScrollTop } : prevScrollData;
          });
        }
      }
    }
  }, [liveEvents]);

  useEffect(() => {
    if (!liveEvent || isError(liveEvent)) return;

    const liveEntry = getEntry(liveEvent);

    if (isLiveOnly) {
      const isNewEvent = !entries.some(entry => entry.context === liveEntry.context);

      // if the new event is an update add lose entry
      if (isNewEvent && liveEntry.modifiedAt !== liveEntry.triggeredAt && entries.length > 0) {
        const addLoseEntry = async () => {
          try {
            const filter = {
              from: new Date(liveEntry.triggeredAt),
              to: new Date(liveEntry.triggeredAt),
              witnesses: liveEntry.witnesses.map(w => w.id),
              categories: ["47", "61"]
            };
            const { data, error } = await getEvents({ variables: { filter } });

            if (data && data.auditEntries) {
              const loseEntry = getEntry(data.auditEntries[0]);
              setEntries(entries => {
                const records = Array.from(entries);
                const index = records.findIndex(record => record.context === loseEntry.context);
                if (index >= 0) {
                  return entries;
                }
                records.unshift(loseEntry);
                if (records.length > eventListLength) {
                  const newRecords = getSlicedEntries(records, eventListLength);
                  return newRecords;
                }
                return records;
              });
            }
            if (error) {
              console.error("getEvents error:", error.message);
            }
          }
          catch (e) {
            console.error(e);
          }
        };

        addLoseEntry();
        return;
      }

      if (isNewEvent && isEntryBelongToFilter(liveEntry)) {
        setScrollData(prevScrollData => {
          if (!prevScrollData) return prevScrollData;
          const { scrollTop: prevScroll } = prevScrollData;
          const newScrollTop = prevScroll + rowHeight;

          if (autoScroll) {
            return { ...prevScrollData,  scrollTop: newScrollTop };
          }

          return prevScroll > 0 ? { ...prevScrollData,  scrollTop: newScrollTop } : prevScrollData;
        });
      }

      setEntries(entries => {
        const records = Array.from(entries);
        const index = records.findIndex(record => record.context === liveEntry.context);
        if (index >= 0) {
          mergeLiveEntry(records, liveEntry, index);
        }
        else {
          records.unshift(liveEntry);
        }

        if (records.length > eventListLength) {
          const newRecords = getSlicedEntries(records, eventListLength);
          return newRecords;
        }

        return records;
      });

    }
    else {
      // update archive events
      setEntries(entries => {
        const index = entries.findIndex(record => record.context === liveEvent.context);
        if (index >= 0) {
          const records = Array.from(entries);
          mergeLiveEntry(records, liveEntry, index);
          return records;
        }

        return entries;
      });
    }
  }, [liveEvent]);

  useEffect(() => {
    if (!scrollData || !isLive) return;
    const { clientHeight, scrollHeight, scrollTop } = scrollData;
    if (scrollTop > 0 && !autoScroll) {
      setAutoScroll(true);
    }
    if (scrollTop === 0 && autoScroll) {
      setAutoScroll(false);
    }
    // updating table rows on new Event then scroll in the end table
    if (clientHeight + scrollTop >= scrollHeight) {
      tableRef.current?.Grid.handleScrollEvent({ scrollTop });
    }
  }, [scrollData]);

  useEffect(() => {
    if (isLiveOnly) {
      !witnessesList && getDevices();
      const isQueryParamProcessed = witnesses && categories && categories.length > 0;
      const isQueryExecuted = eventAcknowledgment ? reqAckEvents?.requireAcknowledgeEntries : archEvents?.auditEntries;
      const isQueryCanExecute = !isQueryExecuted && isQueryParamProcessed;
      if (!isQueryCanExecute) return;

      const filter: AuditEntryFilter = {
        limit,
        from: new Date(Date.now() - fromDays * 24 * 60 * 60 * 1000),
        to: new Date(),
        start: 0,
        witnesses,
        categories
      };

      if (eventAcknowledgment) {
        getRequireAckEvents({ variables: { filter } });
      }
      else {
        getArchEvents({ variables: { filter } });
      }
    }
  }, [isLiveOnly, categories, witnesses]);

  useEffect(() => {
    const queryArgs: AuditEntryFilter = {
      witnesses: witnessesList || [],
      categories: filterCategories.map(String),
      from: startFrom,
      to: endTo,
      limit,
      start: 0
    };

    if (updateArchiveEventsFlag !== updateEventsCommand) {
      getArchiveEvents(queryArgs);
      setUpdateArchiveEventsFlag(updateEventsCommand);
      setPaginationEventCount(0);
    }
  }, [updateEventsCommand, startFrom, endTo, witnessesList]);

  const loadMore = useCallback((start: number) => {
    const queryArgs: AuditEntryFilter = {
      witnesses: witnessesList || [],
      categories: filterCategories.map(String),
      from: startFrom,
      to: endTo,
      limit,
      start,
    };

    getArchiveEvents(queryArgs);

  }, [startFrom, endTo, filterCategories, paginationEventCount, witnessesList]);

  useEffect(() => {
    if (witnessesList && JSON.stringify(witnesses) !== JSON.stringify(witnessesList)) {
      setWitnesses(witnessesList);
    }
  }, [witnessesList]);

  useEffect(() => {
    if (devData && devData.devicesByAspectTypes && !witnessesList) {
      const witnessList = devData.devicesByAspectTypes.map(obj => obj.id);
      setWitnesses(witnessList);
      setCategories(["47", "61"]);
    }

    const auditCategories = auditData?.auditCategories;
    if (witnessesList && auditCategories) {
      const rawCategories = auditCategories
        .filter((item) =>
          (!filterCategories || !filterCategories.length || filterCategories.includes(Number(item.id))) &&
          item.id !== "47" && item.id !== "61")
        .map((item) => item.id);
      setCategories(rawCategories);
    }
  }, [auditData, devData]);

  useEffect(() => {
    setSearchText(filterText || "");
  }, [filterText]);

  useEffect(() => {
    if (!archEvents?.auditEntries || searchText || isLive) {
      return;
    }

    const entries: Entry[] = archEvents.auditEntries.map(entry => getEntry(entry));

    if (paginationEventCount > 0) {
      setEntries(prevEntries => {
        const paginationEventIndex = prevEntries.findIndex(entry => entry.message === "paginate");
        paginationEventIndex >= 0 && prevEntries.splice(paginationEventIndex, 1);
        const newEntries = [...prevEntries, ...entries];
        entries.length === limit && newEntries.push(paginateEntry);

        return newEntries;
      });
      return;
    }

    archEvents.auditEntries.length === limit && paginate && entries.push(paginateEntry);
    setEntries(entries);

  }, [archEvents, paginationEventCount, searchText]);

  useEffect(() => {
    if (exportCsvFile) {
      setExportCsv(exportCsvFile);
    }
  }, [exportCsvFile]);

  useEffect(() => {
    if (exportCsv) {
      const rows: string[][] = [
        ["Event ID", "Date/Time", "Witness name", "Message"],
      ];
      eventList.forEach(event => {
        event.message !== "paginate" && rows.push([
          `${event.context}`,
          `"${formatDateTime(new Date(event.triggeredAt), currentLanguage)}"`,
          `${event.witnesses[0].name}`,
          `${event.message}`
        ]);
      });

      const csvContent = "data:text/csv;charset=utf-8," + encodeURIComponent(rows.map(row => row.join(",")).join("\n"));
      const link = document.createElement("a");
      link.setAttribute("href", csvContent);
      link.setAttribute("download", "events.csv");
      document.body.appendChild(link);
      link.click();
      link.remove();

      setExportCsv(false);
    }
  }, [exportCsv]);

  useEffect(() => {
    if (ackData?.auditEntriesAcknowledgment) {
      setSelectedEventIds(new Set());
      setEventList((prevList) => {
        const newList = prevList.map((entry) => {
          entry.selectedEvent = false;
          return entry;
        });
        return newList;
      });
    }
  }, [ackData]);

  useEffect(() => {
    if (ackError) {
      Log.error(ackError.message);
    }
  }, [ackError]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }

    const textFilter =
      <Popup
        content={filterVisible ? __("Hide Filter") : __("Show Filter")}
        position="bottom right"
        trigger={
          <Icon
            name="search"
            className="EventList-TitleIcon"
            onClick={() => setFilterVisible(visible => !visible)}/>
        }
      />;

    const exportCsv =
      <Popup
        trigger={
          <Icon
            disabled={eventFilterChanged}
            name="download"
            className="EventList-TitleIcon"
            onClick={() => setExportCsv(true)}/>}
        content={__("Export CSV")}
        position="bottom right"
      />;

    const eventFilter =
      <Popup
        basic
        hoverable
        className="EventListFilter-Popup"
        trigger={<Icon name="filter" disabled={eventFilterChanged} />}
        content={<EventListFilter onChange={(value) => setEntryTypeFilter(value)} filter={entryTypeFilter} />}
        position="bottom right"
        />;

    const pointerMenuControls =
      <>
        {exportCsv}
        {!withFilter && textFilter}
        {eventFilter}
      </>;

    const isAllowAcknowledge = eventView === EventView.Content && (categories?.includes("47") || filterCategories.includes(47));
    const name = eventAcknowledgment ? __("Events: Acknowledgment") : isLiveOnly ? __("Live") : (deviceName || __("Associated"));
    const headerTitle = eventAcknowledgment ? name : !titleName ? (cellProps?.title ?? "") + (name ? ": " : "") + name : (cellProps?.title ?? "") + (titleName ? ": " : "") + titleName;
    const title =
      <div className="EventList-Header">
        <div className="EventList-Title">
          {headerTitle}
          { selectedEventIds.size > 0 &&
          <Button
            positive
            size="mini"
            className="EventList-AcknowledgeButton"
            loading={ackLoading}
            onClick={acknowledge}
          >
            <Icon name="check" />
            {__("Acknowledge")}
          </Button> }
        </div>
        { isLive && (autoScroll || (isViewHasLiveWidgets && !autoUpdateWidgets)) &&
        <Popup
          trigger={
            <Icon
              className="EventList-AutoUpdateButton"
              name="arrow circle up"
              onClick={() => {
                isViewHasLiveWidgets && AuditEntriesAutoUpdateWidgets(true);
                setSelectedRowId(undefined);
                setAutoScroll(false);
                setScrollData(prevData => {
                  if (!prevData) return prevData;
                  return { ...prevData, scrollTop: 0 };
                });
              }} />}
          content={__("Go Up and Clear Selection")}
          position="bottom right"
          /> }

        { !eventAcknowledgment ?
          <>
            { isAllowAcknowledge &&
            <Popup
              content={multipleSelectColumnVisible ? __("Hide selection") : __("Select Events")}
              position="bottom right"
              trigger={
                <Icon
                  disabled={eventFilterChanged}
                  className="EventList-SelectGroupButton"
                  name="checkmark box"
                  onClick={() => setMultipleSelectColumnVisible(prevState => !prevState)} />}
            /> }

            { !withFilter ?
              <PointMenu>
                {pointerMenuControls}
              </PointMenu> :
              <>
                {exportCsv}
                {isAllowAcknowledge && eventFilter}
              </> }
          </> :
          textFilter
        }
      </div>;
    /* eslint-disable react/jsx-indent, react/jsx-closing-tag-location */

    const queryError = !!archError || !!liveError || reqAckError;
    const queryLoading = archLoading || reqAckLoading;
    if (queryLoading || queryError || subError || (!isLive && newEventCount > 0)) {
      setCellProps({
        title:
          <div className="EventList-Header">
            <div className="EventList-Title">
              <StatusTitle title={headerTitle} loading={queryLoading || (isLive && liveLoading)}
                loadingText={queryLoading ? __("Event List is loading...") : __("Waiting for events...")}
                error={archError || subError}
                errorContent={<>
                  {queryError && __("Event List error: ") + (archError?.message ?? liveError?.message ?? reqAckError?.message)}
                  {queryError && !!subError && <br />}
                  {!!subError && __("Subscription error: ") + subError.message}
                </>} />
            </div>

            { !queryLoading &&
              <PointMenu>
                {pointerMenuControls}
              </PointMenu> }

            {!isLive && newEventCount > 0 &&
              <div className="EventList-NewEvents" onClick={() => switchToLive()}>
                {`${newEventCount >= 100 ? "> " : ""}${newEventCount} ${__("new event(s)")}`}
                <Icon name="refresh" />
              </div>}
          </div>
      });
    } else {
      setCellProps({ title });
    }
    /* eslint-enable react/jsx-indent, react/jsx-closing-tag-location */
  },
  [
    archLoading,
    subLoading,
    liveLoading,
    liveError,
    archError,
    subError,
    isLive,
    deviceName,
    newEventCount,
    isLiveOnly,
    eventFilterChanged,
    autoScroll,
    autoUpdateWidgets,
    selectedEventIds.size,
    multipleSelectColumnVisible,
    entryTypeFilter,
    eventAcknowledgment,
    eventView,
    ackLoading,
    reqAckError,
    reqAckLoading,
    categories
  ]);

  useEffect(() => {
    if (!updatedEvent) {
      return;
    }

    if (event && updatedEvent.context === event.entry.context) {
      setEvent({ entry: updatedEvent, hasSnapshots: event.hasSnapshots });
    }
  }, [updatedEvent]);

  const getArchiveEvents = ({ from, to, witnesses, categories, limit, start }: AuditEntryFilter) => {
    getArchEvents({
      variables: {
        filter: {
          from,
          to,
          witnesses,
          categories,
          limit,
          start
        }
      }
    });

    setPaginationEventCount(start || 0);
  };

  const onWidgetEvent = (args: WidgetEventArgs): void => {
    if (args.event === CommonWidgetEvent.WidgetWarning) {
      const evArgs = args.args as WidgetWarningEventArgs;
      if (evArgs.widgetIndex === widgetIndex) {
        setShowPreview(evArgs.warning && evArgs.warning.code === EventListWarning.TooSmallForPreview ? false : (props.showPreview ?? false));
      }
      return;
    }

    if (args.event !== CommonWidgetEvent.EventClick) {
      return;
    }

    const evArgs = args.args as EventClickEventArgs;

    const widget = widgets && evArgs.widgetIndex < widgets.length ? widgets[evArgs.widgetIndex] : undefined;
    if (!widget || !widget.widgetId) {
      return;
    }

    if (evArgs.widgetIndex === widgetIndex && widget.widgetId === WidgetId.EventList &&
        evArgs.event && evArgs.event.entry.context === event?.entry.context) {
      setEvent(undefined);
      return;
    }

    if (showPreview && evArgs.widgetIndex === widgetIndex && widget.widgetId === WidgetId.EventList) {
      switchToArchive(evArgs, true);
      return;
    }

    if (isLiveOnly || !EventListAssociatedWidgets.includes(widget.widgetId as WidgetId)) {
      return;
    }

    let handleEvent: boolean;
    if (isTimeline) {
      handleEvent = widget.widgetId === WidgetId.Timeline;
    }
    else {
      handleEvent = associatedWidgets.includes(widget.widgetId);
    }

    if (handleEvent) {
      switchToArchive(evArgs);
    }
  };

  const getEntry = (event: AuditEntry | AuditEntryWithSnapshots): Entry => {
    return {
      ...event,
      showSnapshots,
      userInteraction,
      isTimeline,
      widgetEvent,
      widgetIndex,
      selectedEvent: false
    };
  };

  function acknowledge() {
    const eventsToAcknowledge: AcknowledgeAuditEntryInput[] = eventList
      .filter(ev => selectedEventIds.has(ev.context))
      .map((ev) => getAuditEntryInput(ev));
    acknowledgeEvents({ variables: { entries: eventsToAcknowledge } });
  }

  function isEntryBelongToFilter(entry: Entry): boolean {
    const text = searchText.toLocaleUpperCase();

    return (!text ||
      entry.message.toLocaleUpperCase().includes(text) ||
      entry.note?.toLocaleUpperCase().includes(text) ||
      formatDateTime(new Date(entry.triggeredAt), currentLanguage).includes(text) ||
      entry.witnesses.reduce((acc, { name }) => acc + (acc ? ", " : "") + name, "").toLocaleUpperCase().includes(text) ||
      Utils.shortUuid(entry.context, false).toLocaleUpperCase().includes(text)) &&
    (!entryTypeFilter ||
      (entryTypeFilter?.includes(EventTypeFilter.ALERT) && isAlert(entry)) ||
      (entryTypeFilter?.includes(EventTypeFilter.INFO) && !isAlert(entry)) ||
      (entryTypeFilter?.includes(EventTypeFilter.NEEDACKNOWLEDGE) && isAckRequired(entry)));
  }

  function switchToArchive(args: EventClickEventArgs, thisWidget = false): void {
    const { deviceId, deviceName, startTime, endTime, categories, event } = args;

    clearResetToLiveTimer();

    const newWitnesses = [deviceId];
    if (!thisWidget) {
      if (JSON.stringify(witnesses) !== JSON.stringify(newWitnesses)) {
        setEvent(undefined);
      }
      setWitnesses([deviceId]);
      setCategories(categories);
      setDeviceName(deviceName);
    }

    if (!thisWidget || isLive) {
      setStopTime(new Date());
    }

    if (!thisWidget) {
      getArchEvents({
        variables: {
          filter: {
            from: startTime ?? new Date(Date.now() - fromDays * 24 * 60 * 60 * 1000),
            to: endTime ?? new Date(),
            witnesses: newWitnesses,
            categories
          }
        }
      });
    }
    else {
      setEvent(event);
    }

    setIsLive(false);

    const interval = getResetToLiveSeconds();
    if (interval > 0 && !isTimeline) {
      resetToLiveTimeoutRef.current = globalThis.setTimeout(() => switchToLive(), interval * 1000);
    }
  }

  function switchToLive(): void {
    setIsLive(true);
    clearResetToLiveTimer();
  }

  function getResetToLiveSeconds(): number {
    switch (resetToLiveAfter) {
      case ResetToLiveInterval.FiveMin: return 5 * 60;
      case ResetToLiveInterval.TenMin: return 10 * 60;
      case ResetToLiveInterval.FifteenMin: return 15 * 60;
      case ResetToLiveInterval.ThirtyMin: return 30 * 60;
    }
    return 0;
  }

  function clearResetToLiveTimer(): void {
    if (resetToLiveTimeoutRef.current) {
      clearTimeout(resetToLiveTimeoutRef.current);
    }
  }

  function getNewEventCount(): number {
    if (!isLive && !isTimeline && stopTime) {
      return entries.reduce((acc, ev) =>
        acc + (new Date(ev.triggeredAt).getTime() > stopTime.getTime() && ev.witnesses.some(w => witnesses.includes(w.id)) ? 1 : 0), 0);
    }
    return 0;
  }

  function onCellClick(rowId: string) {
    if (isLive) {
      if (selectedRowId !== rowId) {
        setSelectedRowId(rowId);
        !autoScroll && setAutoScroll(true);
        isViewHasLiveWidgets && AuditEntriesAutoUpdateWidgets(false);
      }
      else {
        scrollData?.scrollTop === 0 && autoScroll && setAutoScroll(false);
        setSelectedRowId(undefined);
        isViewHasLiveWidgets && AuditEntriesAutoUpdateWidgets(true);
      }
    }
  }

  function selectAll(selected: boolean) {
    if (selected) {
      const allIdsSet = new Set([...eventList.filter(entry => !isAcknowledged(entry)).map(entry => entry.context)]);
      setSelectedEventIds(allIdsSet);
    }
    else {
      setSelectedEventIds(new Set());
    }
    setEventList(prevEntries => {
      const newEntries = prevEntries.map(entry => {
        if (selected) {
          entry.selectedEvent = selected && !isAcknowledged(entry);
        }
        else {
          entry.selectedEvent = selected;
        }
        return entry;
      });
      return newEntries;
    });
  }

  function onSelectEntry(rowData: Entry, rowIndex: number, checked?: boolean) {
    setEventList(prevEntries => {
      const newEntries = Array.from(prevEntries);
      newEntries[rowIndex].selectedEvent = checked;
      return newEntries;
    });
    const eventId = rowData.context;
    if (checked) {
      selectedEventIds.add(eventId);
    }
    else {
      selectedEventIds.delete(eventId);
    }
  }

  function mergeLiveEntry(entries: Entry[], liveEntry: Entry, liveEntryIndex: number): void {
    const prevEntry = entries[liveEntryIndex];
    let snapshots = isWithSnapshots(prevEntry) ? prevEntry.snapshots : [];
    if (isWithSnapshots(liveEntry) && liveEntry.snapshots.some(snap => snap.default)) {
      snapshots = liveEntry.snapshots;
    }

    const metadata = Array.from(prevEntry.metadata);
    for (let i = 0; i < liveEntry.metadata.length; i++) {
      const liveMeta = liveEntry.metadata[i];
      const liveMetaIndex = metadata.findIndex(data => data.key === liveMeta.key);
      if (liveMetaIndex >= 0) {
        metadata.splice(liveMetaIndex, 1, liveMeta);
      }
      else {
        metadata.push(liveMeta);
      }
    }

    entries.splice(liveEntryIndex, 1, {
      ...liveEntry,
      snapshots,
      metadata,
      selectedEvent: selectedEventIds.has(liveEntry.context),
      message: liveEntry.message ? liveEntry.message : prevEntry.message,
    });
  }

  return (
    <>
      { eventFilterChanged ?
        <EventListStub/> :
        <Ref innerRef={rootRef}>
          <Segment className="EventList">
            <WithQueryStatus
              loading={isLive ? devLoading || (archLoading && !archError) || (reqAckLoading && !reqAckError) || witnessesLoading || auditLoading : (archLoading && paginationEventCount === 0)}
              error={isLive ? (devError || auditError || (entries.length === 0 ? (archError ?? reqAckError) : undefined)) : (archError ?? reqAckError)}>
              { filterVisible && !withFilter &&
                <ListFilter
                  filterTextPlaceholder={__("Filter by text")}
                  nameFilter
                  searchText={searchText}
                  rootRef={rootRef}
                  onSearchTextChange={text => setSearchText(text)}
              /> }

              <div ref={dropRef} className={classNames("EventList-Root", { [previewPlacement]: showPreview && !!event, "EventList-Root_dragOver": dragOver })}>
                {!isLiveOnly && !deviceName && !withFilter ?
                  <div className="EventList-Empty"><p>{!isTimeline ? __("Drop Camera here") : __("Select Event(s) on a Timeline")}</p></div> :
                  <>
                    {showPreview && !!event && (previewPlacement === EventPreviewPlacement.Top || previewPlacement === EventPreviewPlacement.Left) &&
                    <div className={classNames("EventList-Preview", previewPlacement)}>
                      <EventPreview event={event} placement={previewPlacement} widgetEvent={widgetEvent} widgetIndex={widgetIndex} isTimeline={isTimeline}/>
                    </div>}
                    <div className={classNames("EventList-Content", { [previewPlacement]: showPreview && !!event })}>
                      {eventView === EventView.Content ?
                        eventList.length > 0 &&
                        <>
                          <div className={classNames("EventList-Grid", { "withPaginate": withFilter })}>
                            <AutoSizer style={{ width: "100%", height: "100%" }} className={classNames("EventList-AutoSizer")}>
                              {({ width, height }) => (
                                <>
                                  <Table
                                    ref={tableRef}
                                    key={selectedEventIds.size}
                                    className="EventList-Table"
                                    rowClassName={({ index }) => classNames("EventList-TableRow", {
                                      "selected": index >= 0 && eventList[index].context === event?.entry.context,
                                      "selectedRow": typeof selectedRowId === "string" && index >= 0 && eventList[index].context === selectedRowId,
                                      "alert": index >= 0 && eventList[index].metadata && isAlert(eventList[index]),
                                      "info": index >= 0 && eventList[index].metadata && !isAlert(eventList[index])
                                    })}
                                    width={width}
                                    height={height}
                                    rowHeight={rowHeight}
                                    headerHeight={34}
                                    rowCount={eventList.length}
                                    rowGetter={({ index }) => eventList[index]}
                                    overscanRowCount={overscanRows}
                                    onScroll={(props) => isLive && setScrollData(props)}
                                    scrollTop={scrollData?.scrollTop}
                                  >
                                    { multipleSelectColumnVisible &&
                                      <Column dataKey="multipleSelect" width={38} flexGrow={0} flexShrink={0}
                                        label={
                                          <Form.Checkbox
                                            checked={isAllSelected}
                                            onChange={(e, data) => selectAll(!!data.checked)} />
                                        }
                                        cellRenderer={props =>
                                          <Form.Checkbox
                                            disabled={isAcknowledged(props.rowData)}
                                            checked={props.rowData.selectedEvent}
                                            onChange={(e, data) => onSelectEntry(props.rowData, props.rowIndex, data.checked)} />
                                        }
                                      /> }
                                    { showSnapshots &&
                                    <Column dataKey="snapshots" label={__("Event")} width={74} flexGrow={0} flexShrink={0} cellRenderer={props => <SnapshotCell {...props}/>} className="EventList-ImageColumn"/> }
                                    <Column dataKey="id" label={__("Id")} width={84} flexGrow={0} cellRenderer={props => <IdCell {...props} onCellClick={() => onCellClick(props.rowData.context)} onRowClick={onRowClick} />}/>
                                    <Column dataKey="witnesses" label={__("Info")} width={150} flexGrow={1} cellRenderer={props => <InfoCell {...props} onCellClick={() => onCellClick(props.rowData.context)} onRowClick={onRowClick} />}/>
                                    <Column dataKey="message" label={__("Message")} width={450} flexGrow={3} cellRenderer={props =>
                                      <MessageCell
                                        onRowClick={onRowClick}
                                        onCellClick={() => onCellClick(props.rowData.context)}
                                        onClick={() => loadMore(paginationEventCount === 0 ? limit + 1 : paginationEventCount + limit)}
                                        loading={archLoading}
                                        withPaginate={withLoadMoreButton}
                                        {...props}
                                      />}
                                    />
                                  </Table>
                                </>
                              )}
                            </AutoSizer>
                          </div>
                        </> :
                        <TileList entries={eventList.map(entry => ({ ...entry, selected: entry.context === event?.entry.context }))}/>}
                    </div>
                    {showPreview && !!event && (previewPlacement === EventPreviewPlacement.Bottom || previewPlacement === EventPreviewPlacement.Right) &&
                    <div className={classNames("EventList-Preview", previewPlacement)}>
                      <EventPreview event={event} placement={previewPlacement} widgetEvent={widgetEvent} widgetIndex={widgetIndex} isTimeline={isTimeline}/>
                    </div>}
                  </>}
              </div>
            </WithQueryStatus>
          </Segment>
        </Ref> }
    </>
  );
};

type EventCellProps = React.PropsWithChildren<{
  rowData: Entry;
  className?: string;
  onClick?: () => void;
  onRowClick?: (event: EventInfo) => void
}>;

type CollectedProps = {
  dragging: boolean;
};

const EventCell = ({ rowData, className, children, onClick, onRowClick }: EventCellProps) => {
  const { userInteraction, isTimeline, widgetEvent, widgetIndex, selected } = rowData;
  const { setStore } = useStore();
  const dragSpec: DragSourceHookSpec<DragObject, {}, CollectedProps> = {
    item: { type: DragObjectType.Event, obj: getEventInfo(rowData) },
    collect: monitor => ({
      dragging: monitor.isDragging(),
    }),
    canDrag: monitor => userInteraction && !isTimeline
  };
  const [{ dragging }, dragRef] = useDrag<DragObject, {}, CollectedProps>(dragSpec);

  function getEventInfo(entry: Entry): EventInfo {
    const snapshots = isWithSnapshots(entry) ? entry.snapshots : [];
    return {
      entry: { ...entry, snapshots },
      hasSnapshots: entry.showSnapshots || snapshots.length > 0,
    };
  }

  function setEvent(entry: Entry): void {
    const event = getEventInfo(entry);
    if (onRowClick) {
      onRowClick(event);
      return;
    }
    if (!isTimeline && userInteraction) {
      setStore({ workspace: { event } });
    }
    const args: EventClickEventArgs = {
      widgetIndex,
      deviceId: entry.witnesses.length > 0 ? entry.witnesses[0].id : "",
      deviceName: entry.witnesses.length > 0 ? entry.witnesses[0].name : "",
      event
    };
    widgetEvent?.publish({ event: CommonWidgetEvent.EventClick, args });
  }

  return (
    <div ref={dragRef}
      className={classNames("EventList-CellContent",
        { "EventList-CellContent_selected": !!selected },
        { "EventList-CellContent_dragging": dragging },
        { [className ?? ""]: !!className })}
      onClick={() => {
        setEvent(rowData);
        onClick?.();
      }}>
      {children}
    </div>
  );
};

/* eslint-disable react/jsx-indent */
const SnapshotCell = ({ rowData }: { rowData: Entry}) => {
  const popupRef = useRef<HTMLDivElement>(null);
  const snapshot = useMemo(() => getSnapshot(), [rowData]);
  const { userInteraction, isTimeline } = rowData;

  function getSnapshot(): AuditSnapshot | undefined {
    return isWithSnapshots(rowData) ? rowData.snapshots.find(s => s.default) : undefined;
  }
  function getPopupOffset(): [number, number?] {
    if (!popupRef.current) return [0];
    const snapshotItemHeight = popupRef.current.clientHeight;
    const snapshotItemCenter = snapshotItemHeight / 2;
    return [snapshotItemCenter, 0];
  }

  return (
    <EventCell rowData={rowData} className="EventList-ImageContent">
      { snapshot ?
      <Ref innerRef={popupRef}>
        <Popup
          basic
          flowing
          className="EventList-SnapshotPopup"
          position="left center"
          offset={() => getPopupOffset()}
          trigger={<img src={`data:image;base64, ${snapshot.snapshot}`} draggable={userInteraction && !isTimeline ? "true" : "false"} alt=""/>}
        >
          <img src={`data:image;base64, ${snapshot.snapshot}`} draggable="false" alt=""/>
        </Popup>

      </Ref> :
        <FontAwesomeIcon icon={["far", "sticky-note"]} size="4x"/> }
      <AcknowledgeStateIcons rowData={rowData} />
    </EventCell>
  );
};
/* eslint-enable react/jsx-indent */

const InfoCell = ({ rowData, onCellClick, onRowClick }: { rowData: Entry, onCellClick?: () => void, onRowClick?: (event: EventInfo) => void }) => {
  const { triggeredAt, witnesses } = rowData;
  const alert = isAlert(rowData);
  const className = alert ? "alert" : "info";
  const currentLanguage = localStorage.getItem("language");
  return (
    <EventCell rowData={rowData} onClick={onCellClick} onRowClick={onRowClick}>
      <div>{witnesses.reduce((acc, { name }) => acc + (acc ? ", " : "") + name, "")}</div>
      <small>{formatDateTime(new Date(triggeredAt), currentLanguage as string)}</small>
      <span className={`EventList-EventType ${className}`}>
        <Icon name={alert ? "warning sign" : "info circle"} />
        {alert ? "Alert" : "Info"}
      </span>
    </EventCell>
  );
};

const IdCell = ({ rowData, onCellClick, onRowClick }: { rowData: Entry, onCellClick?: () => void, onRowClick?: (event: EventInfo) => void }) => {
  const { context } = rowData;
  const shortId = Utils.shortUuid(context, false);
  return (
    <EventCell rowData={rowData} onClick={onCellClick} onRowClick={onRowClick} >
      <div>{shortId}</div>
    </EventCell>
  );
};

const MessageCell = ({ rowData, onClick, onCellClick, onRowClick, loading, disabled, withPaginate = false }: { rowData: Entry, onCellClick?: () => void, onClick?: () => void, onRowClick?: (event: EventInfo) => void, loading?: boolean, disabled?: boolean, withPaginate?: boolean}) => {
  const messageDivRef = useRef<HTMLDivElement>(null);
  const messageInnerRef = useRef<HTMLDivElement>(null);
  const { message, note } = rowData;

  if (message === "paginate") {
    return (
      withPaginate || loading ?
        <Button
          className="EventList-PaginateButton"
          onClick={() => onClick && onClick()}
          loading={loading}
          disabled={disabled}
        >
          <Icon name="list ol" />
          {__("Load more")}
        </Button> :
        <></>
    );
  }

  function getMessagePopupDisabled(): boolean {
    const msgDiv = messageDivRef.current;
    const innerDiv = messageInnerRef.current;
    if (!msgDiv || !innerDiv) {
      return true;
    }
    const outerRect = msgDiv.getBoundingClientRect();
    const innerRect = innerDiv.getBoundingClientRect();
    return innerRect.height <= outerRect.height;
  }

  return (
    <EventCell rowData={rowData} onClick={onCellClick} onRowClick={onRowClick}>
      <Popup
        disabled={getMessagePopupDisabled()}
        trigger={
          <div className="EventList-CellContent-Message" ref={messageDivRef}>
            <div ref={messageInnerRef}>
              {message}
              {!!note &&
                <>
                  <br/>{__("User Notes:")}<br/>
                  {note}
                </>}
            </div>
          </div>}
        content={
          <>
            <b>{__("Message:")}</b><br/>
            {message}
            {!!note &&
              <>
                <br/><br/><b>{__("User Notes:")}</b><br/>
                {note}
              </>}
          </>}
      />
    </EventCell>
  );
};

/* eslint-disable react/jsx-indent */
const TileItem = React.memo(({ rowData, onCellClick }: { rowData: Entry, onCellClick?: () => void }) => {
  const [messagePopupDisabled, setMessagePopupDisabled] = useState(false);
  const messageDivRef = useRef<HTMLDivElement>(null);
  const messageInnerRef = useRef<HTMLDivElement>(null);
  const snapshot = useMemo(() => getSnapshot(), [rowData]);
  const currentLanguage = localStorage.getItem("language");
  const alert = isAlert(rowData);
  const className = alert ? "alert" : "info";

  const { triggeredAt, witnesses, message, note, showSnapshots } = rowData;

  useEffect(() => {
    setMessagePopupDisabled(getMessagePopupDisabled());
  }, [rowData]);

  function getSnapshot(): AuditSnapshot | undefined {
    return isWithSnapshots(rowData) ? rowData.snapshots.find(s => s.default) : undefined;
  }

  function getMessagePopupDisabled(): boolean {
    const msgDiv = messageDivRef.current;
    const innerDiv = messageInnerRef.current;
    if (!msgDiv || !innerDiv) {
      return true;
    }
    return message.length <= 35;
  }

  return (
    <EventCell rowData={rowData} className="EventList-TileItem" onClick={onCellClick}>
      {showSnapshots &&
      <div className="EventList-TileImage">
        {snapshot ?
          <Popup flowing trigger={
            <img src={`data:image;base64, ${snapshot.snapshot}`} draggable={rowData.userInteraction ? "true" : "false"} alt=""/>}>
            <img src={`data:image;base64, ${snapshot.snapshot}`} draggable="false" alt=""/>
          </Popup> :
          <FontAwesomeIcon icon={["far", "sticky-note"]} size="6x"/>}
      </div>}

      <div className="EventList-TileDetails">
        <div className="EventList-TileText">{witnesses.reduce((acc, { name }) => acc + (acc ? ", " : "") + name, "")}</div>
        <small>{formatDateTime(new Date(triggeredAt), currentLanguage as string)}</small>
        <span className={`EventList-EventType ${className}`}>
        <Icon name={alert ? "warning sign" : "info circle"} />
          {alert ? "Alert" : "Info"}
        </span>
        <Popup
          disabled={messagePopupDisabled}
          trigger={
            <div className="EventList-TileMessage" ref={messageDivRef}>
              <div ref={messageInnerRef} className="EventList-TileText">
                {message}
                {!!note &&
                <>
                  <br/>{__("User Notes:")}<br/>
                  {note}
                </>}
              </div>
            </div>}
          content={
            <>
              <b>{__("Message:")}</b><br/>
              {message}
              {!!note &&
              <>
                <br/><br/><b>{__("User Notes:")}</b><br/>
                {note}
              </>}
            </>}
          />
      </div>
    </EventCell>
  );
});
/* eslint-enable react/jsx-indent */

const TileList = React.memo(({ entries }: { entries: Entry[] }) => {
  return (
    <div className="EventList-Tiles">
      {entries.map(entry => <TileItem key={entry.context} rowData={entry}/>)}
    </div>
  );
});

const AcknowledgeStateIcons = React.memo(({ rowData }: { rowData: Entry }) => {
  const external = isExternal(rowData);
  const acknowledged = isAcknowledged(rowData);
  const isAcknowledgeRequired = isAckRequired(rowData);
  const isAcknowledgeConfirmed = isConfirmed(rowData);

  const externalConfirmed = isAcknowledgeRequired ? isAcknowledgeConfirmed : acknowledged;
  const confirmed = external ? externalConfirmed : acknowledged;
  const needAcknowledge = isAcknowledgeRequired && !acknowledged;
  const isNeedContent = acknowledged || confirmed || needAcknowledge;

  return (
    <Popup
      className="EventList-AcknowledgeStatePopup"
      disabled={!isNeedContent}
      trigger={
        <div className="EventList-AcknowledgeState">
          { (acknowledged || confirmed) &&
            <Icon name="check" color="green" /> }
          { confirmed &&
            <Icon name="check" color="green" /> }
          { needAcknowledge &&
            <Icon name="exclamation circle" color="red" /> }
        </div>
      }
      content={
        <>
          { acknowledged && !confirmed && __("Acknowledge processing") }
          { confirmed && __("Acknowledged") }
          { needAcknowledge && __("Need acknowledge") }
        </>
      }
    />
  );
});

function getSlicedEntries(records: Entry[], listLength: number): Entry[] {
  const slicedRecords = records.slice(0, listLength);
  const newRecords = Array.from(slicedRecords);
  return newRecords;
}

export default withStore(EventList);
