
import React from "react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { Popup } from "semantic-ui-react";
import classNames from "classnames";

import { AuditEntry, HealthStatus, RealmObject } from "@generated/graphql";
import { DeviceWitness, getListItemIcon } from "@core/actions";

import ListText, { ListItem } from "components/Admin/Helpers/ListText";

import "./style.css";

type EventInfoSnapshotsProps = {
  entry:  AuditEntry;
  devData?: DeviceWitness[];
};

const EventInfoSnapshots = React.memo(({ entry, devData }: EventInfoSnapshotsProps) => {

  function getWitnessItem(obj: RealmObject): { items: [ListItem], icons: boolean} {
    const dev = devData?.find(dev => dev.id === obj.id);
    if (!dev || !dev?.deviceType) {
      return { items: [{ ...obj }], icons: false };
    }
    const disabled = !dev.enabled || dev.healthStatus !== HealthStatus.Normal;
    const icon = getListItemIcon(dev.deviceType, disabled);

    return { items: [{ ...dev, faIcon: icon }], icons: true };
  }

  function getSnapshotItem(witnessId: string): React.ReactNode {
    let img = <FontAwesomeIcon icon={["far", "sticky-note"]} size="10x"/>;
    for (const snapshot of entry.snapshots) {
      if (snapshot.tags.find(tag => tag.key === "objid")?.value === witnessId) {
        img =
          <Popup key={`${witnessId + entry.context}`} flowing trigger={
            <img src={`data:image;base64, ${snapshot.snapshot}`} className="EventInfo-Snapshot" alt="" />}>
            <img src={`data:image;base64, ${snapshot.snapshot}`} alt="" />
          </Popup>;
      }
    }

    return img;
  }

  return (
    <div className={classNames("EventInfoSnapshots-Wrapper", { "withSnapshots": entry.snapshots.length > 0 })}>
      { entry.witnesses.map(witness =>
        <div key={witness.id} className="EventInfoSnapshots-Item">
          <ListText {...getWitnessItem(witness)} />
          { entry.snapshots.length > 0 && getSnapshotItem(witness.id) }
        </div>
      )}
    </div>
  );
});

export default EventInfoSnapshots;
