import  React, { useState } from "react";
import { Form, Segment } from "semantic-ui-react";

import { __ } from "@solid/libs";
import { UUID } from "@solid/types";

import MultipleDropdown, { MixedValue, MultipleOption } from "components/MultipleDropdown";
import Period, { PeriodValue } from "components/Period";
import { defaultPeriodMaxValue, defaultPeriodMinValue, defaultPeriodValue } from "components/Events";

import "./style.css";

export type EventFilterState = {
  categories?: MixedValue[];
  cameras?: UUID[];
  avatars?: UUID[];
  sensors?: UUID[];
  users?: UUID[];
  period?: PeriodValue;
  text?: string;
};

export type EventFilterSet = {
  [key in EventFilterKeys]?: boolean;
};

type EventFilterKeys = "categories" | "avatars" | "cameras" | "sensors" | "users";
type EventFilterProps = {
  categories?: MultipleOption[];
  cameras?: MultipleOption[];
  avatars?: MultipleOption[];
  users?: MultipleOption[];
  sensors?: MultipleOption[];
  disabled?: boolean;
  filterSet: EventFilterSet;
  onChange: (filter: keyof EventFilterState, value: EventFilterState[keyof EventFilterState]) => void;
};

const EventFilter = ({
  categories,
  users,
  cameras,
  avatars,
  sensors,
  filterSet,
  disabled = false,
  onChange
}: EventFilterProps) => {
  const [text, setText] = useState<string>("");

  return (
    <Segment className="EventFilter" disabled={disabled}>
      { filterSet.categories &&
        <MultipleDropdown
          label={__("Categories")}
          placeholder={__("Select categories")}
          withAllOption
          groups={categories ?? []}
          onChange={(values) => onChange("categories", values)}
          grouped
      /> }
      { filterSet.users &&
        <MultipleDropdown
          label={__("Users")}
          placeholder={__("Select users")}
          withAllOption
          groups={users || []}
          onChange={(values) => onChange("users", values)}
        /> }
      { filterSet.cameras &&
        <MultipleDropdown
          label={__("Cameras")}
          placeholder={__("Select cameras")}
          withAllOption
          groups={cameras ?? []}
          onChange={(values) => onChange("cameras", values)}
        /> }
      { filterSet.sensors &&
        <MultipleDropdown
          label={__("Sensors")}
          placeholder={__("Select sensors")}
          withAllOption
          groups={sensors || []}
          onChange={(values) => onChange("sensors", values)}
        /> }
      { filterSet.avatars &&
        <MultipleDropdown
          label={__("Avatars")}
          placeholder={__("Select avatars")}
          withAllOption
          groups={avatars || []}
          onChange={(values) => onChange("avatars", values)}
        /> }

      <Period
        maxTime={defaultPeriodMaxValue}
        minTime={defaultPeriodMinValue}
        endTo={defaultPeriodValue.endTo}
        startFrom={defaultPeriodValue.startFrom}
        onChange={(period) => onChange("period", period)}
        />
      <Form.Input
        className="EventFilter-Input"
        label={__("Filter by text")}
        value={text || ""}
        onChange={(e, {value}) => {
          e.preventDefault();
          onChange("text", value);
          setText(value);
        }}
      />
    </Segment>
  );
};

export default EventFilter;
