import React, { useEffect, useRef, useState } from "react";
import { Icon, Popup } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { useReactiveVar } from "@apollo/client";

import { __ } from "@solid/libs";
import { isError, useAuditEntries, isAcknowledged, isAckRequired } from "@core/actions";
import { DeviceFunctionalAspectType, useDeviceListByAspectTypesQuery, useRequireAcknowledgeEntriesLazyQuery } from "@generated/graphql";

import EventAcknowledgment from "components/EventAcknowledgment";
import { StatusBarOpenComponentId } from "components/StatusBar";

import "./style.css";

const queryTypes = [
  { type: DeviceFunctionalAspectType.Media },
  { type: DeviceFunctionalAspectType.Sensor },
];

const now = Date.now();
const entriesFilter = {
  limit: 100,
  from: new Date(now - 24 * 60 * 60 * 1000),
  to: new Date(),
  start: 0,
};

const EATrigger = () => {
  const { subscribe, unsubscribe, subscription: { data } } = useAuditEntries();
  const { data: devData, loading: devLoading, error: devError } = useDeviceListByAspectTypesQuery({ variables: { types: queryTypes } });
  const [getEntries, { data: entriesData, loading: entriesLoading, error: entriesError }] = useRequireAcknowledgeEntriesLazyQuery({ fetchPolicy: "network-only" });
  const [popupOpen, setPopupOpen] = useState(false);
  const [isAcknowledgeRequireEvent, setIsAcknowledgeRequireEvent] = useState<boolean>(false);
  const [eventAcknowledgmentPopupOpen, setEventAcknowledgmentPopupOpen] = useState(false);

  const openComponentId = useReactiveVar(StatusBarOpenComponentId);
  const popupRef = useRef(null);
  const needAcknowledgeMessage = __("You have events needs acknowledge");
  const triggerId = "EATrigger";

  useEffect(() => {
    subscribe(triggerId);

    return () => {
      unsubscribe(triggerId);
    };
  }, []);

  useEffect(() => {
    if (openComponentId && openComponentId !== triggerId) {
      setPopupOpen(false);
      setIsAcknowledgeRequireEvent(false);
      setEventAcknowledgmentPopupOpen(false);
    }
  }, [openComponentId]);

  useEffect(() => {
    if (devData?.devicesByAspectTypes && !devLoading && !devError && !entriesData?.requireAcknowledgeEntries) {
      getEntries({
        variables: {
          filter: {
            ...entriesFilter,
            witnesses: devData.devicesByAspectTypes.map(dev => dev.id),
            categories: ["47", "61"],
          }
        }
      });
    }
  }, [devData, devLoading, devError]);

  useEffect(() => {
    if (entriesData?.requireAcknowledgeEntries && !entriesError && !entriesLoading) {
      const needAcknowledge = entriesData.requireAcknowledgeEntries.some(entry => isAckRequired(entry) && !isAcknowledged(entry));
      setIsAcknowledgeRequireEvent(needAcknowledge);
      setEventAcknowledgmentPopupOpen(needAcknowledge);
    }
  }, [entriesData, entriesLoading, entriesError]);

  useEffect(() => {
    if (data) {
      const entry = !isError(data.auditEntries) ? data.auditEntries : undefined;
      if (!entry) return;
      const needAcknowledge = isAckRequired(entry) && !isAcknowledged(entry);
      needAcknowledge && !isAcknowledgeRequireEvent && setIsAcknowledgeRequireEvent(needAcknowledge);
      needAcknowledge && !popupOpen && setEventAcknowledgmentPopupOpen(needAcknowledge);
    }
  }, [data]);

  function onHandleChange() {
    const openComponentId = popupOpen ? undefined : triggerId;
    StatusBarOpenComponentId(openComponentId);
    setPopupOpen(!popupOpen);
    setIsAcknowledgeRequireEvent(false);
    setEventAcknowledgmentPopupOpen(false);
  }

  return (
    <>
      <Popup
        className="EATrigger-Popup"
        inverted
        flowing
        open={popupOpen}
        position="top right"
        content={<EventAcknowledgment onWindowClose={onHandleChange}/>}
        key="EATrigger"
        context={popupRef}
      />

      { isAcknowledgeRequireEvent &&
      <Popup
        className="NotAcknowledgedEventsPopup"
        flowing
        position="top right"
        open={eventAcknowledgmentPopupOpen}
        context={popupRef}
        content={
          <div className="NotAcknowledgedEventsPopup-Notification">
            {needAcknowledgeMessage}
            <Icon name="close" className="CloseNotAcknowledgedEventsPopup-Icon" onClick={() => setEventAcknowledgmentPopupOpen(false)}/>
          </div>
        }
      /> }

      <Popup
        flowing
        position="top right"
        disabled={eventAcknowledgmentPopupOpen || popupOpen}
        content={isAcknowledgeRequireEvent ? needAcknowledgeMessage : __("Event Acknowledgment")}
        trigger={
          <div className="EATrigger" ref={popupRef} onClick={onHandleChange} >
            <FontAwesomeIcon icon="list-ol" />
            {isAcknowledgeRequireEvent && <div className="EATrigger-Label" />}
          </div>
        }
      />
    </>
  );
};

export default EATrigger;
