import React, {useState, useEffect} from "react";
import { Header, List, Button, Dropdown, Icon } from "semantic-ui-react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faApple } from "@fortawesome/free-brands-svg-icons";
import { useSystemParametersQuery } from "@generated/graphql";
import {__} from "@solid/libs/i18n";

import "./style.css";

type SelectOptions = {
  value: string;
  text: string
};

type App = {
  id: string;
  downloadUrl: string;
  name: string
};

type DownloadInstallerProps = {
  appId: string;
};

const DownloadInstaller = ({appId}: DownloadInstallerProps) => {
  const { data } = useSystemParametersQuery();
  const [links, setLinks] = useState<App[]>([]);
  const [currentApp, setCurrentApp] = useState<App[]>([]);
  const [currentMacApp, setCurrentMacApp] = useState<App[]>([]);
  const [options, setOptions] = useState<SelectOptions[]>([]);

  const download = (downloadUrl: string, fileName: string) => {
    const a = document.createElement("a");
    a.href = downloadUrl;
    a.setAttribute("download", fileName);
    a.click();
  };

  const getCurrentApp = (appId: string) => {
    if (appId) {
      const currentApp = links.filter(link => link.id === appId);
      setCurrentApp(currentApp);
    }
  };

  useEffect(() => {
    //getCurrentApp(appId);
  }, []);

  useEffect(() => {
    if (!data?.systemParameters) return;

    const systemParameters = data.systemParameters;

    const UPDATES_HOST = systemParameters.updatesHost;
    const OVERCAST_EXT_HTTP_URL = location.hostname;
    const SOLID_APP_VERSION = systemParameters.solidAppVersion;
    // const manifestUrl = `http://${UPDATES_HOST}/windows/solid/manifest.json`;
    // const manifestResponse = await fetch(getOrigin() + "/api/call/getInstallerInfo");
    // const manifest: {version: number, fileName: string} = await manifestResponse.json();
    const fileNameMac = `Solid-${OVERCAST_EXT_HTTP_URL}-${SOLID_APP_VERSION}.dmg`;
    const manifestMac: { version: number, fileNameMac: string } = { version: 2, fileNameMac };
    const downloadUrlMac = `https://${UPDATES_HOST}/darwin/solid/${OVERCAST_EXT_HTTP_URL}/${manifestMac.fileNameMac}`;

    const appMac: App = {
      id: "SOLID",
      downloadUrl: downloadUrlMac,
      name: "Solid",
    };

    setCurrentMacApp([appMac]);

    const fileName = `Solid-${OVERCAST_EXT_HTTP_URL}-${SOLID_APP_VERSION}.exe`;
    const manifest: {version: number, fileName: string} = {version: 2, fileName};
    const downloadUrl = `https://${UPDATES_HOST}/windows/solid/${OVERCAST_EXT_HTTP_URL}/${manifest.fileName}`;

    const app: App = {
      id: "SOLID",
      downloadUrl,
      name: "Solid"
    };
    setLinks([app]);
    setCurrentApp([app]);
  }, [data]);

  useEffect(() => {
    const options = links.map(link => ({
      key: link.id,
      value: link.id,
      text: link.name
    })
    );
    setOptions(options);
  }, [links]);

  return (
    <>
      { currentApp && currentApp.map(currentApp =>
        <div className="DownloadInstaller-Root" key={currentApp.id}>
          <div className="DownloadInstaller-Header">
            <Header size="huge">{__("App Downloads")}</Header>
            <Dropdown
              selection
              options={options}
              value={currentApp.name}
              placeholder={currentApp.name}
              onChange={(e, data: { value?: any }) => getCurrentApp(String(data.value))}/>
          </div>
          <div className="DownloadInstaller-Links">
            <List>
              <div className="DownloadInstaller-Link">
                <div className="DownloadInstaller-List-Item">
                  <FontAwesomeIcon icon={["fab", "windows"]} size="6x"/>
                </div>
                <div className="DownloadInstaller-List-Item">
                  <Header size="small">{__("Windows 10")}</Header>
                </div>
                <div className="DownloadInstaller-List-Item">
                  <Button primary onClick={() => download(`${currentApp.downloadUrl}`, `${currentApp.name}.exe`)}>
                    <Icon name="download" />
                    {__("Download")}
                  </Button>
                </div>
              </div>
              {false && currentMacApp && currentMacApp.filter(app => app.name === currentApp.name).map(currentApp => (
                <div className="DownloadInstaller-Link">
                  <div className="DownloadInstaller-List-Item">
                    <FontAwesomeIcon icon={faApple} size="6x" />
                  </div>
                  <div className="DownloadInstaller-List-Item">
                    <Header size="small">{__("MacOS")}</Header>
                  </div>
                  <div className="DownloadInstaller-List-Item">
                    <Button primary onClick={() => {
                      download(`${currentApp.downloadUrl}`, `${currentApp.name}.dmg`);
                    }}>
                      <Icon name="download" />
                      {__("Download")}
                    </Button>
                  </div>
                </div>
              ))}
            </List>
          </div>
        </div>
      )}
    </>
  );
};

export default DownloadInstaller;
