import React, { useState, useEffect, useRef } from "react";
import { useDrop } from "react-dnd";
import classNames from "classnames";
import { Segment, Message } from "semantic-ui-react";
import { makeVar, useReactiveVar } from "@apollo/client";

import { DeviceFunctionalAspectType, AuditEntry, useDeviceListByAspectTypesQuery, useAuditEntriesLazyQuery} from "@generated/graphql";
import { DragObjectType, DragObject, isEvent, PlayerProps } from "@core/types";
import { withStore, WithStoreProps } from "@core/store";
import { MetadataDrawType, PlayerStartFrom, PlayerOnClipEnd } from "@core/playerTypes";
import { AuditEntriesAutoUpdateWidgets, isError, useAuditEntries } from "@core/actions";
import { Utils } from "@solid/libs/utils";
import {__} from "@solid/libs/i18n";
import { UUID } from "@solid/types";

import { WidgetProps, CellProps } from "components/Widgets";
import CellContent from "components/CellContainer/CellContent";

import "../style.css";

export const currentPlayingEvent = makeVar<AuditEntry | undefined>(undefined);

type EventVideoProps = WidgetProps & WithStoreProps & {
  startFrom?: PlayerStartFrom;
  metadataType?: MetadataDrawType;
  onClipEnd?: PlayerOnClipEnd;
  autoUpdate?: boolean;
};

const throttlingTime = 7000;

const EventVideo = (props: EventVideoProps) => {
  const { cellCloseEventPubSub, cellProps, setCellProps, isViewActive,
    startFrom = PlayerStartFrom.Event, metadataType, onClipEnd = PlayerOnClipEnd.Clear, autoUpdate = true,
    store: { workspace: { event } }, setStore } = props;

  const [getAuditEntries, { data: evData }] = useAuditEntriesLazyQuery({ fetchPolicy: "no-cache" });
  const { data: devData, loading: devLoading, error: devError } = useDeviceListByAspectTypesQuery({
    variables: { types: [{ type: DeviceFunctionalAspectType.Media }] }
  });

  const playingEvent = useReactiveVar(currentPlayingEvent);
  const { subscribe, unsubscribe, subscription: { data: subData, loading: subLoading, error: subError } } = useAuditEntries();

  const [object, setObject] = useState<PlayerProps | undefined>();
  const [playedEvents, setPlayedEvents] = useState<Set<UUID>>(new Set());
  const [subErrMsg, setSubErrMsg] = useState("");

  const widgetId = `${props.viewId}_widget${props.index}`;
  const autoUpdateOnEvent = useReactiveVar(AuditEntriesAutoUpdateWidgets);
  const subscribeToEvents = isViewActive && autoUpdate && autoUpdateOnEvent;
  const [skipSubscriptionEvents, setSkipSubscriptionEvents] = useState<boolean>(!subscribeToEvents);
  const onEnd = startFrom === PlayerStartFrom.Event ? onClipEnd : undefined;

  const [{ dragOver }, dropRef] = useDrop<DragObject, {}, { dragOver: boolean }>({
    accept: [DragObjectType.Event],
    drop: (item, monitor) => {
      if (isEvent(item.obj)) {
        const newObj = getObject(item.obj.entry);
        if (newObj) {
          setObject(newObj);
          unsubscribe(widgetId);
        }
      }
      return undefined;
    },
    canDrop: (item, monitor) => !!devData?.devicesByAspectTypes.some(d => isEvent(item.obj) && item.obj.entry.witnesses.some(w => w.id === d.id)),
    collect: monitor => ({
      dragOver: monitor.isOver()
    }),
  });

  const setLiveObjectRef = useRef(Utils.throttle((newObj: PlayerProps, newEvent: AuditEntry, playingEvent: AuditEntry, playedEvents: UUID[], skip: boolean = false) => {
    if (newEvent.context !== playingEvent?.context && !playedEvents.includes(newEvent.context) && !skip) {
      setObject(newObj);
      currentPlayingEvent(newEvent);
      setPlayedEvents(prevSet => {
        const newSet = new Set(prevSet);
        newSet.add(newEvent.context);
        return newSet;
      });
    }
  }, throttlingTime));
  const isMountedRef = useRef(false);

  useEffect(() => {
    return function cleanup() {
      subscribeToEvents && unsubscribe(widgetId);
      subscribeToEvents && currentPlayingEvent(undefined);
      setStore({ workspace: { event: null } });
    };
  }, []);

  useEffect(() => {
    isMountedRef.current = true;

    const id = `${Date.now()}.${Math.random()}`;
    const closeEvent = cellCloseEventPubSub;
    if (closeEvent) {
      closeEvent.subscribe(id, onCellClose);
    }

    return function cleanup() {
      isMountedRef.current = false;

      if (closeEvent) {
        closeEvent.unsubscribe(id);
      }
    };
  });

  useEffect(() => {
    if (!devData?.devicesByAspectTypes) return;

    subscribeToEvents && getAuditEntries({
      variables: {
        filter: {
          from: new Date(Date.now() - 60 * 60 * 1000),
          to: new Date(),
          witnesses: devData.devicesByAspectTypes.map(dev => dev.id),
          categories: ["47", "61"],
          start: 0,
          limit: 50
        }
      }
    });
  }, [isViewActive, autoUpdate, devData, autoUpdateOnEvent]);

  useEffect(() => {
    if (!evData?.auditEntries) return;
    setPlayedEvents(prevSet => {
      const playedEvents = [...prevSet, ...evData.auditEntries.map(ev => ev.context)];
      return new Set(playedEvents);
    });
  }, [evData]);

  useEffect(() => {
    if (!isMountedRef.current || !event) {
      return;
    }

    const entry = event.entry;
    const newObj = getObject(entry);
    if (newObj && (playingEvent?.context !== entry.context)) {
      currentPlayingEvent(entry);
      unsubscribe(widgetId);
      setObject(newObj);
    }
  }, [event]);

  useEffect(() => {
    setSkipSubscriptionEvents(!subscribeToEvents);
  }, [autoUpdateOnEvent]);

  useEffect(() => {
    if (!isMountedRef.current) {
      return;
    }

    if (setCellProps) {
      const hasObject = !!object;
      const props: CellProps = {
        title: !hasObject ? cellProps?.title : undefined,
        closable: hasObject
      };
      setCellProps(props);
    }
  }, [object]);

  useEffect(() => {
    if (!isMountedRef.current) {
      return;
    }

    if (!devLoading && !devError && devData) {
      subscribeToEvents && subscribe(widgetId);
      if (event) {
        const entry = event.entry;
        const newObj = getObject(entry);
        if (newObj && (playingEvent?.context !== entry.context)) {
          setObject(newObj);
          currentPlayingEvent(entry);
        }
      }
    }
    devError && console.error("EventVideo getting devices error:", devError);
  }, [devLoading, devError, devData, autoUpdate, isViewActive, autoUpdateOnEvent]);

  useEffect(() => {
    if (!subLoading && !subError && subData && !isError(subData.auditEntries)) {
      const newObj = getObject({ ...subData.auditEntries, metadata: [] });
      if (newObj) {
        setLiveObjectRef.current(newObj, subData.auditEntries, playingEvent, [...playedEvents], !subscribeToEvents || !isMountedRef.current);
      }
    }

    if (!subscribeToEvents || !isMountedRef.current) {
      return;
    }

    if (subData && isError(subData.auditEntries)) {
      console.error("EventVideo subscription error:", subData.auditEntries.error);
      setSubErrMsg(subData.auditEntries.error);
    }
    else if (subError) {
      console.error("EventVideo subscription error:", subError);
      setSubErrMsg(subError.message);
    }
    else {
      setSubErrMsg("");
    }
  }, [subLoading, subError, subData, playingEvent, playedEvents, skipSubscriptionEvents, isViewActive, autoUpdate]);

  function getObject(event: AuditEntry): PlayerProps | undefined {
    const witness = event.witnesses.find(w => devData?.devicesByAspectTypes.some(d => d.id === w.id));
    if (!witness) {
      return undefined;
    }

    return { "data-id": witness.id, name: witness.name, time: new Date(event.triggeredAt) };
  }

  function onCellClose(): void {
    if (!isMountedRef.current) {
      return;
    }

    setObject(undefined);
  }

  function onStop(): void {
    if (onEnd === PlayerOnClipEnd.Clear) {
      setTimeout(() => {
        if (isMountedRef.current) {
          setObject(undefined);
        }
      }, throttlingTime);
    }
  }

  return (
    <div ref={dropRef} className={classNames("CellContainer-Root", { "CellContainer-Root_dragOver": dragOver })}>
      { !!devError && <Message visible error header={__("Devices query error")} content={devError.message}/> }
      { !!subErrMsg &&
        <Message visible error header={__("Subscription error")} content={`${subErrMsg} ${__("Resubscribing...")}`}/> }
      { !object ?
        <Segment className="CellContainer-Empty">
          <p>{__("Drag Camera Event here")}</p>
        </Segment> :
        <CellContent metadataType={metadataType} onClipEnd={onEnd} onStop={onStop} {...props} obj={{
          ...object,
          time: startFrom === PlayerStartFrom.Event ? object.time : undefined }} /> }
    </div>
  );
};

export default withStore(EventVideo);
