import React, { useState, useEffect, useCallback } from "react";
import { useDrop } from "react-dnd";
import classNames from "classnames";
import { Segment } from "semantic-ui-react";
import { ReactSVG } from "react-svg";
import { DragObjectType, DragObject, isObject, isDragObject, isEvent } from "@core/types";
import {
  CameraWidgetProps,
  CameraWidgetPropsInput,
  DeviceFunctionalAspectType,
  ObjectAction,
  useDeviceNameLazyQuery,
  useDeviceListByAspectTypesLazyQuery,
  WidgetId,
} from "@generated/graphql";
import { WidgetProps, CellProps, Widgets,  WidgetEventArgs } from "components/Widgets";
import { withStore, WithStoreProps } from "@core/store";
import CellContent from "components/CellContainer/CellContent";
import { queryToInput } from "utils";
import { PlayerOptions, PlayerCommand, PlayerCommandArgs } from "@core/playerTypes";
import {__} from "@solid/libs/i18n";
import { useObjectUpdate } from "@core/actions";

import "../style.css";


export type CameraCellProps = WithStoreProps & WidgetProps & PlayerOptions & {
  object?: DragObject;
  canReplaceSource?: boolean;
};

export function isCameraCellProps(props: any): props is CameraCellProps {
  return !!props && isDragObject((props as CameraCellProps).object);
}

const CameraCell = (props: CameraCellProps) => {
  const {
    viewId,
    index,
    widgetId,
    cellCloseEventPubSub,
    cellProps,
    setCellProps,
    store: { workspace: { cameraWidgetProps } },
    setStore,
    canReplaceSource = true,
    loader,
    widgetEvent,
    widgets,
    stepBackControl
  } = props;

  const item = cameraWidgetProps?.find(item => item.viewId === viewId && item.index === index);

  const [object, setObject] = useState<DragObject | undefined>(!item ? props.object : getObject(item));

  const [getDevices, { data: devData, error: devError }] = useDeviceListByAspectTypesLazyQuery();
  const { data: devSubData, error: devSubError } = useObjectUpdate();
  const [getDeviceName, { data: devNameData, error: devNameError }] = useDeviceNameLazyQuery();

  const [{ dragOver }, dropRef] = useDrop<DragObject, {}, { dragOver: boolean }>({
    accept: widgetId !== WidgetId.ArchiveViewer ? [DragObjectType.Camera] : [DragObjectType.Camera, DragObjectType.Event],

    drop: useCallback((item, monitor) => {
      let object = item;
      if (isEvent(object.obj)) {
        const obj = object.obj;
        const device = devData?.devicesByAspectTypes.find(d => isEvent(obj) && obj.entry.witnesses.some(w => w.id === d.id));
        object = {
          type: DragObjectType.Camera,
          obj: {
            "data-id": device?.id,
            name: device?.name,
            time: new Date(obj.entry.triggeredAt)
          }
        };
      }
      setObject(object);
      storeCameraWidgetProps(object);
      return undefined;
    }, [cameraWidgetProps, devData]),

    collect: monitor => ({
      dragOver: monitor.isOver()
    }),

    canDrop: useCallback(
      (item: DragObject, monitor) =>
        canReplaceSource && (
          isObject(item.obj)
          || (
            widgetId === WidgetId.ArchiveViewer
            && !!devData?.devicesByAspectTypes.some(d => isEvent(item.obj) && item.obj.entry.witnesses.some(w => w.id === d.id))
          )
        ),
      [devData])
  });

  const [outOfSync, setOutOfSync] = useState(false);

  const widget = Widgets.find(w => w.id === widgetId);

  useEffect(() => {
    if (props.object) {
      setObject(props.object);
      getDeviceName({ variables: { id: props.object.obj["data-id"] } }); // Check does camera still exist.
    }
  }, [props.object]);

  useEffect(() => {
    if (object?.obj && object.obj["data-id"]) {
      getDeviceName({ variables: { id: object.obj["data-id"] } }); // Check does camera still exist.
    }
    if (widgetId === WidgetId.ArchiveViewer) {
      getDevices({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
    }
  }, []);

  useEffect(() => {
    const id = `${Date.now()}.${Math.random()}`;
    const closeEvent = cellCloseEventPubSub;
    closeEvent?.subscribe(id, onCellClose);
    widgetEvent?.subscribe(id, onWidgetEvent);

    return function cleanup() {
      closeEvent?.unsubscribe(id);
      widgetEvent?.unsubscribe(id);
    };
  });

  useEffect(() => {
    const item = cameraWidgetProps?.find(item => item.viewId === viewId && item.index === index);
    if (item) {
      const { deviceId, time } = item;
      if (deviceId) {
        setObject(object =>
          !object || !isObject(object.obj) || object.obj["data-id"] !== deviceId || object.obj.time !== time
            ? getObject(item)
            : object
        );
      }
      else if (object) {
        setObject(undefined);
      }
    }
  }, [cameraWidgetProps]);

  useEffect(() => {
    if (setCellProps) {
      const hasObject = !!object && isObject(object.obj);
      let dragObject = cellProps?.dragObject;
      if (object && isObject(object.obj)) {
        const dragObj: DragObject = { ...object, sourceWidgetId: widgetId };
        dragObject = dragObj;
      }
      const props: CellProps = {
        title: !hasObject ? __("No camera assigned") : undefined,
        closable: hasObject && canReplaceSource,
        hideIcon: hasObject,
        dragObject
      };
      setCellProps(props);
    }
  }, [object]);

  useEffect(() => {
    devError && console.error("ArchiveViewer get devices error:", devError);
  }, [devError]);

  useEffect(() => {
    devSubError && console.error("CameraCell object action subscription error:", devSubError);

    if (!devSubData) {
      return;
    }

    const { id, action } = devSubData.objectUpdate;
    if (object?.obj && id === object.obj["data-id"]) {
      switch (action) {
        case ObjectAction.Update:
          getDeviceName({ variables: { id } });
          break;

        case ObjectAction.Delete:
          setObject(undefined);
          storeCameraWidgetProps(undefined);
          break;
      }
    }
  }, [devSubData, devSubError]);

  useEffect(() => {
    devNameError && console.error("CameraCell get device name error:", devNameError);

    if (!devNameData) {
      return;
    }

    if (devNameData.device) {
      const { id, name } = devNameData.device;
      setObject(value => {
        if (!value?.obj || !isObject(value.obj) || value.obj["data-id"] !== id || value.obj.name === name) {
          return value;
        }
        return { ...value, obj: { ...value.obj, name } };
      });
    }
    else {
      storeCameraWidgetProps(undefined);
      setObject(value => {
        if (!value?.obj || !value.obj["data-id"]) {
          return value;
        }
        return undefined; // Camera was probably deleted.
      });
    }
  }, [devNameData, devNameError]);

  function getObject(item: CameraWidgetProps): DragObject | undefined {
    const { deviceId, name, time } = item;
    return deviceId
      ? {
        type: DragObjectType.Camera,
        obj: {
          "data-id": deviceId,
          name,
          time: time ? new Date(time) : undefined
        }
      }
      : undefined;
  }

  const storeCameraWidgetProps = useCallback((object: DragObject | undefined): void => {
    if (!viewId || index === undefined) {
      return;
    }

    let cameraProps: CameraWidgetPropsInput = {
      viewId,
      index,
      deviceId: "",
      name: "",
      time: null
    };
    if (object && isObject(object.obj)) {
      const { name, time } = object.obj;
      cameraProps = {
        ...cameraProps,
        deviceId: object.obj["data-id"],
        name,
        time: time ? time.getTime() : null
      };
    }
    const newItems = Array.from(cameraWidgetProps || []);
    const i = newItems.findIndex(item => item.viewId === viewId && item.index === index);
    let change = false;
    if (i >= 0) {
      change = JSON.stringify(queryToInput<CameraWidgetProps, CameraWidgetPropsInput>(newItems[i])) !== JSON.stringify(cameraProps);
      newItems[i] = { ...newItems[i], ...cameraProps };
    }
    else {
      newItems.push(cameraProps);
      change = true;
    }
    if (change) {
      setStore({ workspace: { cameraWidgetProps: queryToInput<CameraWidgetProps[], CameraWidgetPropsInput[]>(newItems) } });
    }
  }, [cameraWidgetProps]);

  const onCellClose = useCallback((): void => {
    setObject(undefined);
    storeCameraWidgetProps(undefined);
  }, [storeCameraWidgetProps]);

  const onWidgetEvent = (args: WidgetEventArgs): void => {
    if (args.event === PlayerCommand.SetSelected) {
      const { widgetIndices } = args.args as PlayerCommandArgs;
      if (widgetIndices && widgetIndices.length > 0 && widgetIndices[0] === index && widgets && index !== undefined) {
        loader?.setSelected(widgets[index].id, true, true);
      }
    }
    if (args.event === PlayerCommand.ClearSelected && widgets) {
      for (const widget of widgets) {
        loader?.setSelected(widget.id, false);
      }
    }
    if (args.event === PlayerCommand.OutOfSync || args.event === PlayerCommand.BackToSync) {
      const { widgetIndices } = args.args as PlayerCommandArgs;
      if (widgetIndices && widgetIndices.length > 0 && widgetIndices[0] === index) {
        setOutOfSync(args.event === PlayerCommand.OutOfSync);
      }
    }
  };

  return (
    <div ref={dropRef} className={classNames("CellContainer-Root", { "CellContainer-Root_dragOver": dragOver })}>
      { !object &&
        <Segment className="CellContainer-Empty">
          { !!widget && <ReactSVG role="icon" src={widget.icon} /> }
          <p>{widget?.title}</p>
        </Segment> }
      { !!object && <CellContent {...props} {...object} outOfSync={outOfSync} stepBackControl={stepBackControl} /> }
    </div>
  );
};

export default withStore(CameraCell);
