import React, { useState, useEffect, useRef, CSSProperties } from "react";
import { useDrag } from "react-dnd";
import { Checkbox, Portal, Ref, Segment } from "semantic-ui-react";
import classNames from "classnames";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { BaseDeviceType } from "@generated/graphql";
import { getListItemIcon } from "@core/actions";
import { ObjProps, DragObjectType, DragObject } from "@core/types";
import { SelectionStyle } from "../CameraList";
import Snapshot from "components/Snapshot";
import useIsMounted from "@core/useIsMounted";

import "./style.css";

type AttributePosition = "left" | "right";

type CameraListItemProps = ObjProps & {
  disabled?: boolean;
  style?: CSSProperties;
  selectable?: boolean;
  selected?: boolean;
  selectionStyle?: SelectionStyle;
  widgetId?: string;
  parentId?: string;
  parentRef?: React.RefObject<HTMLDivElement>;
  type?: BaseDeviceType;
  onClick?: (e: React.MouseEvent<HTMLDivElement> | undefined, id: string, name: string, parentId?: string) => void;
};

function CameraListItem({
  "data-id": id,
  name,
  style,
  type = BaseDeviceType.Camera,
  disabled = false,
  selectable = false,
  selected = false,
  selectionStyle = SelectionStyle.Border,
  widgetId,
  parentId,
  parentRef,
  onClick
}: CameraListItemProps) {
  const [{ dragging }, dragRef] = useDrag<DragObject, {}, { dragging: boolean }>({
    item: { type: DragObjectType.Camera, obj: { "data-id": id, name }, sourceWidgetId: widgetId },
    collect: monitor => ({
      dragging: monitor.isDragging(),
    }),
    begin: () => hideSnapshot()
  });
  const itemRef = useRef<HTMLDivElement>(null);
  const snapshotRef = useRef<HTMLDivElement>(null);
  const [snapshotOpen, setSnapshotOpen] = useState(false);
  const [/*isMouseIn*/, setMouseIn] = useState(false);
  const isMounted = useIsMounted();
  const timeoutRef = useRef<NodeJS.Timeout | undefined>(undefined);

  useEffect(() => {
    if (snapshotOpen) {
      positionSnapshot();
    }
  }, [snapshotOpen, parentRef]);

  function click(e: React.MouseEvent<HTMLDivElement> | undefined, id: string, name: string): void {
    onClick && onClick(e, id, name, parentId);
  }

  function hideSnapshot(): void {
    setMouseIn(false);
    setSnapshotOpen(false);
  }

  function showSnapshot(): void {
    setMouseIn(true);

    timeoutRef.current = setTimeout(() => {
      if (!isMounted()) {
        return;
      }

      setMouseIn(isMouseIn => {
        if (isMouseIn) {
          setSnapshotOpen(true);
        }
        return isMouseIn;
      });
    }, 500);
  }

  function positionSnapshot(): void {
    const item = itemRef.current;
    const popup = snapshotRef.current;
    const parent = parentRef?.current;
    if (!item || !popup || !parent) {
      return;
    }
    const rootElem = document.documentElement;
    const itemRect = item.getBoundingClientRect();
    const popupRect = popup.getBoundingClientRect();
    const parentRect = parent.getBoundingClientRect();
    const right = parentRect.right;
    const left = parentRect.left;

    let attributePosition: AttributePosition = "right";
    let positionX = (rootElem.clientWidth - right) + parentRect.width;
    let positionY = itemRect.top;

    if ((rootElem.clientWidth - right) >= 300) {
      positionX = right;
      attributePosition = "left";
    }

    if (right === rootElem.clientWidth) {
      attributePosition = "right";
      positionX = parentRect.width;
    }

    if (((rootElem.clientWidth - right) < 300) && (left < 300)) {
      positionX = left;
      attributePosition = "left";
      positionY = itemRect.bottom;
      if ((itemRect.bottom + popupRect.height) > rootElem.clientHeight) {
        positionY = itemRect.top - popupRect.height;
      }
    }

    if (!(((rootElem.clientWidth - right) < 300) && (left < 300)) && (itemRect.top + popupRect.height) >= rootElem.clientHeight) {
      positionY = rootElem.clientHeight - popupRect.height;
    }

    popup.setAttribute("style", `${attributePosition}: ${positionX}px; top: ${positionY}px`);
  }

  useEffect(() => {
    return () => {
      clearTimeout(timeoutRef.current);
    };
  }, []);

  /* eslint-disable react/jsx-indent, react/jsx-curly-newline, react/jsx-curly-spacing */
  return (
    <>
      <div ref={itemRef} style={style}>
        <div
          ref={dragRef}
          className={classNames("CameraListItem-Root",
            { "CameraListItem-Root_dragging": dragging },
            { "CameraListItem-Root_selected_border": selectable && selected && selectionStyle === SelectionStyle.Border },
            { "CameraListItem-Root_selected_background": selectable && selected && selectionStyle === SelectionStyle.Background },
            { "CameraListItem-Root_selected_checkbox": selectable && selected && selectionStyle === SelectionStyle.Checkbox },
            { "CameraListItem-Root_selectable_border": selectable && selectionStyle === SelectionStyle.Border },
            { "CameraListItem-Root_selectable_background": selectable && selectionStyle === SelectionStyle.Background },
            { "CameraListItem-Root_selectable_checkbox": selectable && selectionStyle === SelectionStyle.Checkbox },
            { "CameraListItem-Root_disabled": disabled })}
          onClick={e => {
            hideSnapshot();
            if (selectionStyle !== SelectionStyle.Checkbox) {
              click(e, id, name);
            }
          }}
          onMouseEnter={showSnapshot}
          onMouseLeave={hideSnapshot}>

          <div>
            {selectable && selectionStyle === SelectionStyle.Checkbox &&
            <Checkbox checked={selected} onChange={() => click(undefined, id, name)}/>}
            <div className="CameraListItem-Content">
              <FontAwesomeIcon icon={getListItemIcon(type, disabled)}/>
              <div className="CameraListItem-Name">{name}</div>
            </div>
          </div>
        </div>
      </div>

      <Portal
        open={snapshotOpen}
        onMount={() => positionSnapshot()}>
        <Ref innerRef={snapshotRef}>
          <Segment className="CameraListItem-Snapshot">
            <div>
              <Snapshot snapshotId={id} load={snapshotOpen} />
            </div>
          </Segment>
        </Ref>
      </Portal>
    </>
  );
}

export default CameraListItem;
