import React from "react";
import { Form } from "semantic-ui-react";
import BaseField, { BaseFieldProps, BaseFieldState } from "../BaseField";
import {__} from "@solid/libs/i18n";

type FloatFieldState = BaseFieldState & {
  strValue: string;
};

class FloatField extends BaseField<BaseFieldProps, FloatFieldState> {
  constructor(props: BaseFieldProps) {
    super(props);
    this.state = { value: null, strValue: "" };
    this.onInput = this.onInput.bind(this);
  }

  override getDefaultValue(): any {
    return null;
  }

  override setValue(value: any): any {
    let val: number | null = null;
    let strValue = "";
    if (typeof value === "number" && !isNaN(value)) {
      // val = Math.trunc(value);
      strValue = value.toString();
    }
    if (typeof value === "string") {
      const float = this.parseFloat(value);
      if (!isNaN(float)) {
        val = float;
        strValue = val.toString();
      }
    }
    this.setState({ value: val, strValue });
    return val;
  }

  override getError(value: any): string | undefined {
    let error = super.getError(value);
    const { field } = this.state;
    if (field && typeof value === "number") {
      if (!error && isNaN(value)) {
        error = __("{{value}} should be a valid float number", {value: field.label});
      }
      if (!error && field.minValue !== undefined && value < field.minValue) {
        error = __("{{value}} should be greater or equal than {{number}}", {value: field.label, number: field.minValue});
      }
      if (!error && field.maxValue !== undefined && value > field.maxValue) {
        error = __("{{value}} should be less or equal than {{number}}", {value: field.label, number: field.maxValue});
      }
    }
    return error;
  }

  onInput(e: React.FormEvent<HTMLInputElement>): void {
    const strValue = e.currentTarget.value;
    const { field } = this.state;
    const allowNegative = !field || field.minValue === undefined || field.minValue < 0;
    const allowInput = allowNegative ? /^[+-]?([0-9]+([.][0-9]*)?|[.][0-9]+)$/.test(strValue) : /^[+]?([0-9]+([.][0-9]*)?|[.][0-9]+)$/.test(strValue);
    if (!allowInput) {
      return;
    }
    this.setState({ strValue });
    this.onChange(strValue);
  }

  parseFloat(value: string): number {
    if (/^[+-]?([0-9]+([.][0-9]*)?|[.][0-9]+)$/.test(value)) {
      return Number(value);
    }
    return NaN;
  }

  override render() {
    const { field, strValue, error, hidden, disabled, readOnly } = this.state;
    if (!field || hidden) {
      return null;
    }
    return (
      <Form.Field
        control={Form.Input}
        label={field.label}
        placeholder={field.label}
        required={field.required}
        value={strValue}
        error={error ? { content: error, pointing: "below" } : undefined}
        disabled={disabled}
        readOnly={readOnly}
        onInput={this.onInput}
        className={`FloatField__${field.name}`}
      />
    );
  }
}

export default FloatField;
