import React from "react";
import { Form, DropdownItemProps, FormSelect, Button, Icon } from "semantic-ui-react";
import { FieldSchema } from "../AutoForm";
import BaseField, { BaseFieldProps, BaseFieldState } from "../BaseField";
import "./style.css";

type DropdownFieldState = BaseFieldState & {
  options: DropdownItemProps[];
};

class DropdownField extends BaseField<BaseFieldProps, DropdownFieldState> {
  constructor(props: BaseFieldProps) {
    super(props);
    this.state = { value: "", options: [] };
  }

  override componentDidUpdate(prevProps: BaseFieldProps, prevState: DropdownFieldState): void {
    if (
      (prevState.value !== this.state.value || prevState.options !== this.state.options) &&
      this.state.options.length > 0 &&
      !this.state.options.some((option) => option.value === this.state.value)
    ) {
      this.onChange(this.state.options[0].value);
    }
  }

  override setFieldSchema(field?: FieldSchema): void {
    super.setFieldSchema(field);
    if (field && typeof field.dataSource === "object") {
      const items = field.dataSource;
      if (!field.required) {
        items.unshift({ id: "", name: "" });
      }
      this.setState({ options: items.map(({ id, name }) => ({ key: id, value: id, text: name })) });
    }
  }

  override updateState(fieldSchema?: FieldSchema): FieldSchema | undefined {
    const field = super.updateState(fieldSchema);
    if (field && typeof field.dataSource === "function") {
      const items = field.dataSource(this.getValues());
      if (!field.required) {
        items.unshift({ id: "", name: "" });
      }
      const options = items.map(({ id, name }) => ({ key: id, value: id, text: name }));
      this.setState((state) => {
        if (JSON.stringify(state.options) === JSON.stringify(options)) {
          return null;
        }
        return { options };
      });
    }
    return field;
  }

  override compareValues(a: any, b: any): boolean {
    return a === b || (!a && !b);
  }

  override render() {
    const { field, value, error, hidden, disabled, readOnly, options } = this.state;
    if (!field || hidden) {
      return null;
    }
    if (field.action)
      return (
        <Form.Field
          required={field.required}
          error={error ? { content: error, pointing: "below" } : undefined}
          disabled={disabled}
          className={`DropdownFieldAction__${field.name}`}
        >
          <>
            <label>{field.label}</label>
            <FormSelect
              options={options}
              value={value}
              onChange={(e, { value }) => this.onChange(value)}
              search
              readOnly={readOnly}
              className="DropdownFieldAction-wrap"
              label={
                <Button icon onClick={() => field.action && field.action()}>
                  <Icon name="refresh" />
                </Button>
              }
            />
          </>
        </Form.Field>
      );
    return (
      <Form.Select
        label={field.label}
        required={field.required}
        options={options}
        value={value}
        error={error ? { content: error, pointing: "below" } : undefined}
        disabled={disabled}
        readOnly={readOnly}
        search
        onChange={(e, { value }) => this.onChange(value)}
        className={`DropdownField__${field.name}`}
      />
    );
  }
}

export default DropdownField;
