import React from "react";
import AutoForm, { FieldSchema, AutoFormContext, FieldValues } from "../AutoForm";
import {__} from "@solid/libs/i18n";

export type BaseFieldProps = {
  name: string;
};

export type BaseFieldState = {
  field?: FieldSchema;
  value: any;
  error?: string;
  hidden?: boolean;
  disabled?: boolean;
  readOnly?: boolean;
};

class BaseField<TProps extends BaseFieldProps = BaseFieldProps, TState extends BaseFieldState = BaseFieldState> extends React.Component<TProps, TState> {
  static override contextType = AutoFormContext;
  protected form?: AutoForm;
  protected isValid?: boolean;

  override componentDidMount(): void {
    this.form = this.context?.form;
    this.setState({ value: this.getDefaultValue() });
    this.form?.registerField(this.props.name, this);
  }

  override componentWillUnmount(): void {
    this.form?.unregisterField(this.props.name);
  }

  setFieldSchema(field?: FieldSchema): void {
    this.setState({ field });
    this.updateState(field);
  }

  setValue(value: any): any {
    const val = value ?? this.getDefaultValue();
    this.setState({ value: val });
    return val;
  }

  getDefaultValue(): any {
    return "";
  }

  getError(value: any): string | undefined {
    if (this.state.field?.required) {
      const val = typeof value === "string" ? value.trim() : value;
      if (val === undefined || val === null || val === "") {
        return __("{{name}} should be not empty", {name: this.state.field.label});
      }
    }
    return undefined;
  }

  validateValue(value: any): boolean {
    const { field, hidden } = this.state;
    const prevValid = this.isValid;
    try {
      if (hidden) {
        /* eslint-disable react/no-unused-state */
        this.setState({ error: undefined });
        /* eslint-enable react/no-unused-state */
        this.isValid = true;
        return true;
      }
      let error = this.getError(value);
      if (!error && field && field.validate) {
        error = field.validate(field!, value, this.getValues());
      }
      /* eslint-disable react/no-unused-state */
      this.setState({ error });
      /* eslint-enable react/no-unused-state */
      this.isValid = !error;
      return !error;
    }
    finally {
      if (prevValid !== this.isValid && this.isValid !== undefined) {
        field && this.form?.onFieldValidChange(field!, this.isValid);
      }
    }
  }

  validate(): boolean {
    return this.validateValue(this.state.value);
  }

  getIsValid(): boolean | undefined {
    return this.isValid;
  }

  onChange(rawValue: any): void {
    this.setState({ value: rawValue });
    const value = typeof rawValue === "string" ? rawValue.trim() : rawValue;
    this.validateValue(value);
    const { field } = this.state;
    field && this.form?.onChange(field!, value);
  }

  protected getValues(): FieldValues {
    return this.form?.getValues() ?? {};
  }

  protected isHidden(field: FieldSchema): boolean {
    return !!field.hideCondition && field.hideCondition(this.getValues());
  }

  protected isDisabled(field: FieldSchema): boolean {
    return !!field.disableCondition && field.disableCondition(this.getValues());
  }

  protected isReadOnly(field: FieldSchema): boolean {
    return !!field.readOnlyCondition && field.readOnlyCondition(this.getValues());
  }

  updateState(field?: FieldSchema): FieldSchema | undefined {
    const thisField = field ?? this.state.field;
    if (!thisField) {
      return undefined;
    }
    this.setState({
      hidden: this.isHidden(thisField!),
      /* eslint-disable react/no-unused-state */
      disabled: this.isDisabled(thisField!),
      readOnly: this.isReadOnly(thisField!)
      /* eslint-enable react/no-unused-state */
    });
    return thisField;
  }

  compareValues(a: any, b: any): boolean {
    return a === b;
  }

  isValueIgnored(): boolean {
    return !!this.state.hidden || !!this.state.disabled || !!this.state.readOnly;
  }
}

export default BaseField;
