import React from "react";
import { Form } from "semantic-ui-react";
import classNames from "classnames";
import BaseField from "../BaseField";
import { clone } from "@solid/libs/utils";

import "./style.css";

export type FieldType = "string" | "integer" | "float" | "dropdown" | "checkbox" | "password" | "button";

export type DataSourceItem = {
  id: string;
  name: string;
};

export type FieldSchema = {
  name: string;
  label: string;
  type?: FieldType;
  required?: boolean;
  dataSource?: DataSourceItem[] | ((values: FieldValues) => DataSourceItem[]);
  minValue?: number;
  maxValue?: number;
  validate?: (schema: FieldSchema, value: any, values: FieldValues) => string;
  hideCondition?: (values: FieldValues) => boolean;
  disableCondition?: (values: FieldValues) => boolean;
  readOnlyCondition?: (values: FieldValues) => boolean;
  action?: () => void;
};

export type FormSchema = (FieldSchema | undefined)[];

export type FieldValues = Record<string, any>;

export type Context = {
  form?: AutoForm;
};

export const AutoFormContext = React.createContext<Context>({});

export type SchemaObserver = (schema: FormSchema) => void;

export type AutoFormProps = React.PropsWithChildren<{
  schema: FormSchema;
  values?: FieldValues;
  onChange?: (name: string, value: any, values: FieldValues, form: AutoForm) => void;
  onValidChange?: (form: AutoForm, isValid: boolean) => void;
  className?: string;
}>;

export type AutoFormState = {};

class AutoForm extends React.Component<AutoFormProps, AutoFormState> {
  private fields: Record<string, BaseField> = {};
  private values: FieldValues = {};
  private observers: Record<string, SchemaObserver> = {};
  private isValid?: boolean;
  private isChanged = false;

  constructor(props: AutoFormProps) {
    super(props);
    this.state = {};
  }

  override componentDidMount(): void {
    this.values = this.props.values ? clone(this.props.values) : {};
  }

  override componentDidUpdate(prevProps: AutoFormProps, prevState: AutoFormState): void {
    if (prevProps.schema !== this.props.schema) {
      for (const id in this.observers) {
        this.observers[id](this.props.schema);
      }
      for (const name in this.fields) {
        this.fields[name].setFieldSchema(this.props.schema.find(field => field?.name === name));
      }
    }
    if (prevProps.values !== this.props.values && this.props.values) {
      this.isChanged = false;
      this.values = clone(this.props.values);
      for (const name in this.fields) {
        this.fields[name].setValue(this.props.values[name]);
      }
    }
  }

  registerField(name: string, field: BaseField): void {
    this.fields[name] = field;
    field.setFieldSchema(this.props.schema.find(field => field?.name === name));
    const value = field.setValue(this.props.values ? this.props.values[name] : undefined);
    this.values[name] = value;
  }

  unregisterField(name: string): void {
    delete this.fields[name];
  }

  registerObserver(id: string, observer: SchemaObserver): void {
    this.observers[id] = observer;
    observer(this.props.schema);
  }

  unregisterObserver(id: string): void {
    delete this.observers[id];
  }

  getValues(): FieldValues {
    return this.values;
  }

  onChange(schema: FieldSchema, value: any): void {
    this.values[schema.name] = value;

    if (!this.props.values) {
      this.isChanged = true;
    }
    else {
      let result = false;
      for (const name in this.fields) {
        if (!this.fields[name].isValueIgnored() &&
            !this.fields[name].compareValues(this.values[name], this.props.values[name])) {
          result = true;
          break;
        }
      }
      this.isChanged = result;
    }

    for (const name in this.fields) {
      if (schema.name !== name) {
        this.fields[name].updateState();
      }
    }
    this.props.onChange && this.props.onChange(schema.name, value, this.getValues(), this);
  }

  areValuesChanged(names: string[], valuesToCompare?: FieldValues): boolean {
    const values = valuesToCompare ?? this.props.values;
    if (!values) {
      return true;
    }
    for (const name of names) {
      const field = this.fields[name];
      if (field && !field.isValueIgnored() && !field.compareValues(this.values[name], values[name])) {
        return true;
      }
    }
    return false;
  }

  onFieldValidChange(schema: FieldSchema, isValid: boolean): void {
    const prevValid = this.isValid;
    if (!isValid) {
      this.isValid = false;
    }
    else {
      let result = true;
      for (const name in this.fields) {
        result &&= this.fields[name].getIsValid() !== false;
      }
      this.isValid = result;
    }
    if (prevValid !== this.isValid && this.props.onValidChange) {
      this.props.onValidChange(this, this.isValid);
    }
  }

  validate(): boolean {
    let result = true;
    for (const name in this.fields) {
      result = this.fields[name].validate() && result;
    }
    return result;
  }

  setValue(name: string, value: any): void {
    const field = this.fields[name];
    if (field) {
      this.values[name] = field.setValue(value);
    }
  }

  getIsChanged(): boolean {
    return this.isChanged;
  }

  getIsValid(): boolean {
    return this.isValid !== false;
  }

  override render() {
    const { className } = this.props;
    return (
      <AutoFormContext.Provider value={{ form: this }}>
        <Form autoComplete="off" className={classNames("AutoForm", { [className ?? ""]: !!className })} noValidate onSubmit={e => { e.preventDefault(); }}>
          {this.props.children}
        </Form>
      </AutoFormContext.Provider>
    );
  }
}

export default AutoForm;
