import React, { useState, useEffect } from "react";
import { useNavigate, useParams, useLocation } from "react-router-dom";
import { Segment, Table, Icon, Popup, Input, Modal, Button, Header } from "semantic-ui-react";
import {useApolloClient} from "@apollo/client";
import {useUsersQuery, UsersQuery, useDeleteUserMutation, UsersDocument, GroupsDocument, GroupsQuery} from "generated/graphql";
import { PropType } from "utils";
import WithQueryStatus from "components/WithQueryStatus";
import IconButton from "components/IconButton";
import Loading from "components/Loading";
import CreateUpdateUser from "components/Admin/Users/CreateUpdateUser";
import { WidgetProps } from "components/Widgets";
import ListText from "components/Admin/Helpers/ListText";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import HelpMD from "./help.md";
import produce from "immer";
import {UUID} from "@solid/types";
import { RouteParams } from "@core/types";

import "./style.css";

type UsersProps = WidgetProps;

type User = PropType<UsersQuery, "users">[0];

const Users = ({ cellProps, setCellProps }: UsersProps) => {
  const client = useApolloClient();
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const { viewId = "", userId = "" } = useParams<RouteParams>();
  const { data, error, loading } = useUsersQuery();
  const [users, setUsers] = useState<User[]>([]);
  const [searchText, setSearchText] = useState("");
  const [selectedUserId, setSelectedUserId] = useState("");
  const selectedUser = selectedUserId ? users.find(user => user.id === selectedUserId) : undefined;
  const [createUpdateOpen, setCreateUpdateOpen] = useState(false);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [deleteUser, { error: deleteError, loading: deleteLoading }] = useDeleteUserMutation();

  useEffect(() => {
    pathname.includes("/view/user/add") && onCreateClick();
  }, [pathname]);

  useEffect(() => {
    if (!userId || !data) return;

    const isUserExist = data.users.some(user => user.id === userId);
    isUserExist && onUpdateClick(userId);
  }, [data, userId]);

  useEffect(() => {
    if (viewId && createUpdateOpen && !deleteConfirmOpen) {
      setCreateUpdateOpen(false);
    }
  }, [viewId]);

  useEffect(() => {
    if (data) {
      const users = Array.from(data.users)
        .filter(user => !searchText || user.name.toLocaleUpperCase().includes(searchText.toLocaleUpperCase()))
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
      setUsers(users);
    }
  }, [data, searchText]);

  useEffect(() => {
    if (deleteError) {
      Log.error(`Could not delete user: ${deleteError.message}`);
    }
  }, [deleteError]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }
    let title = cellProps?.title ?? "";
    if (createUpdateOpen) {
      title = selectedUser ? __("Edit User") : __("Create User");
    }
    setCellProps({ title });
  }, [createUpdateOpen, selectedUser]);

  function onCreateClick(): void {
    setSelectedUserId("");
    setCreateUpdateOpen(true);
    !pathname.includes("/view/user/add") && navigate("/view/user/add");
  }

  function onUpdateClick(userId: UUID): void {
    setSelectedUserId(userId);
    setCreateUpdateOpen(true);
    !pathname.includes(`/view/user/edit/${userId}`) && navigate(`/view/user/edit/${userId}`);
  }

  function onDeleteClick(user: User): void {
    setSelectedUserId(user.id);
    setDeleteConfirmOpen(true);
  }

  async function onDeleteUser(): Promise<void> {
    setDeleteConfirmOpen(false);
    if (!selectedUser) {
      return;
    }

    const userId = selectedUser.id;
    const response = await deleteUser({
      variables: { id: userId }
    });

    // update cache
    // update UsersDocument and GroupsDocument
    if (response.data?.deleteUser) {
      const usersCacheData = client.readQuery<UsersQuery>({
        query: UsersDocument
      });

      const usersCache = usersCacheData?.users ?? [];
      const userIndex = usersCache.findIndex((user) => user.id === userId);
      const users = produce(usersCache, (draft) => {
        draft.splice(userIndex, 1);
      });

      client.writeQuery<UsersQuery>({
        query: UsersDocument,
        data: {
          users
        }
      });

      const groupsCacheData = client.readQuery<GroupsQuery>({
        query: GroupsDocument
      });

      const groupsCache = groupsCacheData?.groups ?? [];
      const groupIndexToUserIndex = new Map<number, number>();
      groupsCache.forEach((group, groupIndex) => {
        const userIndex = group.users.findIndex((user) => user.id === userId);
        if (userIndex !== -1) {
          groupIndexToUserIndex.set(groupIndex, userIndex);
        }
      });

      if (groupIndexToUserIndex.size > 0) {
        const groups = produce(groupsCache, (draft) => {
          for (const [groupIndex, userIndex] of groupIndexToUserIndex) {
            draft[groupIndex].users.splice(userIndex, 1);
          }
        });

        client.writeQuery<GroupsQuery>({
          query: GroupsDocument,
          data: {
            groups
          }
        });
      }
    }
  }

  const afterCreateUser = (userId: UUID) => {
    setCreateUpdateOpen(false);
    setSelectedUserId(userId);
    navigate(`/view/user/edit/${userId}`, { replace: true });
  };

  return !createUpdateOpen ?
    <Segment className="Users">
      <WithQueryStatus loading={loading} error={error}>
        <div className="Users-Content">
          <div className="Users-Top">
            <Input
              placeholder={__("Filter by name")}
              icon="search"
              value={searchText}
              onChange={e => setSearchText(e.currentTarget.value)}
            />
            <IconButton
              icon="plus"
              hint={__("Create User")}
              onClick={onCreateClick}
            />
          </div>

          <div className="Users-Data">
            <Table celled compact>
              <Table.Header>
                <Table.Row>
                  <Table.HeaderCell/>
                  <Table.HeaderCell>{__("Name")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Groups")}</Table.HeaderCell>
                </Table.Row>
              </Table.Header>

              <Table.Body>
                {users.map(user =>
                  <Table.Row key={user.id}>
                    <Table.Cell collapsing>
                      <Popup trigger={
                        <Icon name="edit" className="Users-IconButton" onClick={() => onUpdateClick(user.id)}/>
                        }
                        content={__("Edit")}
                      />
                      <Popup trigger={
                        <Icon name="trash alternate" className="Users-IconButton" onClick={() => onDeleteClick(user)}/>
                        }
                        content={__("Delete")}
                      />
                    </Table.Cell>
                    <Table.Cell>
                      {user.name}
                    </Table.Cell>
                    <Table.Cell>
                      <ListText items={user.groups} modalHeader={__("Groups")}/>
                    </Table.Cell>
                  </Table.Row>)}

                {users.length === 0 &&
                <Table.Row>
                  <Table.Cell colSpan={3} textAlign="center">
                    {__("No users found")}
                  </Table.Cell>
                </Table.Row>}
              </Table.Body>
            </Table>
          </div>
        </div>

        <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
          <Header>{__("Delete User")}</Header>
          <Modal.Content>{__("Are you sure you want to delete user '{{userName}}'?", {userName: selectedUser?.name})}</Modal.Content>
          <Modal.Actions>
            <Button negative onClick={() => onDeleteUser()}>
              <Icon name="trash"/>{__("Delete")}
            </Button>
            <Button onClick={() => setDeleteConfirmOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {deleteLoading && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment> :
    <CreateUpdateUser
      user={selectedUser}
      afterCreateUser={afterCreateUser}
      onBack={() => navigate(-1)}
    />;
};

export default Users;
