import React, { useState, useEffect, useRef } from "react";
import { Segment, Form, Button, Icon, Message, Tab, Modal, Header, Menu } from "semantic-ui-react";
import {
  UsersQuery,
  useUsersQuery,
  useGroupsQuery,
  useCreateUserMutation,
  useUpdateUserMutation,
  useChangePasswordMutation,
  useSetPasswordMutation,
  UsersDocument,
  GroupsDocument,
  useUnblockUserMutation,
  GroupsQuery,
  useUserInfoLazyQuery
} from "generated/graphql";
import { withStore, WithStoreProps } from "@core/store";
import { PropType } from "utils";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import AssignObjects from "components/Admin/Helpers/AssignObjects";
import EventList from "components/EventList";
import EventInfoComponent from "components/EventInfo";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import HelpMD from "./help.md";
import {CheckboxProps} from "semantic-ui-react/dist/commonjs/modules/Checkbox/Checkbox";
import type {ArrayElement, UUID} from "@solid/types";
import produce from "immer";
import {useApolloClient} from "@apollo/client";

import "./style.css";

type CreateUpdateUserProps = WithStoreProps & {
  user?: User;
  onBack?: () => void;
  afterCreateUser?: (userId: UUID) => void;
};

type User = PropType<UsersQuery, "users">[0];

enum PasswordMode {
  Change = "Change",
  Set = "Set"
}

const CreateUpdateUser = ({ user, onBack, afterCreateUser, store: { session: { info } }, setStore }: CreateUpdateUserProps) => {
  const client = useApolloClient();
  const [getUserInfo] = useUserInfoLazyQuery();
  const { data: usersData, error: usersError, loading: usersLoading } = useUsersQuery();
  const { data: groupsData, error: groupsError, loading: groupsLoading } = useGroupsQuery();
  const [groupIdSet, setGroupIdSet] = useState(new Set<string>());
  const [createUser, { error: createError, loading: createLoading }] = useCreateUserMutation();
  const [updateUser, { data: updateData, error: updateError, loading: updateLoading }] = useUpdateUserMutation();
  const [changeUserPassword, { data: chPassData, error: chPassError, loading: chPassLoading }] = useChangePasswordMutation();
  const [setUserPassword, { data: setPassData, error: setPassError, loading: setPassLoading }] = useSetPasswordMutation();
  const [unblockUser/*, { data: unblockUserData, error: unblockUserError, loading: unblockUserLoading }*/] = useUnblockUserMutation();
  const [error, setError] = useState<Error | undefined>();
  const [name, setName] = useState(user?.name ?? "");
  const [nameError, setNameError] = useState("");
  const [email, setEmail] = useState(user?.email ?? "");
  const [sendNotification, setSendNotification] = useState(user?.sendNotification ?? false);
  const [password, setPassword] = useState("");
  const [passwordError, setPasswordError] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [confirmPasswordError, setConfirmPasswordError] = useState("");
  const [currentPassword, setCurrentPassword] = useState("");
  const [currentPasswordError, setCurrentPasswordError] = useState("");
  const [passwordMode, setPasswordMode] = useState(PasswordMode.Change);
  const [activeTab, setActiveTab] = useState(0);
  const [changesDialogOpen, setChangesDialogOpen] = useState(false);
  const backOnUpdateRef = useRef(false);
  const [getCsvFile, setGetCsvFile] = useState<boolean>(false);
  const nextTabRef = useRef(0);
  const prevTabRef = useRef(-1);
  const prevActiveTabIndex = useRef(0);
  const tabCount = 3;
  const [wasOneActive, setWasOneActive] = useState(false);

  useEffect(() => {
    setName(user?.name ?? "");
    setEmail(user?.email ?? "");
    setSendNotification(!!user?.sendNotification);
    setGroupIdSet(new Set<string>(user?.groups.map(group => group.id) ?? []));
  }, [user]);

  useEffect(() => {
    setError(createError);
  }, [createError]);

  useEffect(() => {
    setError(updateError);
  }, [updateError]);

  useEffect(() => {
    setError(chPassError);
  }, [chPassError]);

  useEffect(() => {
    setError(setPassError);
  }, [setPassError]);

  useEffect(() => {
    if (updateData || chPassData || setPassData) {
      prevTabRef.current = -1;
      setPassword("");
      setConfirmPassword("");
      setCurrentPassword("");
    }
  }, [updateData, chPassData, setPassData]);

  useEffect(() => {
    if ((updateError || chPassError || setPassError) && prevTabRef.current >= 0) {
      setActiveTab(prevTabRef.current);
      prevTabRef.current = -1;
    }
  }, [updateError, chPassError, setPassError]);

  useEffect(() => {
    if (user && !hasChangesOnTab(1)) {
      discardTabChanges(1);
    }
  }, [password, confirmPassword, currentPassword]);

  useEffect(() => {
    if (getCsvFile) {
      setGetCsvFile(false);
    }
  }, [getCsvFile]);

  const cancel = () => {
    onBack && onBack();
  };

  function getNameError(name: string): string {
    let error = "";
    const userId = user?.id;
    if (!name.trim()) {
      error = __("User name should be not empty");
    } else
    if (usersData?.users.find(user => user.id !== userId && user.name.toLocaleUpperCase() === name.trim().toLocaleUpperCase())) {
      error = __("User with the same name already exists");
    }
    return error;
  }

  function getPasswordError(password: string): string {
    let error = "";
    if (!password.trim()) {
      error = !user ? __("Password should be not empty") : __("New Password should be not empty");
    }
    return error;
  }

  function getConfirmPasswordError(password: string, confirmPassword: string): string {
    let error = "";
    if (!confirmPassword.trim()) {
      error = !user ? __("Confirm Password should be not empty") : __("Confirm New Password should be not empty");
    } else
    if (password !== confirmPassword) {
      error = !user ? __("Password and Confirm Password do not match") : __("New Password and Confirm New Password do not match");
    }
    return error;
  }

  function getCurrentPasswordError(password: string): string {
    let error = "";
    if (!password.trim()) {
      error = __("Current Password should be not empty");
    }
    return error;
  }

  function onNameInput(e: React.FormEvent<HTMLInputElement>): void {
    const name = e.currentTarget.value;
    setName(name);
    setNameError(getNameError(name));
  }

  function onEmailInput(e: React.FormEvent<HTMLInputElement>): void {
    const email = e.currentTarget.value;
    setEmail(email);
  }

  function onSendNotificationChange(e: React.FormEvent<HTMLInputElement>, data: CheckboxProps): void {
    setSendNotification(!!data?.checked);
  }

  function onPasswordInput(e: React.FormEvent<HTMLInputElement>): void {
    const password = e.currentTarget.value;
    setPassword(password);
    setPasswordError(getPasswordError(password));
    if (confirmPassword && password === confirmPassword) {
      setConfirmPasswordError("");
    }
  }

  function onConfirmPasswordInput(e: React.FormEvent<HTMLInputElement>): void {
    const confirmPassword = e.currentTarget.value;
    setConfirmPassword(confirmPassword);
    setConfirmPasswordError(getConfirmPasswordError(password, confirmPassword));
  }

  function onCurrentPasswordInput(e: React.FormEvent<HTMLInputElement>): void {
    const password = e.currentTarget.value;
    setCurrentPassword(password);
    setCurrentPasswordError(getCurrentPasswordError(password));
  }

  function onUnblockUser() {
    if (user?.id) {
      unblockUser({
        variables: {
          id: user.id
        },
      });
    }
  }

  async function onCreateUpdate(backOnUpdate = false, currentTab = activeTab): Promise<void> {
    const userName = name.trim();
    const userEmail = email.trim();

    if (!user) {
      for (let i = 0; i < tabCount; i++) {
        if (!validateTab(i)) {
          setActiveTab(i);
          return;
        }
      }

      const response = await createUser({
        variables: {
          user: {
            name: userName,
            email: userEmail,
            sendNotification,
            password: password.trim(),
            confirmPassword: confirmPassword.trim(),
            addToGroupIds: Array.from(groupIdSet.keys())
          }
        }
      });

      const userId = response.data?.createUser.id;
      // update cache
      // update UsersDocument and GroupsDocument
      if (userId) {
        const usersCacheData = client.readQuery<UsersQuery>({
          query: UsersDocument
        });

        const allGroups = groupsData?.groups ?? [];
        const groups = allGroups.filter((group) => groupIdSet.has(group.id));

        const newUser: ArrayElement<UsersQuery["users"]> = {
          __typename: "User",
          id: userId,
          name: userName,
          email: userEmail,
          emailConfirmed: false,
          sendNotification,
          groups: groups.map((group) => { return { __typename: "Group", id: group.id, name: group.name, isSystemManaged: group.isSystemManaged }; })
        };

        const usersCache = usersCacheData?.users ?? [];
        const users = produce(usersCache, (draft) => {
          draft.push(newUser);
        });

        client.writeQuery<UsersQuery>({
          query: UsersDocument,
          data: {
            users
          }
        });

        if (groups.length > 0) {
          const groupsCacheData = client.readQuery<GroupsQuery>({
            query: GroupsDocument
          });

          const groupsCache = groupsCacheData?.groups ?? [];
          const groupsIndex: number[] = [];
          groupsCache.forEach((group, groupIndex) => {
            if (groupIdSet.has(group.id)) {
              groupsIndex.push(groupIndex);
            }
          });

          if (groupsIndex.length > 0) {
            const groups = produce(groupsCache, (draft) => {
              for (const groupIndex of groupsIndex) {
                draft[groupIndex].users.push(newUser);
              }
            });

            client.writeQuery<GroupsQuery>({
              query: GroupsDocument,
              data: {
                groups
              }
            });
          }
        }

        if (response.data?.createUser.warning) {
          Log.warning(response.data.createUser.warning);
        }
        afterCreateUser && afterCreateUser(userId);
      }
    } else {
      if (!validateTab(currentTab)) {
        return;
      }

      backOnUpdateRef.current = backOnUpdate;
      prevTabRef.current = activeTab;

      const userId = user.id;

      let response;
      switch (currentTab) {
        case 0:
          response = await updateUser({
            variables: {
              id: userId,
              user: {
                name: userName,
                email: userEmail,
                sendNotification
              }
            }
          });

          // update cache
          // update UsersDocument
          if (response.data?.updateUser) {
            const usersCacheData = client.readQuery<UsersQuery>({
              query: UsersDocument
            });

            const usersCache = usersCacheData?.users ?? [];
            const userIndex = usersCache.findIndex((user) => user.id === userId);
            const users = produce(usersCache, (draft) => {
              draft[userIndex].name = userName;
              draft[userIndex].email = userEmail;
              draft[userIndex].sendNotification = sendNotification;
            });

            client.writeQuery<UsersQuery>({
              query: UsersDocument,
              data: {
                users
              }
            });
          }
          break;
        case 1:
          if (passwordMode === PasswordMode.Change) {
            response = await changeUserPassword({
              variables: {
                input: {
                  userId,
                  currentPassword: currentPassword.trim(),
                  newPassword: password.trim()
                }
              }
            });
          } else {
            response = await setUserPassword({
              variables: {
                input: {
                  userId,
                  newPassword: password.trim()
                }
              }
            });
          }
          break;
        case 2:
          const addToGroupIds = Array.from(groupIdSet).filter(id => !user.groups.some(g => g.id === id));
          const removeFromGroupIds = user.groups.filter(group => !groupIdSet.has(group.id)).map(group => group.id);
          response = await updateUser({
            variables: {
              id: userId,
              user: {
                addToGroupIds,
                removeFromGroupIds
              }
            }
          });

          // update cache
          // update UsersDocument, GroupsDocument
          if (response.data?.updateUser) {
            const usersCacheData = client.readQuery<UsersQuery>({
              query: UsersDocument
            });

            const allGroups = groupsData?.groups ?? [];
            const groups = allGroups.filter((group) => groupIdSet.has(group.id));

            const changedUser = produce(user, (draft) => {
              draft.groups = groups;
            });
            const usersCache = usersCacheData?.users ?? [];
            const userIndex = usersCache.findIndex((user) => user.id === userId);
            const users = produce(usersCache, (draft) => {
              draft[userIndex].groups = changedUser.groups;
            });

            client.writeQuery<UsersQuery>({
              query: UsersDocument,
              data: {
                users
              }
            });

            const groupsCacheData = client.readQuery<GroupsQuery>({
              query: GroupsDocument
            });

            // add and remove user from groups
            const groupsCache = groupsCacheData?.groups ?? [];
            const addGroupIndex: number[] = [];
            const removeGroupIndexToUserIndex = new Map<number, number>();
            groupsCache.forEach((group, groupIndex) => {
              if (addToGroupIds.includes(group.id)) {
                const userIndex = group.users.findIndex((user) => user.id === userId);
                if (userIndex === -1) {
                  addGroupIndex.push(groupIndex);
                }
              }
              if (removeFromGroupIds.includes(group.id)) {
                const userIndex = group.users.findIndex((user) => user.id === userId);
                if (userIndex !== -1) {
                  removeGroupIndexToUserIndex.set(groupIndex, userIndex);
                }
              }
            });

            if (addGroupIndex.length > 0 || removeGroupIndexToUserIndex.size > 0) {
              const groups = produce(groupsCache, (draft) => {
                // add to groups
                for (const groupIndex of addGroupIndex) {
                  draft[groupIndex].users.push(changedUser);
                }
                // remove from groups
                for (const [groupIndex, userIndex] of removeGroupIndexToUserIndex) {
                  draft[groupIndex].users.splice(userIndex, 1);
                }
              });

              client.writeQuery<GroupsQuery>({
                query: GroupsDocument,
                data: {
                  groups
                }
              });
            }
          }
          break;
      }

      if (response?.data) {
        if (info?.user.id && info?.user.id === user?.id) {
          const { data } = await getUserInfo();
          setStore({ session: { isLoggedIn: true, info: data?.userInfo } });
        }
        backOnUpdateRef.current && onBack && onBack();
      }
    }
  }

  function onActiveTabChange(index: number | string | undefined): void {
    if (typeof index !== "number") {
      return;
    }
    if (index === 3 && !wasOneActive) {
      setWasOneActive(true);
    }

    if (!user) {
      if (index < activeTab || validateTab(activeTab)) {
        setActiveTab(index);
      }
    } else {
      if (!hasChangesOnTab(activeTab)) {
        setActiveTab(index);
        return;
      }

      if (!validateTab(activeTab)) {
        return;
      }

      nextTabRef.current = index;
      setChangesDialogOpen(true);
    }
    prevActiveTabIndex.current = activeTab;
    setActiveTab(index);
  }

  function hasChangesOnTab(tabIndex: number): boolean {
    if (!user) {
      switch (tabIndex) {
        case 0: return !!name.trim() || !!email.trim();
        case 1: return !!password.trim() || !!confirmPassword.trim();
        case 2: return groupIdSet.size > 0;
        default: return false;
      }
    } else {
      switch (tabIndex) {
        case 0: return name.trim() !== user.name || email.trim() !== user.email || sendNotification !== user.sendNotification;
        case 1: return !!password.trim() || !!confirmPassword.trim() || !!currentPassword.trim();
        case 2: return user.groups.some(group => !groupIdSet.has(group.id)) ||
          Array.from(groupIdSet).some(id => !user.groups.some(g => g.id === id));
        default: return false;
      }
    }
  }

  function validateTab(tabIndex: number): boolean {
    let result = true;
    switch (tabIndex) {
      case 0:
        const nameError = getNameError(name);
        if (nameError) {
          setNameError(nameError);
          result = false;
        }
        break;
      case 1:
        const passwordError = getPasswordError(password);
        if (passwordError) {
          setPasswordError(passwordError);
          result = false;
        }
        const confirmPasswordError = getConfirmPasswordError(password, confirmPassword);
        if (confirmPasswordError) {
          setConfirmPasswordError(confirmPasswordError);
          result = false;
        }
        if (user && passwordMode === PasswordMode.Change) {
          const currentPasswordError = getCurrentPasswordError(currentPassword);
          if (currentPasswordError) {
            setCurrentPasswordError(currentPasswordError);
            result = false;
          }
        }
        break;
    }
    return result;
  }

  function discardTabChanges(tabIndex: number): void {
    switch (tabIndex) {
      case 0:
        setName(user?.name ?? "");
        setNameError("");
        setEmail(user?.email ?? "");
        break;
      case 1:
        setPassword("");
        setConfirmPassword("");
        setCurrentPassword("");
        setPasswordError("");
        setConfirmPasswordError("");
        setCurrentPasswordError("");
        break;
      case 2:
        setGroupIdSet(new Set<string>(user?.groups.map(group => group.id) ?? []));
        break;
    }
  }

  const panes = [
    {
      menuItem: __("User Properties"),
      pane: (
        <Tab.Pane key="userProperties">
          <Form className="CreateUpdateUser-Form" onSubmit={e => { e.preventDefault(); }}>
            <Form.Field
              control={Form.Input}
              label={__("Name")}
              placeholder={__("Name")}
              autoComplete="username"
              autoFocus
              value={name}
              error={nameError ? { content: nameError, pointing: "below" } : undefined}
              onInput={onNameInput}
            />
            <Form.Field
              control={Form.Input}
              label={__("Email") + ` (${user?.emailConfirmed ? __("Confirmed") : __("Not Confirmed")})`}
              placeholder={__("Email")}
              autoComplete="email"
              value={email}
              onInput={onEmailInput}
            />
            <Form.Field
              control={Form.Checkbox}
              label={__("Send notification")}
              toggle
              checked={sendNotification}
              onChange={onSendNotificationChange}
            />
            {user &&
              <Form.Field
                control={Form.Button}
                label={__("Unblock")}
                content={__("Unblock")}
                onClick={onUnblockUser}
              />}
          </Form>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Password"),
      pane: (
        <Tab.Pane key="userPassword">
          <Form className="CreateUpdateUser-Form" onSubmit={e => { e.preventDefault(); }}>
            {!!user &&
            <>
              <Form.Radio
                label={__("Change password")}
                value={PasswordMode.Change}
                checked={passwordMode === PasswordMode.Change}
                onChange={(e, { value }) => setPasswordMode(value === PasswordMode.Change ? PasswordMode.Change : PasswordMode.Set)}
              />
              <Form.Radio
                label={__("Set password (will be marked as expired)")}
                value={PasswordMode.Set}
                checked={passwordMode === PasswordMode.Set}
                onChange={(e, { value }) => setPasswordMode(value === PasswordMode.Change ? PasswordMode.Change : PasswordMode.Set)}
              />
            </>}
            {!!user && passwordMode === PasswordMode.Change &&
            <Form.Field
              control={Form.Input}
              label={__("Current Password")}
              placeholder={__("Current Password")}
              type="password"
              autoComplete="current-password"
              autoFocus
              value={currentPassword}
              error={currentPasswordError ? { content: currentPasswordError, pointing: "below" } : undefined}
              onInput={onCurrentPasswordInput}
            />}
            <Form.Field
              control={Form.Input}
              label={!user ? __("Password") : __("New Password")}
              placeholder={!user ? __("Password") : __("New Password")}
              type="password"
              autoFocus={!user || passwordMode === PasswordMode.Set}
              autoComplete="new-password"
              value={password}
              error={passwordError ? { content: passwordError, pointing: "below" } : undefined}
              onInput={onPasswordInput}
            />
            <Form.Field
              control={Form.Input}
              label={!user ? __("Confirm Password") : __("Confirm New Password")}
              placeholder={!user ? __("Confirm Password") : __("Confirm New Password")}
              type="password"
              autoComplete="new-password"
              value={confirmPassword}
              error={confirmPasswordError ? { content: confirmPasswordError, pointing: "below" } : undefined}
              onInput={onConfirmPasswordInput}
            />
          </Form>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Groups"),
      pane: (
        <Tab.Pane key="userGroups">
          <AssignObjects
            items={groupsData?.groups.map(group => ({ ...group, children: group.policies }))}
            assignedIdSet={groupIdSet}
            itemSingleText={__("Group")}
            itemPluralText={__("Groups")}
            parentSingleText={__("User")}
            childPluralText={__("Policies")}
            selectPrompt={__("Select groups that user will be assigned to:")}
            removePrompt={__("Remove User from the Group")}
            onAssignedIdSetChange={idSet => setGroupIdSet(idSet)}
          />
        </Tab.Pane>
      )
    },
    {
      menuItem: !user ? <Menu.Item key="Audit" className="Audit-Disabled" disabled content={__("Audit")}/> : __("Audit"),
      pane: (
        <Tab.Pane key="userAudit">
          {user && (activeTab === 3 || wasOneActive) &&
            <div className="audit">
              <div className="event-control">
                <div className="event-list-control">
                  <Header size="tiny">{__("Events: {{userName}}", {userName: user.name})}</Header>
                  <Button
                    className="export-audit"
                    size="mini"
                    onClick={() => setGetCsvFile(true)}>
                    <Icon name="download"/>
                    {__("Export CSV")}
                  </Button>
                </div>
                <div className="event-info-control">
                  <Header size="tiny">{__("Event details")}</Header>
                </div>
              </div>
              <div className="audit-content">
                <div className="event-list">
                  <div className="events">
                    <EventList witnessesList={[user.id || ""]} exportCsvFile={getCsvFile}/>
                  </div>
                </div>
                <div className="event-info">
                  <div className="info">
                    <EventInfoComponent isCell/>
                  </div>
                </div>
              </div>
            </div>}
        </Tab.Pane>
      )
    }
  ];

  return (
    <Segment className="CreateUpdateUser">
      <WithQueryStatus error={usersError || groupsError} loading={usersLoading || groupsLoading}>
        {!!error && <Message error content={error.message}/>}

        <div className="CreateUpdateUser-TopButtons">
          <Button onClick={() => cancel()}>
            <Icon name="cancel"/>{__("Cancel")}
          </Button>
          {!user &&
          <>
            <Button disabled={activeTab <= 0} onClick={() => onActiveTabChange(activeTab - 1)}>
              <Icon name="arrow alternate circle left"/>{__("Back")}
            </Button>
            {activeTab < tabCount - 1 &&
            <Button positive onClick={() => onActiveTabChange(activeTab + 1)}>
              <Icon name="arrow alternate circle right"/>{__("Next")}
            </Button>}
          </>}
          {(!!user || activeTab >= tabCount - 1) &&
          <Button
            positive
            disabled={!!user && !hasChangesOnTab(activeTab)}
            onClick={() => onCreateUpdate(true)}
          >
            <Icon name={!user ? "plus" : "check"}/>{!user ? __("Create") : __("Save")}
          </Button>}
        </div>

        <Tab
          panes={panes}
          className="CreateUpdateUser-Tab"
          activeIndex={activeTab}
          renderActiveOnly={false}
          onTabChange={(e, { activeIndex }) => onActiveTabChange(activeIndex)}
        />

        <Modal open={changesDialogOpen} onClose={() => setChangesDialogOpen(false)}>
          <Header>{__("Unsaved Changes")}</Header>
          <Modal.Content>{__("User has unsaved changes. Would you like to save the changes?")}</Modal.Content>
          <Modal.Actions>
            <Button positive onClick={async () => {
              await onCreateUpdate(false, prevActiveTabIndex.current);
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="check"/>{__("Save")}
            </Button>
            <Button negative onClick={() => {
              discardTabChanges(activeTab);
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="undo"/>{__("Discard")}
            </Button>
            <Button onClick={() => setChangesDialogOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {(createLoading || updateLoading || chPassLoading || setPassLoading) && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment>
  );
};

export default withStore(CreateUpdateUser);
