import React, { useState, useEffect, useMemo } from "react";
import { Button, Header, Icon, Input, Modal, Popup, Segment, Table } from "semantic-ui-react";
import { useApolloClient } from "@apollo/client";
import { useNavigate, useParams, useLocation } from "react-router-dom";
import produce from "immer";
import WithQueryStatus from "components/WithQueryStatus";
import IconButton from "components/IconButton";
import ListText, { ListItem } from "../Helpers/ListText";
import Loading from "components/Loading";
import Help from "components/Help";
import CreateUpdateSet from "./CreateUpdateSet";
import { WidgetProps } from "components/Widgets";
import type { ArrayElement, UUID } from "@solid/types";
import {
  DeviceFunctionalAspectType,
  HealthStatus,
  LabelsDocument,
  LabelsQuery,
  LabelType,
  PoliciesDocument,
  PoliciesQuery,
  SetsDocument,
  SetsQuery,
  useDeleteSetMutation,
  useDeviceListByAspectTypesQuery,
  useSetsQuery
} from "@generated/graphql";
import { __ } from "@solid/libs/i18n";
import { Log } from "@solid/libs/log";
import { RouteParams } from "@core/types";

import "./style.css";
import HelpMD from "./help.md";
import { DeviceList, getListItemIcon } from "@core/actions";

type SetsProps = WidgetProps;

export type DeviceSet = ArrayElement<SetsQuery["sets"]>;

const deviceQueryTypes = [{ type: DeviceFunctionalAspectType.Media }, { type: DeviceFunctionalAspectType.Sensor }];

const Sets = ({ cellProps, setCellProps }: SetsProps) => {
  const client = useApolloClient();
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const { viewId = "", setId = "" } = useParams<RouteParams>();
  const { loading: deviceLoading, error: deviceError, data: deviceData } = useDeviceListByAspectTypesQuery({ variables: { types: deviceQueryTypes } });
  const { data, error, loading } = useSetsQuery();
  const [sets, setSets] = useState<DeviceSet[]>([]);
  const [createUpdateOpen, setCreateUpdateOpen] = useState(false);
  const [searchText, setSearchText] = useState<string | undefined>();
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [selectedSetId, setSelectedSetId] = useState("");
  const selectedSet = selectedSetId ? sets.find(set => set.id === selectedSetId) : undefined;
  const [deleteSet, { error: deleteError, loading: deleteLoading }] = useDeleteSetMutation();

  const deviceMap: Map<string, DeviceList> = useMemo(() => {
    if (!deviceData?.devicesByAspectTypes) return new Map();

    const map = deviceData?.devicesByAspectTypes.reduce((acc, dev) => {
      acc.set(dev.id, dev);
      return acc;
    }, new Map());

    return map;
  }, [deviceData]);

  useEffect(() => {
    pathname.includes("/view/set/add") && onCreateClick();
  }, [pathname]);

  useEffect(() => {
    if (!setId || !data) return;

    const isSetExist = data.sets.some(set => set.id === setId);
    isSetExist && onUpdateClick(setId);

  }, [data, setId]);

  useEffect(() => {
    if (viewId && createUpdateOpen && !deleteConfirmOpen) {
      setCreateUpdateOpen(false);
    }
  }, [viewId]);

  useEffect(() => {
    if (data) {
      const sets = Array.from(data.sets)
        .filter(set => !searchText || set.name.toLocaleUpperCase().includes(searchText.toLocaleUpperCase()))
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
      setSets(sets);
    }
  }, [data, searchText]);

  useEffect(() => {
    if (deleteError) {
      Log.error(`Could not delete set: ${deleteError.message}`);
    }
  }, [deleteError]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }
    let title = cellProps?.title ?? "";
    if (createUpdateOpen) {
      title = selectedSet ? __("Edit Set") : __("Create Set");
    }
    setCellProps({ title });
  }, [createUpdateOpen, selectedSet]);

  function onCreateClick(): void {
    setSelectedSetId("");
    setCreateUpdateOpen(true);
    !pathname.includes("/view/set/add") && navigate("/view/set/add");
  }

  function onUpdateClick(setId: UUID): void {
    setSelectedSetId(setId);
    setCreateUpdateOpen(true);
    !pathname.includes(`/view/set/edit/${setId}`) && navigate(`/view/set/edit/${setId}`);
  }

  function onDeleteClick(set: DeviceSet): void {
    setSelectedSetId(set.id);
    setDeleteConfirmOpen(true);
  }

  function getListTextItem(dev: { id: string; name: string }): ListItem {
    const device = deviceMap.get(dev.id);
    if (!device || !device.deviceType) return { ...dev };
    const disabled = !device.enabled || device.healthStatus !== HealthStatus.Normal;
    const icon = getListItemIcon(device.deviceType, disabled);
    return { ...dev, faIcon: icon };
  }

  async function onDeleteSet(): Promise<void> {
    setDeleteConfirmOpen(false);
    if (!selectedSet) {
      return;
    }

    const response = await deleteSet({
      variables: { id: selectedSet.id }
    });

    // update cache
    // update SetsDocument, PoliciesDocument, LabelsDocument
    if (response.data?.deleteSet) {
      // update SetsDocument
      const zonesCacheData = client.readQuery<SetsQuery>({
        query: SetsDocument
      });

      const setId = selectedSet.id;
      const setsCache = zonesCacheData?.sets ?? [];
      const setIndex = setsCache.findIndex((set) => set.id === setId);
      const sets = produce(setsCache, (draft) => {
        draft.splice(setIndex, 1);
      });

      client.writeQuery<SetsQuery>({
        query: SetsDocument,
        data: {
          sets
        }
      });

      // update PoliciesDocument
      const policiesCacheData = client.readQuery<PoliciesQuery>({
        query: PoliciesDocument
      });

      const policiesCache = policiesCacheData?.policies ?? [];
      if (policiesCache.length > 0) {
        const policies = produce(policiesCache, (draft) => {
          for (const policy of draft) {
            for (const statement of policy.statements) {
              for (let i = 0; i < statement.resources.length; i++) {
                const resource = statement.resources[i];
                if (resource.id === setId) {
                  statement.resources.splice(i, 1);
                  break;
                }
              }
            }
          }
        });

        client.writeQuery<PoliciesQuery>({
          query: PoliciesDocument,
          data: {
            policies
          }
        });
      }

      // update LabelsDocument
      const labelsCacheData = client.readQuery<LabelsQuery>({
        query: LabelsDocument
      });

      const labelsCache = labelsCacheData?.labels ?? [];
      if (labelsCache.length > 0) {
        const labelIndex = labelsCache.findIndex((label) => label.type === LabelType.Set && label.id === setId);
        const labels = produce(labelsCache, (draft) => {
          draft.splice(labelIndex, 1);
        });

        client.writeQuery<LabelsQuery>({
          query: LabelsDocument,
          data: {
            labels
          }
        });
      }
    }
  }

  return !createUpdateOpen ?
    <Segment className="Sets" error={error || deviceError} loading={loading || deviceLoading}>
      <WithQueryStatus >
        <div className="Sets-Content">
          <div className="Sets-Top">
            <Input
              placeholder={__("Filter by name")}
              icon="search"
              value={searchText}
              onChange={e => setSearchText(e.currentTarget.value)}
          />
            <IconButton
              icon="plus"
              hint={__("Create Set")}
              onClick={onCreateClick}
          />
          </div>

          <div className="Sets-Data">
            <Table celled compact>
              <Table.Header>
                <Table.Row>
                  <Table.HeaderCell/>
                  <Table.HeaderCell>{__("Name")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Devices")}</Table.HeaderCell>
                </Table.Row>
              </Table.Header>

              <Table.Body>
                {sets.map(set =>
                  <Table.Row key={set.id}>
                    <Table.Cell collapsing>
                      {!set.isSystemManaged &&
                        <>
                          <Popup trigger={
                            <Icon name="edit" className="Sets-IconButton" onClick={() => onUpdateClick(set.id)} />
                          }
                            content={__("Edit")} />
                          <Popup trigger={
                            <Icon name="trash alternate" className="Sets-IconButton" onClick={() => onDeleteClick(set)} />
                          }
                            content={__("Delete")} />
                        </>
                      }
                    </Table.Cell>
                    <Table.Cell>
                      {set.name}
                    </Table.Cell>
                    <Table.Cell>
                      <ListText items={set.devices.map(dev => getListTextItem(dev))} maxItems={5} modalHeader={__("Devices")} icons/>
                    </Table.Cell>
                  </Table.Row>)}

                {sets.length === 0 &&
                <Table.Row>
                  <Table.Cell colSpan={3} textAlign="center">
                    {__("No sets found")}
                  </Table.Cell>
                </Table.Row>}
              </Table.Body>
            </Table>
          </div>
        </div>

        <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
          <Header>{__("Delete Set")}</Header>
          <Modal.Content>{__("Are you sure you want to delete set '{{name}}'?", {name: selectedSet?.name})}</Modal.Content>
          <Modal.Actions>
            <Button negative onClick={() => onDeleteSet()}>
              <Icon name="trash"/>{__("Delete")}
            </Button>
            <Button onClick={() => setDeleteConfirmOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {deleteLoading && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment> :
    <CreateUpdateSet set={selectedSet} onBack={() => navigate(-1)}/>;
};

export default Sets;
