import React, { useEffect, useRef, useState } from "react";
import { useLocation, useNavigate, useParams } from "react-router-dom";
import { Segment } from "semantic-ui-react";

import { RouteParams } from "@core/types";
import { UUID } from "@solid/types";
import { DeviceFunctionalAspectType, useDeleteDeviceMutation, useSensorListByAspectTypeQuery } from "@generated/graphql";
import { SensorShort, useDeviceActions } from "@core/actions";
import { Log, __ } from "@solid/libs";

import SensorSettings from "./SensorSettings";
import SensorList from "./SensorList";
import DeleteModal from "./DeleteModal";
import { WidgetProps } from "components/Widgets";

import "./style.css";

type SensorsProps = WidgetProps & {
  avatarId?: UUID;
};

const Sensors = ({ avatarId, ...props }: SensorsProps) => {
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const { viewId = "", sensorId = "" } = useParams<RouteParams>();
  const { loading, error, data } = useSensorListByAspectTypeQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Sensor }] } });
  const [deleteDevice, { error: deleteError, loading: deleteLoading }] = useDeleteDeviceMutation();
  const { deleteCachedDevice } = useDeviceActions({ skipSubscription: true });

  const [configOpen, setConfigOpen] = useState<boolean>(false);
  const [deleteModalOpen, setDeleteModalOpen] = useState<boolean>(false);
  const [deviceToRemove, setDeviceToRemove] = useState<SensorShort >();
  const [selectedId, setSelectedId] = useState<string>();
  const [createdDeviceId, setCreatedDeviceId] = useState<string>();

  const mountedRef = useRef<boolean>(false);

  useEffect(() => {
    mountedRef.current = true;

    return () => {
      mountedRef.current = false;
    };
  }, []);

  useEffect(() => {
    if (sensorId) {
      setConfigOpen(true);
      setSelectedId(sensorId);
    }
    if (viewId) {
      setConfigOpen(false);
      setSelectedId(undefined);
    }
    if (pathname === "/view/sensor/add") {
      setConfigOpen(true);
      setSelectedId(undefined);
    }
  }, [viewId, sensorId, pathname]);

  useEffect(() => {
    if (deleteError) {
      Log.error(__("Could not delete device: {{message}}", { message: deleteError.message }));
    }
  }, [deleteError]);

  function onCreateSensor() {
    navigate("/view/sensor/add");
  }

  function onSelectSensor(sensor: SensorShort) {
    setSelectedId(sensor.id);
    navigate(`/view/sensor/edit/${sensor.id}`);
  }

  async function deleteSensor(sensor: SensorShort) {
    onModalClose();
    const { data } = await deleteDevice({ variables: { id: sensor.id } });
    if (data?.deleteDevice) {
      deleteCachedDevice(sensor.id);
    }
  }

  function onHandleDeleteSensor(sensor: SensorShort) {
    setDeleteModalOpen(true);
    setDeviceToRemove(sensor);
  }

  function onModalClose() {
    setDeleteModalOpen(false);
    setDeviceToRemove(undefined);
  }

  function onCreated(id: string) {
    if (!mountedRef.current) return;

    navigate(-1);
    setCreatedDeviceId(id);
  }

  return (
    <Segment className="Sensors" loading={deleteLoading}>
      { configOpen ?
        <SensorSettings
          id={selectedId}
          sensors={data?.devicesByAspectTypes}
          onCreated={onCreated}
          {...props}
        />
        :
        <>
          <SensorList
            data={data}
            loading={loading}
            error={error}
            onSelect={onSelectSensor}
            onDelete={onHandleDeleteSensor}
            onCreate={onCreateSensor}
            createdId={createdDeviceId}
            avatarId={avatarId}
            {...props}
          />
          { deleteModalOpen && deviceToRemove &&
          <DeleteModal
            device={deviceToRemove}
            onDelete={deleteSensor}
            onCancel={onModalClose}
          /> }
        </>
      }
    </Segment>
  );
};

export default Sensors;
