import React, { useEffect, useState } from "react";
import { Message, Segment } from "semantic-ui-react";

import { SensorEvent } from "@generated/graphql";
import { __ } from "@solid/libs";

import AssignedEvents from "./AssignedEvents";
import EventsToAssign from "./EventsToAssign";

import "./style.css";

type UserEventsProps = {
  assignedEvents: string[];
  eventsToAssign: SensorEvent[];
  firstProbe?: boolean;
  onChange: (events: string[]) => void;
};

const UserEvents = ({ assignedEvents, eventsToAssign, onChange, firstProbe = false }: UserEventsProps) => {
  const [checkedEvents, setCheckedEvents] = useState<SensorEvent[]>([]);
  const [events, setEvents] = useState<SensorEvent[]>([]);
  const [eventsToAssignVisible, setEventsToAssignVisible] = useState<boolean>(false);

  useEffect(() => {
    if (assignedEvents.length > 0 && eventsToAssign.length > 0) {
      const checkedEvents = eventsToAssign.filter(event => assignedEvents.includes(event.type));
      setCheckedEvents(checkedEvents);
    }

    setEvents(eventsToAssign);
  }, [assignedEvents, eventsToAssign]);

  function removeEvent(type: string) {
    const assignedEvents = checkedEvents.filter(event => event.type !== type).map(camera => camera.type);
    onChange(assignedEvents);
    setCheckedEvents(prevEvents => (prevEvents.filter(camera => camera.type !== type)));
  }

  function getSelectedEvents(events: SensorEvent[]) {
    const assignedEvents = [...checkedEvents, ...events].map(event => event.type);
    onChange(assignedEvents);
    setCheckedEvents(prevEvents => ([...prevEvents, ...events]));
  }

  const uncheckedEvents = checkedEvents.length === 0 ? events : events.filter(event => !checkedEvents.includes(event));

  return (
    <Segment className="UserEvents">
      { !firstProbe && checkedEvents.length === 0 && <Message error content={__("At least one event must be selected")} /> }

      <div className="UserEvents-Controls">
        <AssignedEvents
          events={checkedEvents}
          getEventsToAssign={() => setEventsToAssignVisible(true)}
          removeEvent={removeEvent}
      />
        { eventsToAssignVisible &&
        <EventsToAssign
          events={uncheckedEvents}
          getSelectedEvents={getSelectedEvents}
          onCancel={() => setEventsToAssignVisible(false)}
        /> }
      </div>
    </Segment>
  );
};

export default UserEvents;
