import React, { useState } from "react";
import { Button, Checkbox, Header, Segment } from "semantic-ui-react";

import { SensorEvent } from "@generated/graphql";
import { __ } from "@solid/libs";

import ButtonsWrapper from "components/Admin/Sensors/ButtonsWrapper";
import EntityTable, { EntityTableProps } from "components/EntityTable";

import "./style.css";

type EventsToAssignProps = {
  events?: SensorEvent[];
  getSelectedEvents: (events: SensorEvent[]) => void;
  onCancel: () => void;
};

const EventsToAssign = ({ events, getSelectedEvents, onCancel }: EventsToAssignProps) => {
  const [selectedEvents, setSelectedEvents] = useState<SensorEvent[]>([]);

  function selectUnselectSet(index: number, checked: boolean) {
    const event = events?.[index];
    if (!event) return;
    checked && setSelectedEvents(prevEvents => ([...prevEvents, event]));
    !checked && setSelectedEvents(prevEvents => (prevEvents.filter(ev => ev.type !== event.type)));
  }

  const eventsToAssignTableProps: EntityTableProps = {
    columns: ["", "Description"],
    widths: [undefined, 16],
    emptyLine: __("Link has no sensor events"),
    rows: events?.reduce((acc, event, index) => {
      acc.push([
        <Checkbox checked={selectedEvents.includes(event)} onChange={(e, data) => selectUnselectSet(index, !!data.checked)}/>,
        <span>&nbsp;{event.description}</span>
      ]);
      return acc;
    }, [] as React.ReactNode[][])
  };

  function onSelectClick() {
    getSelectedEvents(selectedEvents);
    setSelectedEvents([]);
  }

  return (
    <Segment className="EventsToAssign">
      <ButtonsWrapper>
        <Header as="h5" className="EventsToAssign-Header">{__("Select events to assign")}</Header>
      </ButtonsWrapper>
      <Segment className="EventsToAssign-Controls">
        <EntityTable {...eventsToAssignTableProps} />
        <ButtonsWrapper>
          <Button positive disabled={selectedEvents.length === 0} onClick={() => onSelectClick()}>{__("Select")}</Button>
          <Button onClick={() => onCancel()}>{__("Cancel")}</Button>
        </ButtonsWrapper>
      </Segment>
    </Segment>
  );
};

export default EventsToAssign;
