import React, { useEffect, useMemo, useState } from "react";
import { Segment } from "semantic-ui-react";

import { DeviceFunctionalAspectType, useDeviceWitnessesByAspectTypeQuery } from "@generated/graphql";
import { __ } from "@solid/libs";
import { DeviceWitness } from "@core/actions";

import AssignedWitnesses from "./AssignedWitnesses";
import WitnessesToAssign from "./WitnessToAssign";

import "./style.css";

type SensorWitnessesProps = {
  assignedIds: string[];
  onChange: (ids: string[]) => void;
};

const SensorWitnesses = ({ assignedIds, onChange }: SensorWitnessesProps) => {
  const { data: camerasData, loading: cameraLoading, error: cameraError } = useDeviceWitnessesByAspectTypeQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });

  const [assignedCameras, setAssignedCameras] = useState<DeviceWitness[]>([]);
  const [camerasToAssignVisible, setCamerasToAssignVisible] = useState<boolean>(false);

  const cameras = useMemo<DeviceWitness[]>(() => camerasData?.devicesByAspectTypes ?? [], [camerasData]);

  useEffect(() => {
    if (!camerasData?.devicesByAspectTypes) return;

    if (assignedIds.length > 0) {
      const witnesses: DeviceWitness[] = [];
      for (const cameraId of assignedIds) {
        const camera = camerasData.devicesByAspectTypes.find(camera => camera.id === cameraId);
        camera && witnesses.push(camera);
      }
      witnesses.length > 0 && setAssignedCameras(witnesses);
    }
  }, [camerasData, assignedIds]);

  function removeWitness(id: string) {
    const assignedIds = assignedCameras.filter(camera => camera.id !== id).map(camera => camera.id);
    onChange(assignedIds);
    setAssignedCameras(prevCameras => (prevCameras.filter(camera => camera.id !== id)));
  }

  function getSelectedWitnesses(witnesses: DeviceWitness[]) {
    const assignedIds = [...assignedCameras, ...witnesses].map(camera => camera.id);
    onChange(assignedIds);
    setAssignedCameras(prevCameras => ([...prevCameras, ...witnesses]));
  }

  function changeSelectedWitnessesPriority(dragIndex: number, hoverIndex: number): void {
    const device = assignedCameras[dragIndex];
    const newDevices = Array.from(assignedCameras);
    newDevices.splice(dragIndex, 1);
    newDevices.splice(hoverIndex, 0, device);
    setAssignedCameras(newDevices);
    onChange(newDevices.map(dev => dev.id));
  }

  const assignedCameraIds = assignedCameras.map(cam => cam.id);
  const camerasToAssign = assignedCameraIds.length === 0 ? cameras : cameras.filter(camera => !assignedCameraIds.includes(camera.id));
  const idsToAssign = camerasToAssign.map(camera => camera.id);
  const hiddenIds = [...assignedCameraIds, ...(camerasData?.devicesByAspectTypes.filter(cam => !idsToAssign.includes(cam.id)).map(cam => cam.id) || [])];

  return (
    <Segment loading={cameraLoading} error={cameraError} className="SensorWitnesses">
      <AssignedWitnesses
        witnesses={assignedCameras}
        getWitnessesToAssign={() => setCamerasToAssignVisible(true)}
        removeWitness={removeWitness}
        moveWitness={changeSelectedWitnessesPriority}
      />
      { camerasToAssignVisible &&
        <WitnessesToAssign
          hiddenIds={hiddenIds}
          witnesses={camerasToAssign}
          getSelectedWitnesses={getSelectedWitnesses}
          onCancel={() => setCamerasToAssignVisible(false)}
        /> }
    </Segment>
  );
};

export default SensorWitnesses;
