import React, { useEffect, useImperativeHandle, useMemo, useRef, useState } from "react";
import { Segment } from "semantic-ui-react";
import { ApolloError } from "@apollo/client";

import { __ } from "@solid/libs";
import { AspectSensorCategory, AspectSensorModel, AuthType, DeviceBaseConfigType, DeviceFunctionalAspectType, DevicesByAspectTypesShortQuery, Dfa_Sensor, ZonesQuery } from "@generated/graphql";
import { useAccessability } from "@core/store/actions/accessability";

import { AutoForm, AutoLayout, FieldSchema, FieldValues, FormSchema } from "components/AutoForm";
import { ComponentWithFormRef, DeviceProperties } from "components/Admin/Sensors/SensorSettings";

import "./style.css";

type SensorPropertiesProps = {
  sensor?: DeviceProperties;
  zoneData?: ZonesQuery;
  avatarData?: DevicesByAspectTypesShortQuery;
  error?: ApolloError;
  onChange: (property: DeviceProperties) => void;
};

const generalFields = ["destination", "platformId", "name", "enabled", "location", "zoneId"];
const connectFields = ["host", "port", "user", "pass"];
const configFields = ["model"];
const descriptionFields = ["category"];

const SensorProperties = React.forwardRef<ComponentWithFormRef, SensorPropertiesProps>(({ sensor, zoneData, avatarData, error, onChange }, ref) => {
  const { config: accessConfig } = useAccessability();
  const [isCloudDirect, setIsCloudDirect] = useState(false);

  const formRef = useRef<AutoForm>(null);

  const schema = useMemo(() => {
    const devicePlatformId = sensor?.platform ? sensor.platform.id : "";
    const devicePlatformName = sensor?.platform ? sensor.platform.name : "";

    const destinationDataSource = sensor && !sensor.platform ?
      [{name: "Cloud Direct", id: "cloudDirect"}] :
      [{name: "Avatar", id: "avatar"}, {name: "Cloud Direct", id: "cloudDirect"}];

    const platformIdDataSource = isCloudDirect ?
      [{id: "", name: ""}] :
      sensor ?
        Array.from([{id: devicePlatformId, name: devicePlatformName}]) :
        Array.from(avatarData?.devicesByAspectTypes ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}));

    const schema: FormSchema = [
      {
        name: "destination",
        label: __("Destination"),
        type: "dropdown",
        required: true,
        dataSource: Array.from(destinationDataSource ?? []),
        disableCondition: () => !!sensor
      },
      {
        name: "platformId",
        label: __("Avatar"),
        type: "dropdown",
        required: true,
        dataSource: Array.from(platformIdDataSource ?? []),
        disableCondition: () => !!sensor?.platform?.id,
        hideCondition: values => values["destination"] === "cloudDirect",
      },
      {
        name: "name",
        label: __("Name"),
        required: true
      },
      {
        name: "enabled",
        label: __("State"),
        type: "dropdown",
        required: true,
        dataSource: [
          {id: "on", name: __("On")},
          {id: "off", name: __("Off")},
        ]
      },
      {
        name: "location",
        label: __("Location"),
      },
      !accessConfig?.limitedZonesAccess ?
        {
          name: "zoneId",
          label: __("Zone"),
          type: "dropdown",
          dataSource: Array.from(zoneData?.zones ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}))
        } :
        undefined,
      {
        name: "category",
        label: __("Category"),
        type: "dropdown",
        required: true,
        dataSource: Array.from(Object.values(AspectSensorCategory)).map((category) => ({id: category, name: category}))
      },
      {
        name: "model",
        label: __("Model"),
        type: "dropdown",
        required: true,
        disableCondition: () => true,
        dataSource: Array.from(Object.values(AspectSensorModel)).map((model) => ({id: model, name: model}))
      },
      {
        name: "host",
        label: __("Sensor IP/Host"),
        required: true,
      },
      {
        name: "port",
        label: __("HTTP Port"),
        type: "integer",
        required: true,
        minValue: 1,
        maxValue: 65535,
      },
      {
        name: "user",
        label: __("Username")
      },
      {
        name: "pass",
        label: __("Password"),
        type: "password",
      },
    ];

    if (!accessConfig?.limitedZonesAccess) {
      const zonesDropdown: FieldSchema = {
        name: "zoneId",
        label: __("Zone"),
        type: "dropdown",
        dataSource: Array.from(zoneData?.zones ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, {sensitivity: "base"}))
      };
      schema.splice(4, 0, zonesDropdown);
    }

    return schema;
  }, [avatarData, zoneData, sensor, accessConfig, isCloudDirect]);

  const values = useMemo(() => {
    if (sensor) {
      const { name, enabled, location, zone, platform, config, aspects } = sensor;
      const sensorCategory = (aspects?.find(aspect => aspect.type === DeviceFunctionalAspectType.Sensor) as Dfa_Sensor)?.category || AspectSensorCategory.None;

      return {
        name,
        enabled: enabled ? "on" : "off",
        location,
        zoneId: zone?.id ?? "",
        platformId: platform?.id,
        host: config?.connect.host,
        port: config?.connect.port || 80,
        user: config?.connect.user,
        pass: config?.connect.pass,
        model: config?.make,
        category: sensorCategory
      };
    }

    return {
      enabled: "on",
      platformId: "",
      port: 80,
      category: AspectSensorCategory.None,
      model: AspectSensorModel.Hikvision
    };
  }, [sensor]);

  useEffect(() => {
    const values = getValues();
    values && onChange(values);
  }, [values]);

  useImperativeHandle(ref, () => ({
    getFormRef() {
      return formRef.current;
    }
  }));

  function onFormChange(name: string, value: any, values: FieldValues, form: AutoForm) {
    form.setValue(name, value);

    if (name === "destination" && value === "cloudDirect") {
      setIsCloudDirect(true);
      form.setValue("platformId", "");
      onChange({ platform: { id: "", name: "", aspects: [] } });
      return;
    }
    if (name === "destination" && value === "avatar") {
      setIsCloudDirect(false);
    }
    if (name === "platformId") {
      const platform = avatarData?.devicesByAspectTypes.find(avatar => avatar.id === value);
      platform && onChange({ platform: { id: platform.id, name: platform.name, aspects: platform.aspects } });
      return;
    }
    if ([...configFields, ...connectFields].includes(name)) {
      const config = getConfigProperties(values);
      onChange(config);
      return;
    }
    if (name === "enabled") {
      onChange({ enabled: value === "on" });
      return;
    }
    if (name === "zoneId") {
      const zone = zoneData?.zones.find(zone => zone.id === value);
      if (zone) {
        onChange({ zone: { ...zone, __typename: "ObjectDescriptor" }});
      }
      else {
        onChange({ zone: null });
      }
      return;
    }

    onChange({ [name]: value });
  }

  function getValues(): DeviceProperties | undefined {
    if (!formRef.current) return undefined;
    const values = formRef.current.getValues();
    const platform = avatarData?.devicesByAspectTypes.find(avatar => avatar.id === values["platformId"]);
    const config = getConfigProperties(values);

    const device: DeviceProperties = {
      name: values["name"],
      enabled: values["enabled"] === "on",
      location: values["location"],
      ...config,
    };

    if (platform) {
      device.platform = { id: platform.id, name: platform.name, aspects: platform.aspects };
    }

    return device;
  }

  return (
    <Segment error={error} className="SensorProperties">
      <AutoForm ref={formRef} schema={schema} values={values} onChange={onFormChange} onValidChange={() => !!formRef.current?.getIsValid()}>
        <div className="SensorProperties-FormLayout">
          <div className="SensorProperties-FormColumn1">
            <AutoLayout names={generalFields}/>
          </div>
          <div className="SensorProperties-FormColumn2">
            <AutoLayout names={[...descriptionFields, ...configFields, ...connectFields]}/>
          </div>
        </div>
      </AutoForm>
    </Segment>
  );
});

export default SensorProperties;

function getConfigProperties(values: FieldValues) {
  const newConfigValues = { config: {
    configType: DeviceBaseConfigType.Sensor,
    make: values["model"],
    connect: {
      authType: AuthType.NoAuth
    }
  } };

  for (const field of connectFields) {
    newConfigValues.config.connect[field] = values[field];
  }

  return newConfigValues;
}
