import React from "react";
import { Segment } from "semantic-ui-react";
import { AutoSizer, Column, Table } from "react-virtualized";

import { healthStatusText, SensorShort } from "@core/actions";
import { __ } from "@solid/libs";
import { UUID } from "@solid/types";
import { useAccessability } from "@core/store/actions/accessability";

import TableActions from "./TableActions";

type SensorTableProps = {
  list: SensorShort[];
  loading?: boolean;
  createdDeviceId?: UUID;
  onSelect: (device: SensorShort) => void;
  onDelete: (device: SensorShort) => void;
};

const SensorTable = ({ list, loading, createdDeviceId, onSelect, onDelete }: SensorTableProps) => {
  const { config: accessConfig } = useAccessability();

  return (
    <Segment loading={loading} className="SensorTable-Wrapper">
      <AutoSizer style={{ width: "100%", height: "100%" }}>
        {({ width, height }) => (
          <Table
            className="SensorTable"
            width={width}
            height={height}
            rowHeight={34}
            headerHeight={58}
            rowCount={list.length}
            rowGetter={({ index }) => list[index]}
            rowClassName={({ index }) => list[index] && createdDeviceId === list[index].id ? "added" : ""}
          >
            <Column
              dataKey="actions"
              label=""
              width={60}
              flexShrink={0}
              cellRenderer={({ rowData }: { rowData: SensorShort }) => <TableActions onSelect={onSelect} onDelete={onDelete} sensor={rowData} />}
            />
            <Column
              dataKey="name"
              label={__("Name")}
              width={100}
              flexGrow={1}
            />
            <Column
              dataKey="location"
              label={__("Location")}
              width={100}
              flexGrow={1}
            />
            { !accessConfig?.limitedZonesAccess &&
            <Column
              dataKey="zone"
              label={__("Zone")}
              width={100}
              flexGrow={1}
              cellDataGetter={({ rowData }: { rowData: SensorShort }) => rowData.zone?.name ?? ""}
            /> }
            <Column
              dataKey="enabled"
              label={__("State")}
              width={60}
              cellDataGetter={({ rowData }: { rowData: SensorShort }) => getSensorState(rowData.enabled)}
            />
            <Column
              dataKey="healthStatus"
              label={__("Status")}
              width={80}
              cellDataGetter={({ rowData }: { rowData: SensorShort }) => healthStatusText[rowData.healthStatus]}
            />
            <Column
              dataKey="platform"
              label={__("Avatar")}
              width={100}
              flexGrow={1}
              cellDataGetter={({ rowData }: { rowData: SensorShort }) => rowData.platform?.name ?? ""}
            />
          </Table>
        )}
      </AutoSizer>
    </Segment>
  );
};

export default SensorTable;

function getSensorState(enabled: boolean): any {
  return enabled ? __("On") : __("Off");
}
