import React, { useState, useEffect } from "react";
import { useNavigate, useParams, useLocation } from "react-router-dom";
import { Segment, Table, Icon, Popup, Input, Modal, Button, Header } from "semantic-ui-react";
import {useApolloClient} from "@apollo/client";
import {usePoliciesQuery, PoliciesQuery, useDeletePolicyMutation, PoliciesDocument, GroupsDocument, GroupsQuery} from "generated/graphql";
import { PropType } from "utils";
import WithQueryStatus from "components/WithQueryStatus";
import IconButton from "components/IconButton";
import Loading from "components/Loading";
import CreateUpdatePolicy from "components/Admin/Policies/CreateUpdatePolicy";
import ListText from "components/Admin/Helpers/ListText";
import { WidgetProps } from "components/Widgets";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import HelpMD from "./help.md";
import produce from "immer";
import { RouteParams } from "@core/types";
import { UUID } from "@solid/types";

import "./style.css";

type PoliciesProps = WidgetProps;

type Policy = PropType<PoliciesQuery, "policies">[0];

const Policies = (props: PoliciesProps) => {
  const client = useApolloClient();
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const { viewId = "", policyId = "" } = useParams<RouteParams>();
  const { cellProps, setCellProps } = props;
  const { data, error, loading } = usePoliciesQuery();
  const [policies, setPolicies] = useState<Policy[]>([]);
  const [searchText, setSearchText] = useState("");
  const [selectedPolicyId, setSelectedPolicyId] = useState("");
  const selectedPolicy = selectedPolicyId ? policies.find(policy => policy.id === selectedPolicyId) : undefined;
  const [createUpdateOpen, setCreateUpdateOpen] = useState(false);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [deletePolicy, { error: deleteError, loading: deleteLoading }] = useDeletePolicyMutation();
  const [updateDefaultTab, setUpdateDefaultTab] = useState(0);

  useEffect(() => {
    pathname.includes("/view/policy/add") && onCreateClick();
  }, [pathname]);

  useEffect(() => {
    if (!policyId || !data) return;

    const isPolicyExist = data.policies.some(policy => policy.id === policyId);
    isPolicyExist && onUpdateClick(policyId);
  }, [data, policyId]);

  useEffect(() => {
    if (viewId && createUpdateOpen && !deleteConfirmOpen) {
      setCreateUpdateOpen(false);
    }
  }, [viewId]);

  useEffect(() => {
    if (data) {
      const policies = Array.from(data.policies)
        .filter(policy => !searchText || policy.name.toLocaleUpperCase().includes(searchText.toLocaleUpperCase()))
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
      setPolicies(policies);
    }
  }, [data, searchText]);

  useEffect(() => {
    if (deleteError) {
      Log.error(`Could not delete policy: ${deleteError.message}`);
    }
  }, [deleteError]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }
    let title = cellProps?.title ?? "";
    if (createUpdateOpen) {
      title = selectedPolicy ? __("Edit Policy") : __("Create Policy");
    }
    setCellProps({ title });
  }, [createUpdateOpen, selectedPolicy]);

  function onCreateClick(): void {
    setSelectedPolicyId("");
    setUpdateDefaultTab(0);
    setCreateUpdateOpen(true);
    !pathname.includes("/view/policy/add") && navigate("/view/policy/add");
  }

  function onUpdateClick(policyId: UUID): void {
    setSelectedPolicyId(policyId);
    setUpdateDefaultTab(0);
    setCreateUpdateOpen(true);
    !pathname.includes(`/view/policy/edit/${policyId}`) && navigate(`/view/policy/edit/${policyId}`);
  }

  function onDeleteClick(policy: Policy): void {
    setSelectedPolicyId(policy.id);
    setDeleteConfirmOpen(true);
  }

  async function onDeletePolicy(): Promise<void> {
    setDeleteConfirmOpen(false);
    if (!selectedPolicy) {
      return;
    }

    const policyId = selectedPolicy.id;
    const response = await deletePolicy({
      variables: { id: policyId }
    });

    // update cache
    // update PoliciesDocument and GroupsDocument
    if (response.data?.deletePolicy) {
      const policiesCacheData = client.readQuery<PoliciesQuery>({
        query: PoliciesDocument
      });

      const policiesCache = policiesCacheData?.policies ?? [];
      const policyIndex = policiesCache.findIndex((policy) => policy.id === policyId);

      const policies = produce(policiesCache, (draft) => {
        draft.splice(policyIndex, 1);
      });

      client.writeQuery<PoliciesQuery>({
        query: PoliciesDocument,
        data: {
          policies
        }
      });

      const groupsCacheData = client.readQuery<GroupsQuery>({
        query: GroupsDocument
      });

      const groupsCache = groupsCacheData?.groups ?? [];
      let policyInGroupIndex = -1;
      const groupIndex = groupsCache.findIndex((group) => {
        const index = group.policies.findIndex((policy) => policy.id === policyId);
        if (index !== -1) {
          policyInGroupIndex = index;
        }
        return index !== -1;
      });

      if (groupIndex !== -1 && policyInGroupIndex !== -1) {
        const groups = produce(groupsCache, (draft) => {
          draft[groupIndex].policies.splice(policyInGroupIndex, 1);
        });

        client.writeQuery<GroupsQuery>({
          query: GroupsDocument,
          data: {
            groups
          }
        });
      }
    }
  }

  function onBack(policyId?: string): void {
    if (policyId) {
      setSelectedPolicyId(policyId);
      setUpdateDefaultTab(1);
      setCreateUpdateOpen(true);
      navigate(`/view/policy/edit/${policyId}`, {replace: true});
      return;
    }
    navigate(-1);
  }

  return !createUpdateOpen ?
    <Segment className="Policies">
      <WithQueryStatus loading={loading} error={error}>
        <div className="Policies-Content">
          <div className="Policies-Top">
            <Input
              placeholder={__("Filter by name")}
              icon="search"
              value={searchText}
              onChange={e => setSearchText(e.currentTarget.value)}
            />
            <IconButton
              icon="plus"
              hint={__("Create Policy")}
              onClick={onCreateClick}
            />
          </div>

          <div className="Policies-Data">
            <Table celled compact>
              <Table.Header>
                <Table.Row>
                  <Table.HeaderCell/>
                  <Table.HeaderCell>{__("Name")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Statements")}</Table.HeaderCell>
                </Table.Row>
              </Table.Header>

              <Table.Body>
                {policies.map(policy =>
                  <Table.Row key={policy.id}>
                    <Table.Cell collapsing>
                      <Popup trigger={
                        <Icon name="edit" className="Policies-IconButton" onClick={() => onUpdateClick(policy.id)}/>
                        }
                        content={__("Edit")}
                      />
                      <Popup trigger={
                        <Icon name="trash alternate" className="Policies-IconButton" onClick={() => onDeleteClick(policy)}/>
                        }
                        content={__("Delete")}
                      />
                    </Table.Cell>
                    <Table.Cell>
                      {policy.name}
                    </Table.Cell>
                    <Table.Cell>
                      <ListText items={policy.statements} maxItems={3} modalHeader={__("Statements")}/>
                    </Table.Cell>
                  </Table.Row>)}

                {policies.length === 0 &&
                <Table.Row>
                  <Table.Cell colSpan={3} textAlign="center">
                    {__("No policies found")}
                  </Table.Cell>
                </Table.Row>}
              </Table.Body>
            </Table>
          </div>
        </div>

        <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
          <Header>{__("Delete Policy")}</Header>
          <Modal.Content>{__("Are you sure you want to delete policy '{{policyName}}'?", {policyName: selectedPolicy?.name})}</Modal.Content>
          <Modal.Actions>
            <Button negative onClick={() => onDeletePolicy()}>
              <Icon name="trash"/>{__("Delete")}
            </Button>
            <Button onClick={() => setDeleteConfirmOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {deleteLoading && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment> :
    <CreateUpdatePolicy {...props} policy={selectedPolicy} defaultTab={updateDefaultTab} onBack={onBack}/>;
};

export default Policies;
