import React, { useState, useEffect, useRef, useMemo } from "react";
import { Segment, Form, Button, Icon, Message, Popup, Tab, Table, Modal, Header } from "semantic-ui-react";
import {
  PoliciesQuery,
  usePoliciesQuery,
  useActionsQuery,
  useCreatePolicyMutation,
  useUpdatePolicyMutation,
  useDeletePolicyStatementMutation,
  PoliciesDocument,
  PolicyStatement,
  StatementEffect,
  Action,
  RealmObjectType,
  DeviceFunctionalAspectType
} from "generated/graphql";
import { PropType } from "utils";
import WithQueryStatus from "components/WithQueryStatus";
import Loading from "components/Loading";
import { WidgetProps } from "components/Widgets";
import { compareResourceTypes, levelView, levelManage } from "../utils";
import CreateUpdateStatement from "../CreateUpdateStatement";
import ListText, { ListItem } from "components/Admin/Helpers/ListText";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import HelpMD from "./help.md";
import {useApolloClient} from "@apollo/client";
import produce from "immer";
import type {ArrayElement} from "@solid/types";

import "./style.css";

type CreateUpdatePolicyProps = WidgetProps & {
  policy?: Policy;
  defaultTab?: number;
  onBack?: (policyId?: string) => void;
};

type Policy = PropType<PoliciesQuery, "policies">[0];

const CreateUpdatePolicy = ({ policy, defaultTab = 0, onBack, setCellProps }: CreateUpdatePolicyProps) => {
  const client = useApolloClient();
  const { data: policiesData, error: policiesError, loading: policiesLoading } = usePoliciesQuery();
  const { data: actionsData, error: actionsError, loading: actionsLoading } = useActionsQuery();
  const [createPolicy, { error: createError, loading: createLoading }] = useCreatePolicyMutation();
  const [updatePolicy, { error: updateError, loading: updateLoading }] = useUpdatePolicyMutation();
  const [deleteStatement, { error: deleteError, loading: deleteLoading }] = useDeletePolicyStatementMutation();
  const [name, setName] = useState(policy?.name ?? "");
  const [nameError, setNameError] = useState("");
  const [statementId, setStatementId] = useState("");
  const statement = policy?.statements.find(stmt => stmt.id === statementId);
  const [createUpdateOpen, setCreateUpdateOpen] = useState(false);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [activeTab, setActiveTab] = useState(defaultTab);
  const [changesDialogOpen, setChangesDialogOpen] = useState(false);
  const backOnUpdateRef = useRef(false);
  const nextTabRef = useRef(0);
  const prevTabRef = useRef(-1);
  const tabCount = policy ? 2 : 1;
  const actionMap = useMemo(() => {
    const map = new Map<string, Action>();
    if (actionsData) {
      for (const action of actionsData.actions) {
        map.set(action.action, action);
      }
    }
    return map;
  }, [actionsData]);

  useEffect(() => {
    setActiveTab(defaultTab);
  }, [defaultTab]);

  useEffect(() => {
    setName(policy?.name ?? "");
  }, [policy]);

  useEffect(() => {
    if (updateError && prevTabRef.current >= 0) {
      setActiveTab(prevTabRef.current);
      prevTabRef.current = -1;
    }
  }, [updateError]);

  useEffect(() => {
    if (deleteError) {
      Log.error(`Could not delete statement: ${deleteError.message}`);
    }
  }, [deleteError]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }
    let title = __("Edit Policy");
    if (createUpdateOpen) {
      title = statement ? __("Edit Statement") : __("Create Statement");
    }
    setCellProps({ title });
  }, [createUpdateOpen, statement]);

  const cancel = () => {
    onBack && onBack();
  };

  function getNameError(name: string): string {
    let error = "";
    const policyId = policy?.id;
    if (!name.trim()) {
      error = __("Policy name should be not empty");
    }
    else if (policiesData?.policies.find(policy => policy.id !== policyId && policy.name.toLocaleUpperCase() === name.trim().toLocaleUpperCase())) {
      error = __("Policy with the same name already exists");
    }
    return error;
  }

  function onNameInput(e: React.FormEvent<HTMLInputElement>): void {
    const name = e.currentTarget.value;
    setName(name);
    setNameError(getNameError(name));
  }

  async function onCreateUpdate(backOnUpdate = false): Promise<void> {
    const policyName = name.trim();

    if (!policy) {
      for (let i = 0; i < tabCount; i++) {
        if (!validateTab(i)) {
          setActiveTab(i);
          return;
        }
      }

      const response = await createPolicy({
        variables: { policy: { name: policyName } }
      });

      const policyId = response.data?.createPolicy.id;

      // update cache
      if (policyId) {
        const data = client.readQuery<PoliciesQuery>({
          query: PoliciesDocument
        });

        const newPolicy: ArrayElement<PoliciesQuery["policies"]> = {
          __typename: "Policy",
          id: policyId,
          isSystemManaged: false,
          name: policyName,
          statements: []
        };

        client.writeQuery<PoliciesQuery>({
          query: PoliciesDocument,
          data: {
            policies: [
              ...(data?.policies ?? []),
              newPolicy
            ]
          }
        });

        onBack && onBack(policyId);
      }
    } else {
      if (!validateTab(activeTab)) {
        return;
      }

      backOnUpdateRef.current = backOnUpdate;
      prevTabRef.current = activeTab;

      switch (activeTab) {
        case 0:
          const policyId = policy.id;
          const response = await updatePolicy({
            variables: { id: policyId, policy: { name: policyName } }
          });

          // update cache
          if (response.data?.updatePolicy) {
            const data = client.readQuery<PoliciesQuery>({
              query: PoliciesDocument
            });

            const policiesCache = data?.policies ?? [];
            const index = policiesCache.findIndex((policy) => {
              return policy.id === policyId;
            });

            const policies = produce(policiesCache, (draft) => {
              draft[index].name = policyName;
            });

            client.writeQuery<PoliciesQuery>({
              query: PoliciesDocument,
              data: {
                policies
              }
            });

            prevTabRef.current = -1;
            backOnUpdateRef.current && onBack && onBack();
          }
          break;
      }
    }
  }

  function onActiveTabChange(index: number | string | undefined): void {
    if (typeof index !== "number") {
      return;
    }

    if (!policy) {
      if (index < activeTab || validateTab(activeTab)) {
        setActiveTab(index);
      }
    }
    else {
      if (!hasChangesOnTab(activeTab)) {
        setActiveTab(index);
        return;
      }

      if (!validateTab(activeTab)) {
        return;
      }

      nextTabRef.current = index;
      setChangesDialogOpen(true);
    }
  }

  function hasChangesOnTab(tabIndex: number): boolean {
    if (!policy) {
      switch (tabIndex) {
        case 0: return !!name.trim();
        default: return false;
      }
    }
    else {
      switch (tabIndex) {
        case 0: return name.trim() !== policy.name;
        default: return false;
      }
    }
  }

  function validateTab(tabIndex: number): boolean {
    let result = true;
    switch (tabIndex) {
      case 0:
        const nameError = getNameError(name);
        if (nameError) {
          setNameError(nameError);
          result = false;
        }
        break;
    }
    return result;
  }

  function discardTabChanges(tabIndex: number): void {
    switch (tabIndex) {
      case 0:
        setName(policy?.name ?? "");
        setNameError("");
        break;
    }
  }

  function getActionsContent(stmt: PolicyStatement): React.ReactNode {
    if (stmt.actions.includes("**")) {
      return __("All");
    }

    let items: ListItem[] = stmt.actions
      .map(actionId => {
        const action = actionMap.get(actionId);
        return { id: actionId, name: action?.description ?? actionId };
      })
      .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));

    if (actionsData) {
      const deviceActions = actionsData.actions.filter(action =>
        action.resourceTypes.includes(RealmObjectType.Device) &&
        action.deviceFunctionalAspectTypes.includes(DeviceFunctionalAspectType.Media));

      const manageActions = deviceActions.filter(action => action.accessLevel.level === levelManage);
      const manageAll = manageActions.every(action => stmt.actions.includes(action.action));
      if (manageAll) {
        items = items.filter(item => !manageActions.some(action => action.action === item.id));
        items.unshift({ id: levelManage, name: __("Manage") });
      }

      const viewActions = deviceActions.filter(action => action.accessLevel.level === levelView);
      const viewAll = viewActions.every(action => stmt.actions.includes(action.action));
      if (viewAll) {
        items = items.filter(item => !viewActions.some(action => action.action === item.id));
        items.unshift({ id: levelView, name: __("View") });
      }
    }

    return <ListText items={items} maxItems={5} modalHeader={__("Actions")} sort={false}/>;
  }

  function getResourcesContent(stmt: PolicyStatement): React.ReactNode {
    if (stmt.allResources) {
      return __("All");
    }

    const items = Array.from(stmt.resources)
      .sort((a, b) => compareResourceTypes(a.type, b.type) || a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
      .map(resource => ({
        id: resource.id,
        name: resource.name,
        icon: resource.type === RealmObjectType.Zone ? "th large" : undefined,
        faIcon: resource.type === RealmObjectType.Device ? "video" : undefined,
      } as ListItem));

    return <ListText items={items} maxItems={5} modalHeader={__("Resources")} sort={false} icons/>;
  }

  function onCreateClick(): void {
    setStatementId("");
    setCreateUpdateOpen(true);
  }

  function onUpdateClick(stmt: PolicyStatement): void {
    setStatementId(stmt.id);
    setCreateUpdateOpen(true);
  }

  function onDeleteClick(stmt: PolicyStatement): void {
    setStatementId(stmt.id);
    setDeleteConfirmOpen(true);
  }

  async function onDeleteStatement(): Promise<void> {
    setDeleteConfirmOpen(false);
    if (!statement) {
      return;
    }

    const statementId = statement.id;

    const response = await deleteStatement({
      variables: { statementId }
    });

    // update cache
    if (response.data?.deletePolicyStatement) {
      const data = client.readQuery<PoliciesQuery>({
        query: PoliciesDocument
      });

      const policiesCache = data?.policies ?? [];
      let statementIndex = -1;
      const policyIndex = policiesCache.findIndex((policy) => {
        const index = policy.statements.findIndex((statement) => statement.id === statementId);
        if (index !== -1) {
          statementIndex = index;
        }
        return index !== -1;
      });

      if (policyIndex !== -1 && statementIndex !== -1) {
        const policies = produce(policiesCache, (draft) => {
          draft[policyIndex].statements.splice(statementIndex, 1);
        });

        client.writeQuery<PoliciesQuery>({
          query: PoliciesDocument,
          data: {
            policies
          }
        });
      }
    }
  }

  const error = createError ?? updateError;

  const panes = [
    {
      menuItem: __("Policy Properties"),
      render: () => (
        <Tab.Pane>
          <Form className="CreateUpdatePolicy-Form" onSubmit={e => { e.preventDefault(); }}>
            <Form.Field
              control={Form.Input}
              label={__("Name")}
              placeholder={__("Name")}
              autoFocus
              value={name}
              error={nameError ? { content: nameError, pointing: "below" } : undefined}
              readOnly={policy?.isSystemManaged}
              onInput={onNameInput}
            />
          </Form>
        </Tab.Pane>
      )
    },
    {
      menuItem: __("Statements"),
      render: () => (
        <Tab.Pane>
          <div className="CreateUpdatePolicy-Statements">
            {!!policy && !policy.isSystemManaged &&
            <div className="CreateUpdatePolicy-TopTableButtons">
              <Button onClick={onCreateClick}>
                {__("Create Statement")}
              </Button>
            </div>}
            <div className="CreateUpdatePolicy-Table">
              <Table celled compact>
                <Table.Header>
                  <Table.Row>
                    <Table.HeaderCell/>
                    <Table.HeaderCell>{__("Name")}</Table.HeaderCell>
                    <Table.HeaderCell>{__("Effect")}</Table.HeaderCell>
                    <Table.HeaderCell>{__("Actions")}</Table.HeaderCell>
                    <Table.HeaderCell>{__("Resources")}</Table.HeaderCell>
                  </Table.Row>
                </Table.Header>

                <Table.Body>
                  {Array.from(policy?.statements ?? [])
                    .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }))
                    .map(stmt =>
                      <Table.Row key={stmt.id}>
                        <Table.Cell collapsing>
                          {!policy?.isSystemManaged &&
                          <>
                            <Popup trigger={
                              <Icon name="edit" className="CreateUpdatePolicy-IconButton" onClick={() => onUpdateClick(stmt)}/>
                              }
                              content={__("Edit")}
                            />
                            <Popup trigger={
                              <Icon name="trash alternate" className="CreateUpdatePolicy-IconButton" onClick={() => onDeleteClick(stmt)}/>
                              }
                              content={__("Delete")}
                            />
                          </>}
                        </Table.Cell>
                        <Table.Cell>
                          {stmt.name}
                        </Table.Cell>
                        <Table.Cell>
                          {stmt.effect === StatementEffect.Allow ? __("Allow") : __("Deny")}
                        </Table.Cell>
                        <Table.Cell>
                          {getActionsContent(stmt)}
                        </Table.Cell>
                        <Table.Cell>
                          {getResourcesContent(stmt)}
                        </Table.Cell>
                      </Table.Row>)}

                  {policy?.statements.length === 0 &&
                  <Table.Row>
                    <Table.Cell colSpan={5} textAlign="center">
                      {__("No statements defined")}
                    </Table.Cell>
                  </Table.Row>}
                </Table.Body>
              </Table>
            </div>
          </div>
        </Tab.Pane>
      )
    }
  ];

  if (!policy) {
    panes.pop();
  }

  return (
    <Segment className="CreateUpdatePolicy">
      <WithQueryStatus error={policiesError || actionsError} loading={policiesLoading || actionsLoading}>
        {!!error && <Message error content={error.message}/>}

        {!createUpdateOpen ?
          <>
            <div className="CreateUpdatePolicy-TopButtons">
              <Button onClick={() => cancel()}>
                <Icon name="arrow left"/>{__("Back")}
              </Button>
              {activeTab === 0 && !policy?.isSystemManaged &&
              <Button
                positive
                disabled={!!policy && !hasChangesOnTab(activeTab)}
                onClick={() => onCreateUpdate(true)}
              >
                <Icon name={!policy ? "plus" : "check"}/>{!policy ? __("Create") : __("Save")}
              </Button>}
            </div>

            <Tab
              panes={panes}
              className="CreateUpdatePolicy-Tab"
              activeIndex={activeTab}
              onTabChange={(e, { activeIndex }) => onActiveTabChange(activeIndex)}
            />
          </> :
          <>
            {!!policy && <CreateUpdateStatement policy={policy} statement={statement} onBack={() => setCreateUpdateOpen(false)}/>}
          </>}

        <Modal open={changesDialogOpen} onClose={() => setChangesDialogOpen(false)}>
          <Header>{__("Unsaved Changes")}</Header>
          <Modal.Content>{__("Policy has unsaved changes. Would you like to save the changes?")}</Modal.Content>
          <Modal.Actions>
            <Button positive onClick={async () => {
              await onCreateUpdate();
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="check"/>{__("Save")}
            </Button>
            <Button negative onClick={() => {
              discardTabChanges(activeTab);
              setActiveTab(nextTabRef.current);
              setChangesDialogOpen(false);
            }}>
              <Icon name="undo"/>{__("Discard")}
            </Button>
            <Button onClick={() => setChangesDialogOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
          <Header>{__("Delete Statement")}</Header>
          <Modal.Content>{__("Are you sure you want to delete statement '{{statementName}}'?", {statementName: statement?.name})}</Modal.Content>
          <Modal.Actions>
            <Button negative onClick={() => onDeleteStatement()}>
              <Icon name="trash"/>{__("Delete")}
            </Button>
            <Button onClick={() => setDeleteConfirmOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {(createLoading || updateLoading || deleteLoading) && <Loading text={__("Updating...")}/>}

        {!createUpdateOpen && <Help markdown={HelpMD}/>}
      </WithQueryStatus>
    </Segment>
  );
};

export default CreateUpdatePolicy;
