import React, { useState, useEffect, useRef } from "react";
import { Segment, Message, Table, Button, Icon } from "semantic-ui-react";
import { MessageId } from "electron/types";
import { LinkInfo, LinkAddArgs, LinkAddReplyArgs } from "electron/linkProvision";
import { isElectron } from "@solid/libs/utils";
import Loading from "components/Loading";
import { removeTrailingSlash } from "utils";
import {API} from "@solid/libs/api";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import type Electron from "electron";
import { useNavigate } from "react-router-dom";

import "./style.css";

const api = new API();

enum AvatarState {
  None = "",
  Discarded = "discarded",
  Virgin = "virgin",
  Active = "active"
}

type Link = LinkInfo & {
  avatarState: AvatarState;
};

const LinkProvision = () => {
  const [links, setLinks] = useState<Link[]>([]);
  const [refreshing, setRefreshing] = useState(false);
  const [updating, setUpdating] = useState(false);
  const refreshTimeoutRef = useRef<NodeJS.Timeout | undefined>();
  const endpoint = getCurrentEndpoint();
  const navigate = useNavigate();

  useEffect(() => {
    window.ipcRenderer?.on(MessageId.LinkProvisionFound, onLinkFound);
    window.ipcRenderer?.on(MessageId.LinkProvisionError, onLinkError);
    window.ipcRenderer?.on(MessageId.LinkProvisionAddReply, onAddLinkReply);

    return () => {
      window.ipcRenderer?.off(MessageId.LinkProvisionFound, onLinkFound);
      window.ipcRenderer?.off(MessageId.LinkProvisionError, onLinkError);
      window.ipcRenderer?.off(MessageId.LinkProvisionAddReply, onAddLinkReply);
    };
  });

  useEffect(() => {
    window.ipcRenderer?.send(MessageId.LinkProvisionStart);
    setRefreshing(true);

    return () => {
      window.ipcRenderer?.send(MessageId.LinkProvisionStop);
      if (refreshTimeoutRef.current) {
        clearTimeout(refreshTimeoutRef.current);
        refreshTimeoutRef.current = undefined;
      }
    };
  }, []);

  useEffect(() => {
    if (refreshTimeoutRef.current) {
      clearTimeout(refreshTimeoutRef.current);
      refreshTimeoutRef.current = undefined;
    }
    if (refreshing) {
      refreshTimeoutRef.current = setTimeout(() => setRefreshing(false), 5000);
    }
  }, [refreshing]);

  function onLinkFound(event: Electron.IpcRendererEvent, link: LinkInfo): void {
    getAvatarState(link).then(avatarState => {
      updateLink({ ...link, avatarState });
      setRefreshing(false);
    });
  }

  async function getAvatarState(link: LinkInfo): Promise<AvatarState> {
    let avatarState = AvatarState.None;
    try {
      const { state } = await api.getAvatarKeyState({ obj: link.status?.uuid });
      switch (state) {
        case AvatarState.Discarded: avatarState = AvatarState.Discarded; break;
        case AvatarState.Virgin: avatarState = AvatarState.Virgin; break;
        case AvatarState.Active: avatarState = AvatarState.Active; break;
      }
    }
    catch (e: any) {
      console.error(`Avatar ${link.host} get avatar state error:`, e);
      Log.error(__("Avatar {{name}} get avatar state error: {{message}}", {name: link.host, message: e.message}));
    }
    return avatarState;
  }

  function updateLink(link: Link, append = true): void {
    setLinks(value => {
      const newValue = Array.from(value);
      const index = newValue.findIndex(lnk => lnk.status && link.status ?
        lnk.status.uuid === link.status.uuid : lnk.host === link.host && lnk.port === link.port);
      if (index >= 0) {
        newValue.splice(index, 1, link);
      }
      else {
        if (!append) {
          return value;
        }
        newValue.push(link);
      }
      newValue.sort((a, b) => a.host.localeCompare(b.host));
      return newValue;
    });
  }

  function onLinkError(event: Electron.IpcRendererEvent, error: Error): void {
    console.error(error);
    Log.error(error.message);
  }

  function onRefresh(): void {
    setLinks([]);
    setRefreshing(true);
    window.ipcRenderer?.send(MessageId.LinkProvisionStart);
  }

  function canAddLink(link: Link): boolean {
    if (!link.status) {
      return false;
    }
    if (link.status.provisioned) {
      return false;
    }
    if (!link.status.endpoint) {
      return true;
    }
    if (link.status.endpoint.toLocaleUpperCase() !== endpoint.toLocaleUpperCase()) {
      return false;
    }
    if (link.avatarState === AvatarState.Active) {
      return false;
    }
    return true;
  }

  async function onAddLink(link: Link): Promise<void> {
    if (!link.status || !canAddLink(link)) {
      return;
    }

    /* eslint-disable @typescript-eslint/naming-convention */
    const { uuid, publickey, ssh_public_key } = link.status;
    /* eslint-enable @typescript-eslint/naming-convention */
    setUpdating(true);
    if (link.avatarState === AvatarState.None || link.avatarState === AvatarState.Discarded) {
      try {
        await api.registerAvatar({ obj: uuid, public_key: publickey, ssh_public_key });
        link.avatarState = await getAvatarState(link);
        updateLink({...link}, false);
      }
      catch (e: any) {
        console.error("Avatar registration error:", e);
        Log.error(__("Avatar registration error: ") + e.message);
        setUpdating(false);
        return;
      }
    }

    if (!link.status.endpoint) {
      const args: LinkAddArgs = { link, endpoint };
      window.ipcRenderer?.send(MessageId.LinkProvisionAdd, args);
      return;
    }

    if (link.avatarState === AvatarState.Virgin) {
      try {
        await api.activateAvatar({ obj: uuid });
        link.avatarState = await getAvatarState(link);
        updateLink({...link}, false);
      }
      catch (e: any) {
        console.error("Avatar activation error:", e);
        Log.error(__("Avatar activation error: ") + e.message);
      }
    }
    setUpdating(false);
  }

  async function onAddLinkReply(event: Electron.IpcRendererEvent, args: LinkAddReplyArgs): Promise<void> {
    const { link } = args;
    if (!link.status) {
      return;
    }

    let avatarState = await getAvatarState(link);
    if (avatarState === AvatarState.Virgin) {
      try {
        await api.activateAvatar({ obj: link.status.uuid });
        avatarState = await getAvatarState(link);
      }
      catch (e: any) {
        console.error("Avatar activation error:", e);
        Log.error(__("Avatar activation error: ") + e.message);
      }
    }

    updateLink({ ...link, avatarState }, false);
    setUpdating(false);

    setTimeout(() => window.ipcRenderer?.send(MessageId.LinkProvisionStart), 1000);
  }

  function getCurrentEndpoint(): string {
    let endpoint = removeTrailingSlash(process.env.API_BACKEND || window.location.origin);
    if (endpoint.startsWith("https://")) {
      endpoint = endpoint.substring("https://".length);
    }
    if (endpoint.startsWith("http://")) {
      endpoint = endpoint.substring("http://".length);
    }
    return endpoint;
  }

  return (
    <Segment className="LinkProvision">
      {!isElectron() ?
        <Message info>
          {__("Avatar activation via discover is available for desktop application version only")}
        </Message> :
        <div className="LinkProvision-Content">
          {updating && <Loading text={__("Updating...")}/>}

          <div className="LinkProvision-Top">
            <Button onClick={() => navigate(-1)}>
              <Icon name="arrow left"/>{__("Back")}
            </Button>
            <Button className="Refresh_button" onClick={onRefresh}>
              <Icon name="sync"/>{__("Refresh")}
            </Button>
          </div>

          <div className="LinkProvision-Data">
            <Table celled compact>
              <Table.Header>
                <Table.Row>
                  <Table.HeaderCell>{__("Serial Number (GUID)")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("DNS Name")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("IP Address")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("System URL")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Provisioned")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Action")}</Table.HeaderCell>
                </Table.Row>
              </Table.Header>

              <Table.Body>
                {links
                  .map(link =>
                    <Table.Row key={link.status?.uuid || link.host}>
                      <Table.Cell>
                        {link.status ? link.status.uuid : __("Unknown")}
                      </Table.Cell>
                      <Table.Cell>
                        {link.status ? link.status.hostname : __("Unknown")}
                      </Table.Cell>
                      <Table.Cell>{link.host}</Table.Cell>
                      <Table.Cell>
                        {link.status ? link.status.endpoint : __("Unknown")}
                      </Table.Cell>
                      <Table.Cell textAlign="center">
                        {link.status ? (link.status.provisioned ? __("Yes") : __("No")) : __("Unknown")}
                      </Table.Cell>
                      <Table.Cell textAlign="center">
                        {canAddLink(link) && <Button className="Activate" onClick={() => onAddLink(link)}>{__("Activate")}</Button>}
                      </Table.Cell>
                    </Table.Row>)}

                {links.length === 0 &&
                <Table.Row>
                  <Table.Cell colSpan={6} textAlign="center">
                    {refreshing ? __("Refreshing...") : __("No links found")}
                  </Table.Cell>
                </Table.Row>}
              </Table.Body>
            </Table>
          </div>
        </div>}
    </Segment>
  );
};

export default LinkProvision;
