import React, { useState, useEffect } from "react";
import { useNavigate, useLocation, useParams } from "react-router-dom";
import { Segment, Table, Icon, Popup, Input, Modal, Button, Header } from "semantic-ui-react";
import {useApolloClient} from "@apollo/client";
import {useGroupsQuery, GroupsQuery, useDeleteGroupMutation, GroupsDocument, UsersDocument, UsersQuery} from "generated/graphql";
import { PropType } from "utils";
import WithQueryStatus from "components/WithQueryStatus";
import IconButton from "components/IconButton";
import Loading from "components/Loading";
import CreateUpdateGroup from "components/Admin/Groups/CreateUpdateGroup";
import { WidgetProps } from "components/Widgets";
import ListText from "components/Admin/Helpers/ListText";
import Help from "components/Help";
import {Log} from "@solid/libs/log";
import {__} from "@solid/libs/i18n";
import { UUID } from "@solid/types";
import produce from "immer";
import { RouteParams } from "@core/types";

import "./style.css";
import HelpMD from "./help.md";

type GroupsProps = WidgetProps;

type Group = PropType<GroupsQuery, "groups">[0];

const Groups = ({ cellProps, setCellProps }: GroupsProps) => {
  const client = useApolloClient();
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const {viewId = "", groupId = "" } = useParams<RouteParams>();
  const { data, error, loading } = useGroupsQuery();
  const [groups, setGroups] = useState<Group[]>([]);
  const [searchText, setSearchText] = useState("");
  const [selectedGroupId, setSelectedGroupId] = useState("");
  const selectedGroup = selectedGroupId ? groups.find(group => group.id === selectedGroupId) : undefined;
  const [createUpdateOpen, setCreateUpdateOpen] = useState(false);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [deleteGroup, { error: deleteError, loading: deleteLoading }] = useDeleteGroupMutation();

  useEffect(() => {
    pathname.includes("/view/group/add") && onCreateClick();
  }, [pathname]);

  useEffect(() => {
    if (!groupId || !data) return;

    const isGroupExist = data.groups.some(group => group.id === groupId);
    isGroupExist && onUpdateClick(groupId);
  }, [data, groupId]);

  useEffect(() => {
    if (viewId.length > 0 && createUpdateOpen && !deleteConfirmOpen) {
      setCreateUpdateOpen(false);
    }
  }, [viewId]);

  useEffect(() => {
    if (data) {
      const groups = Array.from(data.groups)
        .filter(group => !searchText || group.name.toLocaleUpperCase().includes(searchText.toLocaleUpperCase()))
        .sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
      setGroups(groups);
    }
  }, [data, searchText]);

  useEffect(() => {
    if (deleteError) {
      Log.error(`Could not delete group: ${deleteError.message}`);
    }
  }, [deleteError]);

  useEffect(() => {
    if (!setCellProps) {
      return;
    }
    let title = cellProps?.title ?? "";
    if (createUpdateOpen) {
      title = selectedGroup ? __("Edit Group") : __("Create Group");
    }
    setCellProps({ title });
  }, [createUpdateOpen, selectedGroup]);

  function onCreateClick(): void {
    setSelectedGroupId("");
    setCreateUpdateOpen(true);
    !pathname.includes("/view/group/add") && navigate("/view/group/add");
  }

  function onUpdateClick(groupId: UUID): void {
    setSelectedGroupId(groupId);
    setCreateUpdateOpen(true);
    !pathname.includes(`/view/group/edit/${groupId}`) && navigate(`/view/group/edit/${groupId}`);
  }

  function onDeleteClick(group: Group): void {
    setSelectedGroupId(group.id);
    setDeleteConfirmOpen(true);
  }

  async function onDeleteGroup(): Promise<void> {
    setDeleteConfirmOpen(false);
    if (!selectedGroup) {
      return;
    }

    const groupId = selectedGroup.id;
    const response = await deleteGroup({
      variables: { id: groupId }
    });

    // update cache
    // update GroupsDocument, UsersDocument
    if (response.data?.deleteGroup) {
      const groupsCacheData = client.readQuery<GroupsQuery>({
        query: GroupsDocument
      });

      const groupsCache = groupsCacheData?.groups ?? [];
      const groupIndex = groupsCache.findIndex((group) => group.id === groupId);
      const groups = produce(groupsCache, (draft) => {
        draft.splice(groupIndex, 1);
      });

      client.writeQuery<GroupsQuery>({
        query: GroupsDocument,
        data: {
          groups
        }
      });

      const usersCacheData = client.readQuery<UsersQuery>({
        query: UsersDocument
      });

      const usersCache = usersCacheData?.users ?? [];
      const userIndexToGroupIndex = new Map<number, number>();
      usersCache.forEach((user, userIndex) => {
        const groupIndex = user.groups.findIndex((group) => group.id === groupId);
        if (userIndex !== -1) {
          userIndexToGroupIndex.set(userIndex, groupIndex);
        }
      });

      if (userIndexToGroupIndex.size > 0) {
        const users = produce(usersCache, (draft) => {
          for (const [userIndex, groupIndex] of userIndexToGroupIndex) {
            draft[userIndex].groups.splice(groupIndex, 1);
          }
        });

        client.writeQuery<UsersQuery>({
          query: UsersDocument,
          data: {
            users
          }
        });
      }
    }
  }

  return !createUpdateOpen ?
    <Segment className="Groups">
      <WithQueryStatus loading={loading} error={error}>
        <div className="Groups-Content">
          <div className="Groups-Top">
            <Input
              placeholder={__("Filter by name")}
              icon="search"
              value={searchText}
              onChange={e => setSearchText(e.currentTarget.value)}
            />
            <IconButton
              icon="plus"
              hint={__("Create Group")}
              onClick={onCreateClick}
            />
          </div>

          <div className="Groups-Data">
            <Table celled compact>
              <Table.Header>
                <Table.Row>
                  <Table.HeaderCell/>
                  <Table.HeaderCell>{__("Name")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Users")}</Table.HeaderCell>
                  <Table.HeaderCell>{__("Policies")}</Table.HeaderCell>
                </Table.Row>
              </Table.Header>

              <Table.Body>
                {groups.map(group =>
                  <Table.Row key={group.id}>
                    <Table.Cell collapsing>
                      <Popup trigger={
                        <Icon name="edit" className="Groups-IconButton" onClick={() => onUpdateClick(group.id)}/>
                        }
                        content={__("Edit")}
                      />
                      <Popup trigger={
                        <Icon name="trash alternate" className="Groups-IconButton" onClick={() => onDeleteClick(group)}/>
                        }
                        content={__("Delete")}
                      />
                    </Table.Cell>
                    <Table.Cell>
                      {group.name}
                    </Table.Cell>
                    <Table.Cell>
                      <ListText items={group.users} modalHeader={__("Users")}/>
                    </Table.Cell>
                    <Table.Cell>
                      <ListText items={group.policies} modalHeader={__("Policies")}/>
                    </Table.Cell>
                  </Table.Row>)}

                {groups.length === 0 &&
                <Table.Row>
                  <Table.Cell colSpan={4} textAlign="center">
                    {__("No groups found")}
                  </Table.Cell>
                </Table.Row>}
              </Table.Body>
            </Table>
          </div>
        </div>

        <Modal open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
          <Header>{__("Delete Group")}</Header>
          <Modal.Content>{__("Are you sure you want to delete group '{{name}}'?", {name: selectedGroup?.name})}</Modal.Content>
          <Modal.Actions>
            <Button negative onClick={() => onDeleteGroup()}>
              <Icon name="trash"/>{__("Delete")}
            </Button>
            <Button onClick={() => setDeleteConfirmOpen(false)}>
              <Icon name="cancel"/>{__("Cancel")}
            </Button>
          </Modal.Actions>
        </Modal>

        {deleteLoading && <Loading text={__("Updating...")}/>}

        <Help markdown={HelpMD}/>
      </WithQueryStatus>
    </Segment> :
    <CreateUpdateGroup group={selectedGroup} onBack={() => navigate(-1)}/>;
};

export default Groups;
