import React, { FC, useEffect, useRef, useState } from "react";
import { Segment } from "semantic-ui-react";
import {useLocation, useNavigate, useParams} from "react-router-dom";
import GatewayList from "./GatewayList/GatewayList";
import GatewaySettings from "./GatewaySettings";
import { WidgetProps } from "components/Widgets";
import { DeviceFunctionalAspectType, useDeleteDeviceMutation, useGatewayListByAspectTypeQuery } from "@generated/graphql";
import { GatewayShort, useDeviceActions } from "@core/actions";
import DeleteModal from "./DeleteModal";

import "./style.css";
import {RouteParams} from "@core/types";

const Gateways: FC<WidgetProps> = ({ ...props }) => {
  const navigate = useNavigate();
  const { pathname } = useLocation();
  const { viewId = "", gatewayId = "" } = useParams<RouteParams>();
  const {
    data,
    error: gatewayError,
    loading: gatewayLoading,
  } = useGatewayListByAspectTypeQuery({ variables: { types: [{ type: DeviceFunctionalAspectType.Gateway }] } });
  const [deleteDevice, { error: deleteError, loading: deleteLoading }] = useDeleteDeviceMutation();
  const { deleteCachedDevice } = useDeviceActions({ skipSubscription: true });

  const [selectedId, setSelectedId] = useState<string>();
  const [configOpen, setConfigOpen] = useState<boolean>(false);
  const [createdDeviceId, setCreatedDeviceId] = useState<string>();
  const [deleteModalOpen, setDeleteModalOpen] = useState<boolean>(false);
  const [deviceToRemove, setDeviceToRemove] = useState<GatewayShort>();

  const mountedRef = useRef<boolean>(false);

  useEffect(() => {
    mountedRef.current = true;

    return () => {
      mountedRef.current = false;
    };
  }, []);

  useEffect(() => {
    if (gatewayId) {
      setConfigOpen(true);
      setSelectedId(gatewayId);
    }
    if (viewId) {
      setConfigOpen(false);
      setSelectedId(undefined);
    }
    if (pathname === "/view/gateway/add") {
      setConfigOpen(true);
      setSelectedId(undefined);
    }
  }, [pathname]);

  function onCreateGateway() {
    navigate("/view/gateway/add");
  }

  function onEditGateway(gateway: GatewayShort) {
    setSelectedId(gateway.id);
    navigate(`/view/gateway/edit/${gateway.id}`);
  }

  function onCreated(id: string) {
    if (!mountedRef.current) return;

    navigate(-1);
    setCreatedDeviceId(id);
  }

  function onModalClose() {
    setDeleteModalOpen(false);
    setDeviceToRemove(undefined);
  }

  async function deleteGateway(gateway: GatewayShort) {
    onModalClose();
    const { data } = await deleteDevice({ variables: { id: gateway.id } });
    if (data?.deleteDevice) {
      deleteCachedDevice(gateway.id);
    }
  }

  function onHandleDeleteGateway(gateway: GatewayShort) {
    setDeleteModalOpen(true);
    setDeviceToRemove(gateway);
  }

  const error = gatewayError || deleteError;
  const loading = gatewayLoading || deleteLoading;

  return (
    <Segment className="Gateways">
      {configOpen ? (
        <GatewaySettings id={selectedId} onCreated={onCreated} {...props} />
      ) : (
        <GatewayList
          data={data}
          error={error}
          loading={loading}
          onCreate={onCreateGateway}
          onDelete={onHandleDeleteGateway}
          onSelect={onEditGateway}
          createdId={createdDeviceId}
          {...props}
        />
      )}
      {deleteModalOpen && deviceToRemove && <DeleteModal device={deviceToRemove} onDelete={deleteGateway} onCancel={onModalClose} />}
    </Segment>
  );
};

export default Gateways;
