import React, { useEffect, useImperativeHandle, useMemo, useRef } from "react";

import { Segment } from "semantic-ui-react";
import { AutoForm, AutoLayout, FieldValues, FormSchema } from "components/AutoForm";
import { __ } from "@solid/libs";

import { AuthType, DeviceBaseConfigType, DevicesByAspectTypesShortQuery, GatewayModel, ProtocolType } from "@generated/graphql";
import { ComponentWithFormRef, DeviceProperties } from "../GatewaySettings";

import "../GatewayAuth/style.css";

interface GatewayDirectoriesProps {
  gateway?: DeviceProperties;
  avatars?: DevicesByAspectTypesShortQuery;
  onChange: (property: DeviceProperties) => void;
}

const generalFields = ["platformId", "name", "enabled", "location"];
const connectFields = ["host", "port", "protocol"];
const configFields = ["model"];

const GatewayProperties = React.forwardRef<ComponentWithFormRef, GatewayDirectoriesProps>(({ gateway, avatars, onChange }, ref) => {
  const formRef = useRef<AutoForm>(null);

  const schema = useMemo(() => {
    const devicePlatformId = gateway?.platform ? gateway.platform.id : "";
    const devicePlatformName = gateway?.platform ? gateway.platform.name : "";

    const platformIdDataSource = gateway
      ? Array.from([{ id: devicePlatformId, name: devicePlatformName }] ?? [])
      : Array.from(avatars?.devicesByAspectTypes ?? []).sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: "base" }));
    const schema: FormSchema = [
      {
        name: "platformId",
        label: __("Avatar"),
        required: true,
        type: "dropdown",
        dataSource: Array.from(platformIdDataSource ?? []),
        disableCondition: () => !!gateway?.platform?.id,
      },
      {
        name: "name",
        label: __("Name"),
        required: true,
      },
      {
        name: "enabled",
        label: __("State"),
        type: "dropdown",
        required: true,
        dataSource: [
          { id: "on", name: __("On") },
          { id: "off", name: __("Off") },
        ],
      },
      {
        name: "location",
        label: __("Location"),
      },
      {
        name: "model",
        label: __("Gateway type"),
        type: "dropdown",
        required: true,
        dataSource: [{ id: GatewayModel.Lenel, name: "Lenel" }],
      },
      {
        name: "host",
        label: __("Gateway IP/Host"),
        required: true,
      },
      {
        name: "port",
        label: __("Port"),
        type: "integer",
        required: true,
        minValue: 1,
        maxValue: 65535,
      },
      {
        name: "protocol",
        label: __("HTTP Secured"),
        required: true,
        type: "dropdown",
        dataSource: [
          { id: ProtocolType.Https, name: "Yes" },
          { id: ProtocolType.Http, name: "No" },
        ],
      },
    ];
    return schema;
  }, [gateway, avatars]);

  const values = useMemo(() => {
    if (gateway) {
      const { name, enabled, location, platform, config } = gateway;
      const connectOptions = config?.connect;
      return {
        name,
        enabled: enabled ? "on" : "off",
        location,
        platformId: platform?.id,
        host: connectOptions?.host,
        port: connectOptions?.port,
        protocol: connectOptions?.protocol,
        model: GatewayModel.Lenel,
      };
    }
    return {
      enabled: "on",
      platform: "",
      port: 80,
    };
  }, [gateway]);

  useEffect(() => {
    const values = getValues();
    values && onChange(values);
  }, [values]);

  useImperativeHandle(ref, () => ({
    getFormRef() {
      return formRef.current;
    },
  }));

  function onFormChange(name: string, value: any, values: FieldValues, form: AutoForm) {
    form.setValue(name, value);
    if (name === "platformId") {
      const platform = avatars?.devicesByAspectTypes.find((avatar) => avatar.id === value);
      platform && onChange({ platform: { id: platform.id, name: platform.name, aspects: platform.aspects } });
      return;
    }
    if (name === "enabled") {
      onChange({ enabled: value === "on" });
      return;
    }
    if ([...configFields, ...connectFields].includes(name)) {
      const config = getConfigProperties(values);
      onChange(config);
      return;
    }
    onChange({ [name]: value });
  }

  function getValues(): DeviceProperties | undefined {
    if (!formRef.current) return undefined;
    const values = formRef.current.getValues();
    const platform = avatars?.devicesByAspectTypes.find((avatar) => avatar.id === values["platformId"]);
    const config = getConfigProperties(values);

    const device: DeviceProperties = {
      name: values["name"],
      enabled: values["enabled"] === "on",
      location: values["location"],
      ...config,
    };

    if (platform) {
      device.platform = { id: platform.id, name: platform.name, aspects: platform.aspects };
    }

    return device;
  }

  function getConfigProperties(values: FieldValues) {
    const connect = gateway?.config?.connect || {
      authType: AuthType.NoAuth,
    };
    const newConfigValues = {
      config: {
        configType: DeviceBaseConfigType.Gateway,
        model: values["model"],
        connect: {
          ...connect
        },
      },
    };

    for (const field of connectFields) {
      newConfigValues.config.connect[field] = values[field];
    }

    return newConfigValues;
  }


  return (
    <Segment className="GatewayProperties">
      <AutoForm ref={formRef} schema={schema} values={values} onChange={onFormChange} onValidChange={() => !!formRef.current?.getIsValid()}>
        <div className="GatewayProperties-FormLayout">
          <div className="GatewayProperties-FormColumn1">
            <AutoLayout names={generalFields} />
          </div>
          <div className="GatewayProperties-FormColumn2">
            <AutoLayout names={[...connectFields, ...configFields]} />
          </div>
        </div>
      </AutoForm>
    </Segment>
  );
});

export default GatewayProperties;
