import React, { FC, useEffect, useState } from "react";
import { Icon, Popup, Segment } from "semantic-ui-react";
import { Table, AutoSizer, Column } from "react-virtualized";
import { __ } from "@solid/libs";
import { CameraList } from "./CameraList/CameraList";
import ListFilter from "components/ListFilter";
import {AssociatedGatewayDevice} from "@generated/graphql";

interface DevicesTableProps {
  devices: AssociatedGatewayDevice[];
  onAssign: (device: AssociatedGatewayDevice) => void;
}

const DevicesTable: FC<DevicesTableProps> = ({ devices, onAssign }) => {
  const [searchText, setSearchText] = useState<string>("");
  const [deviceList, setDeviceList] = useState<AssociatedGatewayDevice[]>([]);

  function filterDevices(searchText: string) {
    const text = searchText.toLocaleUpperCase();
    setDeviceList(
      devices
        .filter((device) => !text || device.name.toLocaleUpperCase().includes(text) || device.id.toLocaleUpperCase().includes(text))
        .sort((a, b) => a.id.localeCompare(b.id, undefined, { sensitivity: "base" }))
    );
  }

  useEffect(() => {
    if (searchText) {
      filterDevices(searchText);
    } else {
      setDeviceList(devices);
    }
  }, [devices, searchText]);

  return (
    <Segment loading={false} className="GatewayTable-Wrapper">
      <ListFilter
        searchText={searchText}
        nameFilter
        onSearchTextChange={(text) => {
          setSearchText(text);
        }}
        filterTextPlaceholder={__("Filter")}
      />
      <AutoSizer style={{ width: "100%", height: "100%", marginTop: "0.5rem" }}>
        {({ width, height }) => (
          <Table
            className="GatewayTable"
            width={width}
            height={height}
            rowHeight={34}
            headerHeight={58}
            rowCount={deviceList.length}
            rowGetter={({ index }) => deviceList[index]}
          >
            <Column dataKey="id" label={__("ID")} width={100} flexGrow={1} />
            <Column dataKey="name" label={__("Name")} width={100} flexGrow={1} />
            <Column
              dataKey="assignedCameras"
              label={__("Assigned cameras")}
              width={100}
              flexGrow={1}
              cellRenderer={({ rowData }) => <CameraList cameras={rowData.associatedDevices} />}
            />
            <Column
              dataKey="actions"
              width={40}
              cellRenderer={({ rowData }) => (
                <Popup
                  trigger={<Icon className="action-device-test" name={rowData.associatedDevices.length !== 0 ? "edit" : "plus"} onClick={() => onAssign(rowData)} />}
                  content={rowData.associatedDevices.length !== 0 ? __("Edit") : __("Assign")}
                />
              )}
            />
          </Table>
        )}
      </AutoSizer>
    </Segment>
  );
};
export default DevicesTable;
