import React, { useImperativeHandle, useMemo, useRef, useState } from "react";
import { AutoForm, AutoLayout, FieldValues, FormSchema } from "components/AutoForm";
import { Segment } from "semantic-ui-react";

import { __ } from "@solid/libs";
import { DeviceFunctionalAspectType, GatewayDirectory } from "@generated/graphql";
import { ComponentWithFormRef, DeviceProperties } from "../GatewaySettings";

import "./style.css";

type GatewayAuthProps = {
  loading?: boolean;
  directories: GatewayDirectory[];
  gateway?: DeviceProperties;
  onChange: (property: DeviceProperties) => void;
  getDirectories: () => void;
};

const GatewayAuth = React.forwardRef<ComponentWithFormRef, GatewayAuthProps>(({ loading, onChange, directories, gateway, getDirectories }, ref) => {
  const formRef = useRef<AutoForm>(null);
  const [requiredCredential, setRequiredCredential] = useState({
    logPass: true,
    token: true,
  });

  const generalFields = ["gatewayLogin", "gatewayPass", "oidcToken", "directories"];

  function onFormChange(name: string, value: any, values: FieldValues, form: AutoForm) {
    form.setValue(name, value);
    const properties = form.getValues();
    if (properties.gatewayLogin && properties.gatewayPass) {
      setRequiredCredential({
        logPass: true,
        token: false,
      });
    }
    if (!properties.gatewayLogin || !properties.gatewayPass) {
      setRequiredCredential({
        logPass: true,
        token: true,
      });
    }
    if (properties.oidcToken) {
      setRequiredCredential({
        logPass: false,
        token: true,
      });
    }
    onChange({ [name]: value });
  }

  const schema: FormSchema = [
    {
      name: "gatewayLogin",
      label: __("Gateway login"),
      required: requiredCredential.logPass,
    },
    {
      name: "gatewayPass",
      label: __("Gateway pass"),
      required: requiredCredential.logPass,
      type: "password",
    },
    {
      name: "oidcToken",
      label: __("OIDC token"),
    },
    {
      name: "directories",
      label: __("Directories"),
      type: "dropdown",
      action: () => getDirectories(),
      required: true,
      dataSource: Array.from(directories).map((dir) => ({ id: dir.id, name: dir.name })),
    },
  ];

  useImperativeHandle(ref, () => ({
    getFormRef() {
      return formRef.current;
    },
  }));

  const values = useMemo(() => {
    if (gateway) {
      const authOptions = gateway.config?.connect;
      let directory;
      gateway.aspects?.find((aspect) => {
        if (aspect.__typename === "DFA_Gateway" && aspect.type === DeviceFunctionalAspectType.Gateway) {
          directory = {
            id: aspect.directory.id,
            name: aspect.directory.name,
          };
        }
      });
      return {
        gatewayLogin: authOptions?.user,
        gatewayPass: authOptions?.pass,
        directories: directory,
      };
    }
    return {
      gatewayLogin: "",
      gatewayPass: "",
      oidcToken: "",
      directories: "",
    };
  }, [gateway]);

  return (
    <Segment className="GatewayProperties">
      <AutoForm ref={formRef} values={values} schema={schema} onChange={onFormChange}>
        <div className="GatewayProperties-FormLayout">
          <div className="GatewayProperties-FormColumn1">
            <AutoLayout names={generalFields} />
          </div>
        </div>
      </AutoForm>
    </Segment>
  );
});

export default GatewayAuth;
