import React, { useEffect, useState } from "react";
import { AutoSizer, Column, Table } from "react-virtualized";
import { Button, DropdownItemProps, Form, Icon, Popup, Segment } from "semantic-ui-react";

import { __ } from "@solid/libs/i18n";
import { Device, DeviceFunctionalAspectType, Dfa_Avatar, HealthStatus,  useDevicesByAspectTypesReportLazyQuery } from "@generated/graphql";
import { DeviceReport,  getLinkChannelStatusText, getLinkDeliveryQueue, getLinkStatusUpdateAge, useLinkDeviceInfo } from "@core/actions";
import { formatDateTime } from "@core/utils";

import DeviceInfoTrigger from "components/Admin/AdminLinkList/DeviceInfoTrigger";
import DeviceInfoText from "components/Admin/AdminLinkList/DeviceInfoText";

import "./style.css";

enum ConfigurationStatusReportType {
  CAMERA = "Camera",
  AVATAR = "Avatar",
  VIDEO = "Video",
}

enum DfaMediaItem {
  CODEC = "codec",
  CLOUDARCHIVE = "cloudArchive",
  FPS = "FPS",
  LIVEBITRATE = "liveBitrate",
  SPACEUSAGEDAY = "spaceUsageDay",
  CONSUMEDSPACE = "consumedSpace",
}

const configurationStatusReportTypeOptions: DropdownItemProps[] = [
  {key: ConfigurationStatusReportType.CAMERA, value: ConfigurationStatusReportType.CAMERA, text: __("Camera")},
  {key: ConfigurationStatusReportType.AVATAR, value: ConfigurationStatusReportType.AVATAR, text: __("Avatar")},
  {key: ConfigurationStatusReportType.VIDEO, value: ConfigurationStatusReportType.VIDEO, text: __("Video")},
];

const ConfigurationStatusReport = () => {
  const [getDevices, {data, loading, error}] = useDevicesByAspectTypesReportLazyQuery({fetchPolicy: "no-cache"});
  const { getLinkDevicesCounts, loading: infoLoading, error: infoError } = useLinkDeviceInfo();
  const [reportType, setReportType] = useState<ConfigurationStatusReportType>();
  const [cameras, setCameras] = useState<DeviceReport[]>();
  const [avatars, setAvatars] = useState<DeviceReport[]>();

  const CSCameraReportTitle: string[] = [
    "Name",
    "Avatar name",
    "Type",
    "IP address",
    "Model",
    "Codec",
    "Content delivery settings",
    "FPS",
    "Current status",
    "Live bitrate (bps)",
    "Daily storage usage (MB)",
    "Total storage usage (MB)",
  ];

  const CSAvatarReportTitle: string[] = [
    "Name",
    "Configured cameras",
    "Current connection status",
    "Last status update time",
    "Delivery queue size (MB)",
    "Online cameras",
    "Offline / Broken cameras",
    "Turnaround time (ms)",
  ];

  useEffect(() => {
    if (data && !loading && !error) {
      const devices: DeviceReport[] = data.devicesByAspectTypes as DeviceReport[];
      if (reportType === ConfigurationStatusReportType.CAMERA) {
        setCameras(devices);
      }
      if (reportType === ConfigurationStatusReportType.AVATAR) {
        setAvatars(devices);
      }
    }
  }, [data, loading, error]);

  const getCurrentDevices = () => {
    if (!reportType) {
      return;
    }

    switch (reportType) {
      case ConfigurationStatusReportType.AVATAR:
        getDevices({variables: { types: [{ type: DeviceFunctionalAspectType.Avatar }] } });
        break;
      case ConfigurationStatusReportType.CAMERA:
        getDevices({variables: { types: [{ type: DeviceFunctionalAspectType.Media }] } });
        break;
    }
  };

  const createCsvFile = () => {
    let rows: Array<string[]> = [];
    let fileName = "undefined.csv";
    if (reportType === ConfigurationStatusReportType.CAMERA) {
      fileName = "cameras_configuration_status_report.csv";
      rows = [CSCameraReportTitle];
      if (cameras) {
        cameras.forEach(camera => {
          const row: string[] = [
            camera.name,
            camera.platform?.name ?? "",
            camera.config.configType ?? "",
            camera.config.connect.host ?? "",
            camera.config.model ?? "",
            getDfaMediaItemValue(camera, DfaMediaItem.CODEC),
            getDfaMediaItemValue(camera, DfaMediaItem.CLOUDARCHIVE),
            getDfaMediaItemValue(camera, DfaMediaItem.FPS),
            camera.healthStatus ?? "",
            getDfaMediaItemValue(camera, DfaMediaItem.LIVEBITRATE),
            getDfaMediaItemValue(camera, DfaMediaItem.SPACEUSAGEDAY),
            getDfaMediaItemValue(camera, DfaMediaItem.CONSUMEDSPACE),
          ];

          rows.push(row);
        });
      }
    }
    if (reportType === ConfigurationStatusReportType.AVATAR) {
      fileName = "avatars_configuration_status_report.csv";
      rows = [CSAvatarReportTitle];
      if (avatars) {
        avatars.forEach(avatar => {
          const row: string[] = [avatar.name];

          const {total} = getLinkDevicesCounts(avatar);
          row.push(String(total));

          const status = getLinkChannelStatusText(avatar as Device);
          row.push(status || __("N/A"));

          const age = getAvatarLastStatusUpdate(avatar);
          row.push(age);

          const deliveryQueue = getLinkDeliveryQueue(avatar as Device);
          row.push(deliveryQueue ? String(deliveryQueue) : "");

          const onlineCameras = getAvatarOnlineCameras(avatar as Device);
          row.push(String(onlineCameras));

          const otherCameras = getAvatarOfflineBrokenCameras(avatar as Device);
          row.push(String(otherCameras));

          const turnaroundTime = getAvatarTurnaroundTime(avatar as Device);
          row.push(String(turnaroundTime));

          rows.push(row);
        });
      }
    }

    const csvContent = "data:text/csv;charset=utf-8," + encodeURIComponent(rows.map(e => e.join(",")).join("\n"));
    const link = document.createElement("a");
    link.setAttribute("href", csvContent);
    link.setAttribute("download", fileName);
    document.body.appendChild(link);

    link.click();
    link.remove();
  };

  function getDfaMediaItemValue(camera: DeviceReport, item: DfaMediaItem): string {
    const enabledAspect = camera.aspects.find(aspect => aspect.enabled && aspect.type === DeviceFunctionalAspectType.Media);
    if (!enabledAspect) {
      return "";
    }
    return enabledAspect[item];
  }

  function getDevicesInfoContent(rowData: DeviceReport) {
    const { camera, sensor, total } = getLinkDevicesCounts(rowData);
    if (total === 0) {
      return __("No devices registered");
    }

    return (
      <Popup
        trigger={
          <div className="AdminLinkList-DeviceInfoWrapper">
            { camera.total > 0 && <DeviceInfoTrigger countByStatus={camera.countByStatus} total={camera.total} icon="video" /> }
            { sensor.total > 0 && <DeviceInfoTrigger countByStatus={sensor.countByStatus} total={sensor.total} icon="bell" /> }
          </div>
          }
        >
        <b>{__("Total")} - {total}</b>
        { camera.total > 0 && <DeviceInfoText countByStatus={camera.countByStatus} title={__("Cameras")} /> }
        { sensor.total > 0 && <DeviceInfoText countByStatus={sensor.countByStatus} title={__("Sensors")} /> }
      </Popup>
    );
  }

  function getAvatarCurrentConnectionStatus(device: Device) {
    const status = getLinkChannelStatusText(device);
    return (
      <span>{status || __("N/A")}</span>
    );
  }

  function getAvatarLastStatusUpdateTime(device: Device) {
    const age = getLinkStatusUpdateAge(device);
    return (
      <>{!!age && <>&nbsp;<span>{age}</span></>}</>
    );
  }

  function getAvatarOnlineCameras(device: DeviceReport) {
    const { camera: { countByStatus } } = getLinkDevicesCounts(device);
    return countByStatus[HealthStatus.Normal];
  }

  function getAvatarOfflineBrokenCameras(device: DeviceReport) {
    const { camera: { countByStatus } } = getLinkDevicesCounts(device);
    return countByStatus[HealthStatus.Broken] + countByStatus[HealthStatus.Degraded] + countByStatus[HealthStatus.Off] + countByStatus[HealthStatus.Pending];
  }

  function getAvatarTurnaroundTime(device: DeviceReport) {
    const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar) as Dfa_Avatar;
    return aspect.turnaroundTime || "";
  }

  function getAvatarLastStatusUpdate(device: DeviceReport) {
    const aspect = device.aspects.find(aspect => aspect.type === DeviceFunctionalAspectType.Avatar) as Dfa_Avatar;
    return aspect.lastStatusUpdate ? formatDateTime(new Date(aspect.lastStatusUpdate)).replace(",", "") : __("N/A");
  }

  return (
    <Segment className="ConfigurationStatusReport-Wrapper" key="csrw" loading={infoLoading}>
      <div className="ConfigurationStatusReport-Header">
        <Form.Select
          disabled={infoLoading || loading}
          className="ConfigurationStatusReport-Dropdown"
          options={configurationStatusReportTypeOptions}
          placeholder={__("Check type of report")}
          value={reportType}
          onChange={(e, {value}) => setReportType(value as ConfigurationStatusReportType)}/>
        <Button
          disabled={!reportType || infoLoading || loading}
          onClick={() => getCurrentDevices()}>
          {__("Generate report")}
        </Button>
        {((!!cameras && reportType === ConfigurationStatusReportType.CAMERA) ||
          (!!avatars && reportType === ConfigurationStatusReportType.AVATAR)) &&
          <Button
            onClick={() => createCsvFile()}>
            <Icon name="download"/>
            {__("Get CSV")}
          </Button>}
      </div>
      <Segment loading={loading} error={error || infoError} className="ConfigurationStatusReport-Segment" key="csrs">
        {!!cameras && reportType === ConfigurationStatusReportType.CAMERA &&
          <AutoSizer style={{ width: "100%", height: "100%" }}>
            {({ width, height }) => (
              <Table
                className="ConfigurationStatusReport-Table"
                width={width}
                height={height}
                rowHeight={34}
                headerHeight={58}
                rowCount={cameras.length}
                rowGetter={({ index }) => cameras[index]}>
                <Column
                  dataKey="name"
                  label={__("Name")}
                  width={100}
                  flexGrow={1}/>
                <Column
                  dataKey="avatarName"
                  label={__("Avatar name")}
                  width={100}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => rowData.platform?.name ?? ""}/>
                <Column
                  dataKey="type"
                  label={__("Type")}
                  width={50}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => rowData.config.configType ?? ""}/>
                <Column
                  dataKey="ipAddress"
                  label={__("IP address")}
                  width={100}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => rowData.config.connect.host ?? ""}/>
                <Column
                  dataKey="model"
                  label={__("Model")}
                  width={80}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => rowData.config.model ?? ""}/>
                <Column
                  dataKey="codec"
                  label={__("Codec")}
                  width={50}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getDfaMediaItemValue(rowData, DfaMediaItem.CODEC)}/>
                <Column
                  dataKey="contentDeliverySetting"
                  label={__("Content delivery setting")}
                  width={100}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getDfaMediaItemValue(rowData, DfaMediaItem.CLOUDARCHIVE)}/>
                <Column
                  dataKey="fps"
                  label={__("FPS")}
                  width={100}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getDfaMediaItemValue(rowData, DfaMediaItem.FPS)}/>
                <Column
                  dataKey="currentStatus"
                  label={__("Current status")}
                  width={100}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => rowData.healthStatus}/>
                <Column
                  dataKey="liveBitrate"
                  label={__("Live bitrate, bps")}
                  width={100}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getDfaMediaItemValue(rowData, DfaMediaItem.LIVEBITRATE)}/>
                <Column
                  dataKey="dailyStorageUsage"
                  label={__("Daily storage usage, MB")}
                  width={100}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getDfaMediaItemValue(rowData, DfaMediaItem.SPACEUSAGEDAY)}/>
                <Column
                  dataKey="totalStorageUsage"
                  label={__("Total storage usage, MB")}
                  width={100}
                  flexGrow={1}
                  cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getDfaMediaItemValue(rowData, DfaMediaItem.CONSUMEDSPACE)}/>
              </Table>)}
          </AutoSizer>}
        {!!avatars && reportType === ConfigurationStatusReportType.AVATAR &&
        <AutoSizer style={{ width: "100%", height: "100%" }} key={`as_avatar_${avatars.length}`}>
          {({ width, height }) => (
            <Table
              className="ConfigurationStatusReport-Table"
              width={width}
              height={height}
              rowHeight={34}
              headerHeight={58}
              rowCount={avatars.length}
              rowGetter={({ index }) => avatars[index]}>
              <Column
                dataKey="name"
                label={__("Name")}
                width={100}
                flexGrow={1}/>
              <Column
                dataKey="configuredDevices"
                label={__("Configured devices")}
                width={100}
                flexGrow={1}
                cellRenderer={({ rowData }: { rowData: DeviceReport }) => getDevicesInfoContent(rowData)}/>
              <Column
                dataKey="currentConnectionStatus"
                label={__("Current connection status")}
                width={100}
                flexGrow={1}
                cellRenderer={({ rowData }: { rowData: Device }) => getAvatarCurrentConnectionStatus(rowData)}/>
              <Column
                dataKey="lastStatusUpdateTime"
                label={__("Last status update time")}
                width={100}
                flexGrow={1}
                cellRenderer={({ rowData }: { rowData: Device }) => getAvatarLastStatusUpdateTime(rowData)}/>
              <Column
                dataKey="deliveryQueueSize"
                label={__("Delivery queue size, MB")}
                width={50}
                flexGrow={1}
                cellDataGetter={({ rowData }: { rowData: Device }) => getLinkDeliveryQueue(rowData) || ""}/>
              <Column
                dataKey="onlineCameras"
                label={__("Online cameras")}
                width={50}
                flexGrow={1}
                cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getAvatarOnlineCameras(rowData)}/>
              <Column
                dataKey="offlineBrokenCameras"
                label={__("Offline / Broken cameras")}
                width={50}
                flexGrow={1}
                cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getAvatarOfflineBrokenCameras(rowData)}/>
              <Column
                dataKey="turnaroundTime"
                label={__("Turnaround time, ms")}
                width={50}
                flexGrow={1}
                cellDataGetter={({ rowData }: { rowData: DeviceReport }) => getAvatarTurnaroundTime(rowData)}/>
            </Table>)}
        </AutoSizer>}
      </Segment>
    </Segment>
  );
};

export default ConfigurationStatusReport;
